"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GelfLogDriver = exports.GelfCompressionType = void 0;
const log_driver_1 = require("./log-driver");
const utils_1 = require("./utils");
/**
 * The type of compression the GELF driver uses to compress each log message.
 */
var GelfCompressionType;
(function (GelfCompressionType) {
    GelfCompressionType["GZIP"] = "gzip";
    GelfCompressionType["ZLIB"] = "zlib";
    GelfCompressionType["NONE"] = "none";
})(GelfCompressionType = exports.GelfCompressionType || (exports.GelfCompressionType = {}));
/**
 * A log driver that sends log information to journald Logs.
 */
class GelfLogDriver extends log_driver_1.LogDriver {
    /**
     * Constructs a new instance of the GelfLogDriver class.
     *
     * @param props the gelf log driver configuration options.
     */
    constructor(props) {
        super();
        this.props = props;
        // Validation
        if (props.compressionLevel) {
            utils_1.ensureInRange(props.compressionLevel, -1, 9);
        }
        if (props.tcpMaxReconnect) {
            utils_1.ensurePositiveInteger(props.tcpMaxReconnect);
        }
    }
    /**
     * Called when the log driver is configured on a container
     */
    bind(_scope, _containerDefinition) {
        return {
            logDriver: 'gelf',
            options: utils_1.stringifyOptions({
                'gelf-address': this.props.address,
                'gelf-compression-type': this.props.compressionType,
                'gelf-compression-level': this.props.compressionLevel,
                'gelf-tcp-max-reconnect': this.props.tcpMaxReconnect,
                'gelf-tcp-reconnect-delay': this.props.tcpReconnectDelay && this.props.tcpReconnectDelay.toSeconds(),
                ...utils_1.renderCommonLogDriverOptions(this.props),
            }),
        };
    }
}
exports.GelfLogDriver = GelfLogDriver;
//# sourceMappingURL=data:application/json;base64,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