"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const elb = require("@aws-cdk/aws-elasticloadbalancing");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecs = require("../../lib");
const base_service_1 = require("../../lib/base/base-service");
const placement_1 = require("../../lib/placement");
module.exports = {
    'When creating an EC2 Service': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'Ec2TaskDef0226F28C',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50,
                },
                DesiredCount: 1,
                LaunchType: base_service_1.LaunchType.EC2,
                SchedulingStrategy: 'REPLICA',
                EnableECSManagedTags: false,
            }));
            test.done();
        },
        'with custom cloudmap namespace'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            const cloudMapNamespace = new cloudmap.PrivateDnsNamespace(stack, 'TestCloudMapNamespace', {
                name: 'scorekeep.com',
                vpc,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    failureThreshold: 20,
                    cloudMapNamespace,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'TestCloudMapNamespace1FB9B446',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 20,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'TestCloudMapNamespace1FB9B446',
                        'Id',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PrivateDnsNamespace', {
                Name: 'scorekeep.com',
                Vpc: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            test.done();
        },
        'with all properties set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            // WHEN
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                desiredCount: 2,
                assignPublicIp: true,
                cloudMapOptions: {
                    name: 'myapp',
                    dnsRecordType: cloudmap.DnsRecordType.A,
                    dnsTtl: cdk.Duration.seconds(50),
                    failureThreshold: 20,
                },
                daemon: false,
                healthCheckGracePeriod: cdk.Duration.seconds(60),
                maxHealthyPercent: 150,
                minHealthyPercent: 55,
                deploymentController: {
                    type: ecs.DeploymentControllerType.CODE_DEPLOY,
                },
                securityGroup: new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                    allowAllOutbound: true,
                    description: 'Example',
                    securityGroupName: 'Bob',
                    vpc,
                }),
                serviceName: 'bonjour',
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            });
            service.addPlacementConstraints(placement_1.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*'));
            service.addPlacementStrategies(placement_1.PlacementStrategy.spreadAcross(ecs.BuiltInAttributes.AVAILABILITY_ZONE));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'Ec2TaskDef0226F28C',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 150,
                    MinimumHealthyPercent: 55,
                },
                DeploymentController: {
                    Type: ecs.DeploymentControllerType.CODE_DEPLOY,
                },
                DesiredCount: 2,
                LaunchType: base_service_1.LaunchType.EC2,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup1F554B36F',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPublicSubnet1SubnetF6608456',
                            },
                            {
                                Ref: 'MyVpcPublicSubnet2Subnet492B6BFB',
                            },
                        ],
                    },
                },
                PlacementConstraints: [
                    {
                        Expression: 'attribute:ecs.instance-type =~ t2.*',
                        Type: 'memberOf',
                    },
                ],
                PlacementStrategies: [
                    {
                        Field: 'attribute:ecs.availability-zone',
                        Type: 'spread',
                    },
                ],
                SchedulingStrategy: 'REPLICA',
                ServiceName: 'bonjour',
                ServiceRegistries: [
                    {
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'Ec2ServiceCloudmapService45B52C0F',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            test.done();
        },
        'with multiple security groups, it correctly updates the cfn template'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const securityGroup1 = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                allowAllOutbound: true,
                description: 'Example',
                securityGroupName: 'Bingo',
                vpc,
            });
            const securityGroup2 = new ec2.SecurityGroup(stack, 'SecurityGroup2', {
                allowAllOutbound: false,
                description: 'Example',
                securityGroupName: 'Rolly',
                vpc,
            });
            // WHEN
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                desiredCount: 2,
                assignPublicIp: true,
                daemon: false,
                securityGroups: [securityGroup1, securityGroup2],
                serviceName: 'bonjour',
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'Ec2TaskDef0226F28C',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DesiredCount: 2,
                LaunchType: base_service_1.LaunchType.EC2,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup1F554B36F',
                                    'GroupId',
                                ],
                            },
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup23BE86BB7',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPublicSubnet1SubnetF6608456',
                            },
                            {
                                Ref: 'MyVpcPublicSubnet2Subnet492B6BFB',
                            },
                        ],
                    },
                },
                SchedulingStrategy: 'REPLICA',
                ServiceName: 'bonjour',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Example',
                GroupName: 'Bingo',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Example',
                GroupName: 'Rolly',
                SecurityGroupEgress: [
                    {
                        CidrIp: '255.255.255.255/32',
                        Description: 'Disallow all traffic',
                        FromPort: 252,
                        IpProtocol: 'icmp',
                        ToPort: 86,
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            test.done();
        },
        'throws when both securityGroup and securityGroups are supplied'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const securityGroup1 = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                allowAllOutbound: true,
                description: 'Example',
                securityGroupName: 'Bingo',
                vpc,
            });
            const securityGroup2 = new ec2.SecurityGroup(stack, 'SecurityGroup2', {
                allowAllOutbound: false,
                description: 'Example',
                securityGroupName: 'Rolly',
                vpc,
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                    desiredCount: 2,
                    assignPublicIp: true,
                    maxHealthyPercent: 150,
                    minHealthyPercent: 55,
                    securityGroup: securityGroup1,
                    securityGroups: [securityGroup2],
                    serviceName: 'bonjour',
                    vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
                });
            }, /Only one of SecurityGroup or SecurityGroups can be populated./);
            test.done();
        },
        'throws when task definition is not EC2 compatible'(test) {
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.TaskDefinition(stack, 'FargateTaskDef', {
                compatibility: ecs.Compatibility.FARGATE,
                cpu: '256',
                memoryMiB: '512',
            });
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                });
            }, /Supplied TaskDefinition is not configured for compatibility with EC2/);
            test.done();
        },
        'ignore task definition and launch type if deployment controller is set to be EXTERNAL'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                deploymentController: {
                    type: base_service_1.DeploymentControllerType.EXTERNAL,
                },
            });
            // THEN
            test.deepEqual(service.node.metadata[0].data, 'taskDefinition and launchType are blanked out when using external deployment controller.');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50,
                },
                DesiredCount: 1,
                SchedulingStrategy: 'REPLICA',
                EnableECSManagedTags: false,
            }));
            test.done();
        },
        'errors if daemon and desiredCount both specified'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                    daemon: true,
                    desiredCount: 2,
                });
            }, /Don't supply desiredCount/);
            test.done();
        },
        'errors if daemon and maximumPercent not 100'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                    daemon: true,
                    maxHealthyPercent: 300,
                });
            }, /Maximum percent must be 100 for daemon mode./);
            test.done();
        },
        'errors if minimum not less than maximum'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                    daemon: true,
                    minHealthyPercent: 100,
                    maxHealthyPercent: 100,
                });
            }, /Minimum healthy percent must be less than maximum healthy percent./);
            test.done();
        },
        'Output does not contain DesiredCount if daemon mode is set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // WHEN
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', (service) => {
                return service.LaunchType === base_service_1.LaunchType.EC2 && service.DesiredCount === undefined;
            }));
            test.done();
        },
        'errors if no container definitions'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                });
            });
            test.done();
        },
        'sets daemon scheduling strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                SchedulingStrategy: 'DAEMON',
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
            test.done();
        },
        'with a TaskDefinition with Bridge network mode': {
            'it errors if vpcSubnets is specified'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                // THEN
                test.throws(() => {
                    new ecs.Ec2Service(stack, 'Ec2Service', {
                        cluster,
                        taskDefinition,
                        vpcSubnets: {
                            subnetType: ec2.SubnetType.PUBLIC,
                        },
                    });
                });
                // THEN
                test.done();
            },
            'it errors if assignPublicIp is true'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                // THEN
                test.throws(() => {
                    new ecs.Ec2Service(stack, 'Ec2Service', {
                        cluster,
                        taskDefinition,
                        assignPublicIp: true,
                    });
                }, /vpcSubnets, securityGroup\(s\) and assignPublicIp can only be used in AwsVpc networking mode/);
                // THEN
                test.done();
            },
            'it errors if vpc subnets is provided'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const subnet = new ec2.Subnet(stack, 'MySubnet', {
                    vpcId: vpc.vpcId,
                    availabilityZone: 'eu-central-1a',
                    cidrBlock: '10.10.0.0/20',
                });
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                // THEN
                test.throws(() => {
                    new ecs.Ec2Service(stack, 'Ec2Service', {
                        cluster,
                        taskDefinition,
                        vpcSubnets: {
                            subnets: [subnet],
                        },
                    });
                }, /vpcSubnets, securityGroup\(s\) and assignPublicIp can only be used in AwsVpc networking mode/);
                // THEN
                test.done();
            },
            'it errors if security group is provided'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const securityGroup = new ec2.SecurityGroup(stack, 'MySG', { vpc });
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                // THEN
                test.throws(() => {
                    new ecs.Ec2Service(stack, 'Ec2Service', {
                        cluster,
                        taskDefinition,
                        securityGroup,
                    });
                }, /vpcSubnets, securityGroup\(s\) and assignPublicIp can only be used in AwsVpc networking mode/);
                // THEN
                test.done();
            },
            'it errors if multiple security groups is provided'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const securityGroups = [
                    new ec2.SecurityGroup(stack, 'MyFirstSG', { vpc }),
                    new ec2.SecurityGroup(stack, 'MySecondSG', { vpc }),
                ];
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                // THEN
                test.throws(() => {
                    new ecs.Ec2Service(stack, 'Ec2Service', {
                        cluster,
                        taskDefinition,
                        securityGroups,
                    });
                }, /vpcSubnets, securityGroup\(s\) and assignPublicIp can only be used in AwsVpc networking mode/);
                // THEN
                test.done();
            },
        },
        'with a TaskDefinition with AwsVpc network mode': {
            'it creates a security group for the service'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                    NetworkConfiguration: {
                        AwsvpcConfiguration: {
                            AssignPublicIp: 'DISABLED',
                            SecurityGroups: [
                                {
                                    'Fn::GetAtt': [
                                        'Ec2ServiceSecurityGroupAEC30825',
                                        'GroupId',
                                    ],
                                },
                            ],
                            Subnets: [
                                {
                                    Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                                },
                                {
                                    Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                                },
                            ],
                        },
                    },
                }));
                test.done();
            },
            'it allows vpcSubnets'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                taskDefinition.addContainer('web', {
                    image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                    memoryLimitMiB: 512,
                });
                new ecs.Ec2Service(stack, 'Ec2Service', {
                    cluster,
                    taskDefinition,
                    vpcSubnets: {
                        subnetType: ec2.SubnetType.PUBLIC,
                    },
                });
                // THEN
                test.done();
            },
        },
        'with distinctInstance placement constraint'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                placementConstraints: [ecs.PlacementConstraint.distinctInstances()],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementConstraints: [{
                        Type: 'distinctInstance',
                    }],
            }));
            test.done();
        },
        'with memberOf placement constraints'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementConstraints(placement_1.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementConstraints: [{
                        Expression: 'attribute:ecs.instance-type =~ t2.*',
                        Type: 'memberOf',
                    }],
            }));
            test.done();
        },
        'with spreadAcross container instances strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            // WHEN
            service.addPlacementStrategies(placement_1.PlacementStrategy.spreadAcrossInstances());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Field: 'instanceId',
                        Type: 'spread',
                    }],
            }));
            test.done();
        },
        'with spreadAcross placement strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementStrategies(placement_1.PlacementStrategy.spreadAcross(ecs.BuiltInAttributes.AVAILABILITY_ZONE));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Field: 'attribute:ecs.availability-zone',
                        Type: 'spread',
                    }],
            }));
            test.done();
        },
        'can turn PlacementStrategy into json format'(test) {
            // THEN
            test.deepEqual(placement_1.PlacementStrategy.spreadAcross(ecs.BuiltInAttributes.AVAILABILITY_ZONE).toJson(), [{
                    type: 'spread',
                    field: 'attribute:ecs.availability-zone',
                }]);
            test.done();
        },
        'can turn PlacementConstraints into json format'(test) {
            // THEN
            test.deepEqual(placement_1.PlacementConstraint.distinctInstances().toJson(), [{
                    type: 'distinctInstance',
                }]);
            test.done();
        },
        'errors when spreadAcross with no input'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            test.throws(() => {
                service.addPlacementStrategies(placement_1.PlacementStrategy.spreadAcross());
            }, 'spreadAcross: give at least one field to spread by');
            test.done();
        },
        'errors with spreadAcross placement strategy if daemon specified'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            test.throws(() => {
                service.addPlacementStrategies(placement_1.PlacementStrategy.spreadAcross(ecs.BuiltInAttributes.AVAILABILITY_ZONE));
            });
            test.done();
        },
        'with no placement constraints'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ECS::Service', {
                PlacementConstraints: undefined,
            }));
            test.done();
        },
        'with both propagateTags and propagateTaskTagsFrom defined'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            test.throws(() => new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                propagateTags: base_service_1.PropagatedTagSource.SERVICE,
                propagateTaskTagsFrom: base_service_1.PropagatedTagSource.SERVICE,
            }));
            test.done();
        },
        'with no placement strategy if daemon specified'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: undefined,
            }));
            test.done();
        },
        'with random placement strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc');
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementStrategies(placement_1.PlacementStrategy.randomly());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Type: 'random',
                    }],
            }));
            test.done();
        },
        'errors with random placement strategy if daemon specified'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc');
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            test.throws(() => {
                service.addPlacementStrategies(placement_1.PlacementStrategy.randomly());
            });
            test.done();
        },
        'with packedbyCpu placement strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementStrategies(placement_1.PlacementStrategy.packedByCpu());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Field: 'cpu',
                        Type: 'binpack',
                    }],
            }));
            test.done();
        },
        'with packedbyMemory placement strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementStrategies(placement_1.PlacementStrategy.packedByMemory());
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Field: 'memory',
                        Type: 'binpack',
                    }],
            }));
            test.done();
        },
        'with packedBy placement strategy'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
            });
            service.addPlacementStrategies(placement_1.PlacementStrategy.packedBy(ecs.BinPackResource.MEMORY));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                PlacementStrategies: [{
                        Field: 'memory',
                        Type: 'binpack',
                    }],
            }));
            test.done();
        },
        'errors with packedBy placement strategy if daemon specified'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const service = new ecs.Ec2Service(stack, 'Ec2Service', {
                cluster,
                taskDefinition,
                daemon: true,
            });
            // THEN
            test.throws(() => {
                service.addPlacementStrategies(placement_1.PlacementStrategy.packedBy(ecs.BinPackResource.MEMORY));
            });
            test.done();
        },
    },
    'attachToClassicLB': {
        'allows network mode of task definition to be host'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.HOST });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            service.attachToClassicLB(lb);
            test.done();
        },
        'allows network mode of task definition to be bridge'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.BRIDGE });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            service.attachToClassicLB(lb);
            test.done();
        },
        'throws when network mode of task definition is AwsVpc'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.AWS_VPC });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            test.throws(() => {
                service.attachToClassicLB(lb);
            }, /Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead./);
            test.done();
        },
        'throws when network mode of task definition is none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.NONE });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // THEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            test.throws(() => {
                service.attachToClassicLB(lb);
            }, /Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead./);
            test.done();
        },
    },
    'attachToApplicationTargetGroup': {
        'allows network mode of task definition to be other than none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.AWS_VPC });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
            });
            // THEN
            service.attachToApplicationTargetGroup(targetGroup);
            test.done();
        },
        'throws when network mode of task definition is none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.NONE });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
            });
            // THEN
            test.throws(() => {
                service.attachToApplicationTargetGroup(targetGroup);
            }, /Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead./);
            test.done();
        },
        'correctly setting ingress and egress port': {
            'with bridge/NAT network mode and 0 host port'(test) {
                [ecs.NetworkMode.BRIDGE, ecs.NetworkMode.NAT].forEach((networkMode) => {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                    const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode });
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                        memoryLimitMiB: 512,
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    container.addPortMappings({ containerPort: 8001 });
                    const service = new ecs.Ec2Service(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    listener.addTargets('target', {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: 'MainContainer',
                                containerPort: 8001,
                            })],
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                        Description: 'Load balancer to target',
                        FromPort: 32768,
                        ToPort: 65535,
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                        Description: 'Load balancer to target',
                        FromPort: 32768,
                        ToPort: 65535,
                    }));
                });
                test.done();
            },
            'with bridge/NAT network mode and host port other than 0'(test) {
                [ecs.NetworkMode.BRIDGE, ecs.NetworkMode.NAT].forEach((networkMode) => {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                    const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode });
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                        memoryLimitMiB: 512,
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    container.addPortMappings({ containerPort: 8001, hostPort: 80 });
                    const service = new ecs.Ec2Service(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    listener.addTargets('target', {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: 'MainContainer',
                                containerPort: 8001,
                            })],
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                        Description: 'Load balancer to target',
                        FromPort: 80,
                        ToPort: 80,
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                        Description: 'Load balancer to target',
                        FromPort: 80,
                        ToPort: 80,
                    }));
                });
                test.done();
            },
            'with host network mode'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.HOST });
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                    memoryLimitMiB: 512,
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.Ec2Service(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                listener.addTargets('target', {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: 'MainContainer',
                            containerPort: 8001,
                        })],
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                    Description: 'Load balancer to target',
                    FromPort: 8001,
                    ToPort: 8001,
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                    Description: 'Load balancer to target',
                    FromPort: 8001,
                    ToPort: 8001,
                }));
                test.done();
            },
            'with aws_vpc network mode'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.AWS_VPC });
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                    memoryLimitMiB: 512,
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.Ec2Service(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                listener.addTargets('target', {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: 'MainContainer',
                            containerPort: 8001,
                        })],
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                    Description: 'Load balancer to target',
                    FromPort: 8001,
                    ToPort: 8001,
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                    Description: 'Load balancer to target',
                    FromPort: 8001,
                    ToPort: 8001,
                }));
                test.done();
            },
        },
    },
    'attachToNetworkTargetGroup': {
        'allows network mode of task definition to be other than none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.AWS_VPC });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            const lb = new elbv2.NetworkLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
            });
            // THEN
            service.attachToNetworkTargetGroup(targetGroup);
            test.done();
        },
        'throws when network mode of task definition is none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', { networkMode: ecs.NetworkMode.NONE });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            const lb = new elbv2.NetworkLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
            });
            // THEN
            test.throws(() => {
                service.attachToNetworkTargetGroup(targetGroup);
            }, /Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead./);
            test.done();
        },
    },
    'classic ELB': {
        'can attach to classic ELB'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.HOST });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // WHEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            lb.addTarget(service);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                LoadBalancers: [
                    {
                        ContainerName: 'web',
                        ContainerPort: 808,
                        LoadBalancerName: { Ref: 'LB8A12904C' },
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                // if any load balancer is configured and healthCheckGracePeriodSeconds is not
                // set, then it should default to 60 seconds.
                HealthCheckGracePeriodSeconds: 60,
            }));
            test.done();
        },
        'can attach any container and port as a target'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const cluster = new ecs.Cluster(stack, 'Cluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TD', { networkMode: ecs.NetworkMode.HOST });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
            });
            container.addPortMappings({ containerPort: 808 });
            container.addPortMappings({ containerPort: 8080 });
            const service = new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            // WHEN
            const lb = new elb.LoadBalancer(stack, 'LB', { vpc });
            lb.addTarget(service.loadBalancerTarget({
                containerName: 'web',
                containerPort: 8080,
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                LoadBalancers: [
                    {
                        ContainerName: 'web',
                        ContainerPort: 8080,
                        LoadBalancerName: { Ref: 'LB8A12904C' },
                    },
                ],
            }));
            test.done();
        },
    },
    'When enabling service discovery': {
        'throws if namespace has not been added to cluster'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            // default network mode is bridge
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Service', {
                    cluster,
                    taskDefinition,
                    cloudMapOptions: {
                        name: 'myApp',
                    },
                });
            }, /Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster./);
            test.done();
        },
        'throws if network mode is none'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.NONE,
            });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            cluster.addDefaultCloudMapNamespace({ name: 'foo.com' });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Service', {
                    cluster,
                    taskDefinition,
                    cloudMapOptions: {
                        name: 'myApp',
                    },
                });
            }, /Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead./);
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with bridge network mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            // default network mode is bridge
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                ServiceRegistries: [
                    {
                        ContainerName: 'MainContainer',
                        ContainerPort: 8000,
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'ServiceCloudmapService046058A4',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with host network mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.HOST,
            });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                ServiceRegistries: [
                    {
                        ContainerName: 'MainContainer',
                        ContainerPort: 8000,
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'ServiceCloudmapService046058A4',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
        'throws if wrong DNS record type specified with bridge network mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            // default network mode is bridge
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
            });
            // THEN
            test.throws(() => {
                new ecs.Ec2Service(stack, 'Service', {
                    cluster,
                    taskDefinition,
                    cloudMapOptions: {
                        name: 'myApp',
                        dnsRecordType: cloudmap.DnsRecordType.A,
                    },
                });
            }, /SRV records must be used when network mode is Bridge or Host./);
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with AwsVpc network mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                ServiceRegistries: [
                    {
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'ServiceCloudmapService046058A4',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'A',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with AwsVpc network mode with SRV records'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.Ec2Service(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    dnsRecordType: cloudmap.DnsRecordType.SRV,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                ServiceRegistries: [
                    {
                        ContainerName: 'MainContainer',
                        ContainerPort: 8000,
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'ServiceCloudmapService046058A4',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
    },
    'Metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'FargateTaskDef');
        taskDefinition.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        // WHEN
        const service = new ecs.Ec2Service(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // THEN
        test.deepEqual(stack.resolve(service.metricMemoryUtilization()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
                ServiceName: { 'Fn::GetAtt': ['ServiceD69D759B', 'Name'] },
            },
            namespace: 'AWS/ECS',
            metricName: 'MemoryUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.deepEqual(stack.resolve(service.metricCpuUtilization()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
                ServiceName: { 'Fn::GetAtt': ['ServiceD69D759B', 'Name'] },
            },
            namespace: 'AWS/ECS',
            metricName: 'CPUUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'When import an EC2 Service': {
        'with serviceArn'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            // WHEN
            const service = ecs.Ec2Service.fromEc2ServiceAttributes(stack, 'EcsService', {
                serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
                cluster,
            });
            // THEN
            test.equal(service.serviceArn, 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');
            test.equal(service.serviceName, 'my-http-service');
            test.done();
        },
        'with serviceName'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const pseudo = new cdk.ScopedAws(stack);
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            // WHEN
            const service = ecs.Ec2Service.fromEc2ServiceAttributes(stack, 'EcsService', {
                serviceName: 'my-http-service',
                cluster,
            });
            // THEN
            test.deepEqual(stack.resolve(service.serviceArn), stack.resolve(`arn:${pseudo.partition}:ecs:${pseudo.region}:${pseudo.accountId}:service/my-http-service`));
            test.equal(service.serviceName, 'my-http-service');
            test.done();
        },
        'throws an exception if both serviceArn and serviceName were provided for fromEc2ServiceAttributes'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            test.throws(() => {
                ecs.Ec2Service.fromEc2ServiceAttributes(stack, 'EcsService', {
                    serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
                    serviceName: 'my-http-service',
                    cluster,
                });
            }, /only specify either serviceArn or serviceName/);
            test.done();
        },
        'throws an exception if neither serviceArn nor serviceName were provided for fromEc2ServiceAttributes'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            test.throws(() => {
                ecs.Ec2Service.fromEc2ServiceAttributes(stack, 'EcsService', {
                    cluster,
                });
            }, /only specify either serviceArn or serviceName/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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