"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_1 = require("@aws-cdk/aws-ecr");
const iam = require("@aws-cdk/aws-iam");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const path = require("path");
const ecs = require("../../lib");
module.exports = {
    'When creating an ECS TaskDefinition': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                NetworkMode: ecs.NetworkMode.BRIDGE,
                RequiresCompatibilities: ['EC2'],
            }));
            // test error if no container defs?
            test.done();
        },
        'with all properties set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                executionRole: new iam.Role(stack, 'ExecutionRole', {
                    path: '/',
                    assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com')),
                }),
                family: 'ecs-tasks',
                networkMode: ecs.NetworkMode.AWS_VPC,
                ipcMode: ecs.IpcMode.HOST,
                pidMode: ecs.PidMode.TASK,
                placementConstraints: [ecs.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*')],
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                }),
                volumes: [{
                        host: {
                            sourcePath: '/tmp/cache',
                        },
                        name: 'scratch',
                    }],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ExecutionRole605A040B',
                        'Arn',
                    ],
                },
                Family: 'ecs-tasks',
                NetworkMode: 'awsvpc',
                IpcMode: 'host',
                PidMode: 'task',
                PlacementConstraints: [
                    {
                        Expression: 'attribute:ecs.instance-type =~ t2.*',
                        Type: 'memberOf',
                    },
                ],
                RequiresCompatibilities: [
                    'EC2',
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        'TaskRole30FC0FBB',
                        'Arn',
                    ],
                },
                Volumes: [
                    {
                        Host: {
                            SourcePath: '/tmp/cache',
                        },
                        Name: 'scratch',
                    },
                ],
            }));
            test.done();
        },
        'correctly sets placement constraint'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addPlacementConstraint(ecs.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                PlacementConstraints: [
                    {
                        Expression: 'attribute:ecs.instance-type =~ t2.*',
                        Type: 'memberOf',
                    },
                ],
            }));
            test.done();
        },
        'correctly sets network mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                NetworkMode: ecs.NetworkMode.AWS_VPC,
            }));
            test.done();
        },
        'correctly sets ipc mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                ipcMode: ecs.IpcMode.TASK,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                IpcMode: ecs.IpcMode.TASK,
            }));
            test.done();
        },
        'correctly sets pid mode'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                pidMode: ecs.PidMode.HOST,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                PidMode: ecs.PidMode.HOST,
            }));
            test.done();
        },
        'correctly sets containers'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({
                containerPort: 3000,
            });
            container.addUlimits({
                hardLimit: 128,
                name: ecs.UlimitName.RSS,
                softLimit: 128,
            });
            container.addVolumesFrom({
                sourceContainer: 'foo',
                readOnly: true,
            });
            container.addToExecutionPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['ecs:*'],
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [{
                        Essential: true,
                        Memory: 512,
                        Image: 'amazon/amazon-ecs-sample',
                        Name: 'web',
                        PortMappings: [{
                                ContainerPort: 3000,
                                HostPort: 0,
                                Protocol: aws_ec2_1.Protocol.TCP,
                            }],
                        Ulimits: [
                            {
                                HardLimit: 128,
                                Name: 'rss',
                                SoftLimit: 128,
                            },
                        ],
                        VolumesFrom: [
                            {
                                ReadOnly: true,
                                SourceContainer: 'foo',
                            },
                        ],
                    }],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'ecs:*',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                },
            }));
            test.done();
        },
        'all container definition options defined'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const secret = new secretsmanager.Secret(stack, 'Secret');
            const parameter = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'Parameter', {
                parameterName: '/name',
                version: 1,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 2048,
                cpu: 256,
                disableNetworking: true,
                command: ['CMD env'],
                dnsSearchDomains: ['0.0.0.0'],
                dnsServers: ['1.1.1.1'],
                dockerLabels: { LABEL: 'label' },
                dockerSecurityOptions: ['ECS_SELINUX_CAPABLE=true'],
                entryPoint: ['/app/node_modules/.bin/cdk'],
                environment: { TEST_ENVIRONMENT_VARIABLE: 'test environment variable value' },
                essential: true,
                extraHosts: { EXTRAHOST: 'extra host' },
                healthCheck: {
                    command: ['curl localhost:8000'],
                    interval: cdk.Duration.seconds(20),
                    retries: 5,
                    startPeriod: cdk.Duration.seconds(10),
                },
                hostname: 'webHost',
                linuxParameters: new ecs.LinuxParameters(stack, 'LinuxParameters', {
                    initProcessEnabled: true,
                    sharedMemorySize: 1024,
                }),
                logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' }),
                memoryReservationMiB: 1024,
                privileged: true,
                readonlyRootFilesystem: true,
                secrets: {
                    SECRET: ecs.Secret.fromSecretsManager(secret),
                    PARAMETER: ecs.Secret.fromSsmParameter(parameter),
                },
                user: 'amazon',
                workingDirectory: 'app/',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [
                    {
                        Command: [
                            'CMD env',
                        ],
                        Cpu: 256,
                        DisableNetworking: true,
                        DnsSearchDomains: [
                            '0.0.0.0',
                        ],
                        DnsServers: [
                            '1.1.1.1',
                        ],
                        DockerLabels: {
                            LABEL: 'label',
                        },
                        DockerSecurityOptions: [
                            'ECS_SELINUX_CAPABLE=true',
                        ],
                        EntryPoint: [
                            '/app/node_modules/.bin/cdk',
                        ],
                        Environment: [
                            {
                                Name: 'TEST_ENVIRONMENT_VARIABLE',
                                Value: 'test environment variable value',
                            },
                        ],
                        Essential: true,
                        ExtraHosts: [
                            {
                                Hostname: 'EXTRAHOST',
                                IpAddress: 'extra host',
                            },
                        ],
                        HealthCheck: {
                            Command: [
                                'CMD-SHELL',
                                'curl localhost:8000',
                            ],
                            Interval: 20,
                            Retries: 5,
                            StartPeriod: 10,
                            Timeout: 5,
                        },
                        Hostname: 'webHost',
                        Image: 'amazon/amazon-ecs-sample',
                        LinuxParameters: {
                            Capabilities: {},
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'Ec2TaskDefwebLogGroup7F786C6B',
                                },
                                'awslogs-stream-prefix': 'prefix',
                                'awslogs-region': {
                                    Ref: 'AWS::Region',
                                },
                            },
                        },
                        Memory: 2048,
                        MemoryReservation: 1024,
                        Name: 'web',
                        Privileged: true,
                        ReadonlyRootFilesystem: true,
                        Secrets: [
                            {
                                Name: 'SECRET',
                                ValueFrom: {
                                    Ref: 'SecretA720EF05',
                                },
                            },
                            {
                                Name: 'PARAMETER',
                                ValueFrom: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition',
                                            },
                                            ':ssm:',
                                            {
                                                Ref: 'AWS::Region',
                                            },
                                            ':',
                                            {
                                                Ref: 'AWS::AccountId',
                                            },
                                            ':parameter/name',
                                        ],
                                    ],
                                },
                            },
                        ],
                        User: 'amazon',
                        WorkingDirectory: 'app/',
                    },
                ],
            }));
            test.done();
        },
        'correctly sets containers from ECR repository using all props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromEcrRepository(new aws_ecr_1.Repository(stack, 'myECRImage', {
                    lifecycleRegistryId: '123456789101',
                    lifecycleRules: [{
                            rulePriority: 10,
                            tagPrefixList: ['abc'],
                            maxImageCount: 1,
                        }],
                    removalPolicy: cdk.RemovalPolicy.DESTROY,
                    repositoryName: 'project-a/amazon-ecs-sample',
                })),
                memoryLimitMiB: 512,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECR::Repository', {
                LifecyclePolicy: {
                    // eslint-disable-next-line max-len
                    LifecyclePolicyText: '{"rules":[{"rulePriority":10,"selection":{"tagStatus":"tagged","tagPrefixList":["abc"],"countType":"imageCountMoreThan","countNumber":1},"action":{"type":"expire"}}]}',
                    RegistryId: '123456789101',
                },
                RepositoryName: 'project-a/amazon-ecs-sample',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [{
                        Essential: true,
                        Memory: 512,
                        Image: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::Select': [
                                            4,
                                            {
                                                'Fn::Split': [
                                                    ':',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'myECRImage7DEAE474',
                                                            'Arn',
                                                        ],
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                    '.dkr.ecr.',
                                    {
                                        'Fn::Select': [
                                            3,
                                            {
                                                'Fn::Split': [
                                                    ':',
                                                    {
                                                        'Fn::GetAtt': [
                                                            'myECRImage7DEAE474',
                                                            'Arn',
                                                        ],
                                                    },
                                                ],
                                            },
                                        ],
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                    '/',
                                    {
                                        Ref: 'myECRImage7DEAE474',
                                    },
                                    ':latest',
                                ],
                            ],
                        },
                        Name: 'web',
                    }],
            }));
            test.done();
        },
        'correctly sets containers from ECR repository using default props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromEcrRepository(new aws_ecr_1.Repository(stack, 'myECRImage')),
                memoryLimitMiB: 512,
            });
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ECR::Repository', {}));
            test.done();
        },
        'warns when setting containers from ECR repository using fromRegistry method'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY'),
                memoryLimitMiB: 512,
            });
            // THEN
            test.deepEqual(container.node.metadata[0].data, "Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
            test.done();
        },
        'warns when setting containers from ECR repository by creating a RepositoryImage class'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const repo = new ecs.RepositoryImage('ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY');
            // WHEN
            const container = taskDefinition.addContainer('web', {
                image: repo,
                memoryLimitMiB: 512,
            });
            // THEN
            test.deepEqual(container.node.metadata[0].data, "Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
            test.done();
        },
        'correctly sets containers from asset using default props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '..', 'demo-image')),
                memoryLimitMiB: 512,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [
                    {
                        Essential: true,
                        Image: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '.dkr.ecr.',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    '.',
                                    {
                                        Ref: 'AWS::URLSuffix',
                                    },
                                    '/aws-cdk/assets:baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
                                ],
                            ],
                        },
                        Memory: 512,
                        Name: 'web',
                    },
                ],
            }));
            test.done();
        },
        'correctly sets containers from asset using all props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromAsset(path.join(__dirname, '..', 'demo-image'), {
                    buildArgs: { HTTP_PROXY: 'http://10.20.30.2:1234' },
                }),
                memoryLimitMiB: 512,
            });
            test.done();
        },
        'correctly sets scratch space'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addScratch({
                containerPath: './cache',
                readOnly: true,
                sourcePath: '/tmp/cache',
                name: 'scratch',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [{
                        MountPoints: [
                            {
                                ContainerPath: './cache',
                                ReadOnly: true,
                                SourceVolume: 'scratch',
                            },
                        ],
                    }],
                Volumes: [{
                        Host: {
                            SourcePath: '/tmp/cache',
                        },
                        Name: 'scratch',
                    }],
            }));
            test.done();
        },
        'correctly sets container dependenices'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            const dependency1 = taskDefinition.addContainer('dependency1', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const dependency2 = taskDefinition.addContainer('dependency2', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addContainerDependencies({
                container: dependency1,
            }, {
                container: dependency2,
                condition: ecs.ContainerDependencyCondition.SUCCESS,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [{
                        Name: 'dependency1',
                    },
                    {
                        Name: 'dependency2',
                    },
                    {
                        Name: 'web',
                        DependsOn: [{
                                Condition: 'HEALTHY',
                                ContainerName: 'dependency1',
                            },
                            {
                                Condition: 'SUCCESS',
                                ContainerName: 'dependency2',
                            }],
                    }],
            }));
            test.done();
        },
        'correctly sets links'(test) {
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.BRIDGE,
            });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const linkedContainer1 = taskDefinition.addContainer('linked1', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            const linkedContainer2 = taskDefinition.addContainer('linked2', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addLink(linkedContainer1, 'linked');
            container.addLink(linkedContainer2);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Links: [
                            'linked1:linked',
                            'linked2',
                        ],
                        Name: 'web',
                    },
                    {
                        Name: 'linked1',
                    },
                    {
                        Name: 'linked2',
                    },
                ],
            }));
            test.done();
        },
        'correctly set policy statement to the task IAM role'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addToTaskRolePolicy(new iam.PolicyStatement({
                actions: ['test:SpecialName'],
                resources: ['*'],
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'test:SpecialName',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                },
            }));
            test.done();
        },
        'correctly sets volumes from'(test) {
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {});
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addVolumesFrom({
                sourceContainer: 'SourceContainer',
                readOnly: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [{
                        VolumesFrom: [
                            {
                                SourceContainer: 'SourceContainer',
                                ReadOnly: true,
                            },
                        ],
                    }],
            }));
            test.done();
        },
        'correctly set policy statement to the task execution IAM role'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // WHEN
            taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                actions: ['test:SpecialName'],
                resources: ['*'],
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'test:SpecialName',
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                },
            }));
            test.done();
        },
        'correctly sets volumes'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const volume = {
                host: {
                    sourcePath: '/tmp/cache',
                },
                name: 'scratch',
            };
            // Adding volumes via props is a bit clunky
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                volumes: [volume],
            });
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            // this needs to be a better API -- should auto-add volumes
            container.addMountPoints({
                containerPath: './cache',
                readOnly: true,
                sourceVolume: 'scratch',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                ContainerDefinitions: [{
                        MountPoints: [
                            {
                                ContainerPath: './cache',
                                ReadOnly: true,
                                SourceVolume: 'scratch',
                            },
                        ],
                    }],
                Volumes: [{
                        Host: {
                            SourcePath: '/tmp/cache',
                        },
                        Name: 'scratch',
                    }],
            }));
            test.done();
        },
        'correctly sets placement constraints'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                placementConstraints: [
                    ecs.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*'),
                ],
            });
            taskDefinition.addContainer('web', {
                memoryLimitMiB: 1024,
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                PlacementConstraints: [
                    {
                        Expression: 'attribute:ecs.instance-type =~ t2.*',
                        Type: 'memberOf',
                    },
                ],
            }));
            test.done();
        },
        'correctly sets taskRole'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                }),
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                TaskRoleArn: stack.resolve(taskDefinition.taskRole.roleArn),
            }));
            test.done();
        },
        'automatically sets taskRole by default'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                TaskRoleArn: stack.resolve(taskDefinition.taskRole.roleArn),
            }));
            test.done();
        },
        'correctly sets dockerVolumeConfiguration'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const volume = {
                name: 'scratch',
                dockerVolumeConfiguration: {
                    driver: 'local',
                    scope: ecs.Scope.TASK,
                    driverOpts: {
                        key1: 'value',
                    },
                },
            };
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                volumes: [volume],
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Family: 'Ec2TaskDef',
                Volumes: [{
                        Name: 'scratch',
                        DockerVolumeConfiguration: {
                            Driver: 'local',
                            Scope: 'task',
                            DriverOpts: {
                                key1: 'value',
                            },
                        },
                    }],
            }));
            test.done();
        },
    },
    'throws when setting proxyConfiguration without networkMode AWS_VPC'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const proxyConfiguration = ecs.ProxyConfigurations.appMeshProxyConfiguration({
            containerName: 'envoy',
            properties: {
                ignoredUID: 1337,
                proxyIngressPort: 15000,
                proxyEgressPort: 15001,
                appPorts: [9080, 9081],
                egressIgnoredIPs: ['169.254.170.2', '169.254.169.254'],
            },
        });
        // THEN
        test.throws(() => {
            new ecs.Ec2TaskDefinition(stack, 'TaskDef', { networkMode: ecs.NetworkMode.BRIDGE, proxyConfiguration });
        }, /ProxyConfiguration can only be used with AwsVpc network mode, got: bridge/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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