"use strict";
const assert_1 = require("@aws-cdk/assert");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecs = require("../../lib");
const base_service_1 = require("../../lib/base/base-service");
module.exports = {
    'When creating a Fargate Service': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'FargateTaskDefC6FB60B4',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50,
                },
                DesiredCount: 1,
                LaunchType: base_service_1.LaunchType.FARGATE,
                EnableECSManagedTags: false,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'DISABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'FargateServiceSecurityGroup0A0E79CB',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                            },
                            {
                                Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                            },
                        ],
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Default/FargateService/SecurityGroup',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            test.done();
        },
        'can create service with default settings if VPC only has public subnets'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {
                subnetConfiguration: [
                    {
                        cidrMask: 28,
                        name: 'public-only',
                        subnetType: ec2.SubnetType.PUBLIC,
                    },
                ],
            });
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            // WHEN
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
            });
            // THEN -- did not throw
            test.done();
        },
        'with custom cloudmap namespace'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            const cloudMapNamespace = new cloudmap.PrivateDnsNamespace(stack, 'TestCloudMapNamespace', {
                name: 'scorekeep.com',
                vpc,
            });
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    failureThreshold: 20,
                    cloudMapNamespace,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'A',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'TestCloudMapNamespace1FB9B446',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 20,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'TestCloudMapNamespace1FB9B446',
                        'Id',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PrivateDnsNamespace', {
                Name: 'scorekeep.com',
                Vpc: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            test.done();
        },
        'with all properties set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            const svc = new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                desiredCount: 2,
                assignPublicIp: true,
                cloudMapOptions: {
                    name: 'myapp',
                    dnsRecordType: cloudmap.DnsRecordType.A,
                    dnsTtl: cdk.Duration.seconds(50),
                    failureThreshold: 20,
                },
                healthCheckGracePeriod: cdk.Duration.seconds(60),
                maxHealthyPercent: 150,
                minHealthyPercent: 55,
                deploymentController: {
                    type: ecs.DeploymentControllerType.CODE_DEPLOY,
                },
                securityGroup: new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                    allowAllOutbound: true,
                    description: 'Example',
                    securityGroupName: 'Bob',
                    vpc,
                }),
                serviceName: 'bonjour',
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            });
            // THEN
            test.ok(svc.cloudMapService !== undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'FargateTaskDefC6FB60B4',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 150,
                    MinimumHealthyPercent: 55,
                },
                DeploymentController: {
                    Type: ecs.DeploymentControllerType.CODE_DEPLOY,
                },
                DesiredCount: 2,
                HealthCheckGracePeriodSeconds: 60,
                LaunchType: base_service_1.LaunchType.FARGATE,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup1F554B36F',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPublicSubnet1SubnetF6608456',
                            },
                            {
                                Ref: 'MyVpcPublicSubnet2Subnet492B6BFB',
                            },
                        ],
                    },
                },
                ServiceName: 'bonjour',
                ServiceRegistries: [
                    {
                        RegistryArn: {
                            'Fn::GetAtt': [
                                'FargateServiceCloudmapService9544B753',
                                'Arn',
                            ],
                        },
                    },
                ],
            }));
            test.done();
        },
        'throws when task definition is not Fargate compatible'(test) {
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.TaskDefinition(stack, 'Ec2TaskDef', {
                compatibility: ecs.Compatibility.EC2,
            });
            taskDefinition.addContainer('BaseContainer', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryReservationMiB: 10,
            });
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, 'FargateService', {
                    cluster,
                    taskDefinition,
                });
            }, /Supplied TaskDefinition is not configured for compatibility with Fargate/);
            test.done();
        },
        'ignore task definition and launch type if deployment controller is set to be EXTERNAL'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            const service = new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                deploymentController: {
                    type: base_service_1.DeploymentControllerType.EXTERNAL,
                },
            });
            // THEN
            test.deepEqual(service.node.metadata[0].data, 'taskDefinition and launchType are blanked out when using external deployment controller.');
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50,
                },
                DeploymentController: {
                    Type: 'EXTERNAL',
                },
                DesiredCount: 1,
                EnableECSManagedTags: false,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'DISABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'FargateServiceSecurityGroup0A0E79CB',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                            },
                            {
                                Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                            },
                        ],
                    },
                },
            }));
            test.done();
        },
        'errors when no container specified on task definition'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, 'FargateService', {
                    cluster,
                    taskDefinition,
                });
            });
            test.done();
        },
        'allows specifying assignPublicIP as enabled'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                assignPublicIp: true,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'ENABLED',
                    },
                },
            }));
            test.done();
        },
        'allows specifying 0 for minimumHealthyPercent'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                minHealthyPercent: 0,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MinimumHealthyPercent: 0,
                },
            }));
            test.done();
        },
        'throws when securityGroup and securityGroups are supplied'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const securityGroup1 = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                allowAllOutbound: true,
                description: 'Example',
                securityGroupName: 'Bingo',
                vpc,
            });
            const securityGroup2 = new ec2.SecurityGroup(stack, 'SecurityGroup2', {
                allowAllOutbound: false,
                description: 'Example',
                securityGroupName: 'Rolly',
                vpc,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, 'FargateService', {
                    cluster,
                    taskDefinition,
                    securityGroup: securityGroup1,
                    securityGroups: [securityGroup2],
                });
            }, /Only one of SecurityGroup or SecurityGroups can be populated./);
            test.done();
        },
        'with multiple securty groups, it correctly updates cloudformation template'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const securityGroup1 = new ec2.SecurityGroup(stack, 'SecurityGroup1', {
                allowAllOutbound: true,
                description: 'Example',
                securityGroupName: 'Bingo',
                vpc,
            });
            const securityGroup2 = new ec2.SecurityGroup(stack, 'SecurityGroup2', {
                allowAllOutbound: false,
                description: 'Example',
                securityGroupName: 'Rolly',
                vpc,
            });
            taskDefinition.addContainer('web', {
                image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
            });
            new ecs.FargateService(stack, 'FargateService', {
                cluster,
                taskDefinition,
                securityGroups: [securityGroup1, securityGroup2],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                TaskDefinition: {
                    Ref: 'FargateTaskDefC6FB60B4',
                },
                Cluster: {
                    Ref: 'EcsCluster97242B84',
                },
                DeploymentConfiguration: {
                    MaximumPercent: 200,
                    MinimumHealthyPercent: 50,
                },
                DesiredCount: 1,
                LaunchType: base_service_1.LaunchType.FARGATE,
                EnableECSManagedTags: false,
                NetworkConfiguration: {
                    AwsvpcConfiguration: {
                        AssignPublicIp: 'DISABLED',
                        SecurityGroups: [
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup1F554B36F',
                                    'GroupId',
                                ],
                            },
                            {
                                'Fn::GetAtt': [
                                    'SecurityGroup23BE86BB7',
                                    'GroupId',
                                ],
                            },
                        ],
                        Subnets: [
                            {
                                Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                            },
                            {
                                Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                            },
                        ],
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Example',
                GroupName: 'Bingo',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Example',
                GroupName: 'Rolly',
                SecurityGroupEgress: [
                    {
                        CidrIp: '255.255.255.255/32',
                        Description: 'Disallow all traffic',
                        FromPort: 252,
                        IpProtocol: 'icmp',
                        ToPort: 86,
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            test.done();
        },
    },
    'When setting up a health check': {
        'grace period is respected'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            // WHEN
            new ecs.FargateService(stack, 'Svc', {
                cluster,
                taskDefinition,
                healthCheckGracePeriod: cdk.Duration.seconds(10),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                HealthCheckGracePeriodSeconds: 10,
            }));
            test.done();
        },
    },
    'When adding an app load balancer': {
        'allows auto scaling by ALB request per target'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.FargateService(stack, 'Service', { cluster, taskDefinition });
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
                targets: [service],
            });
            // WHEN
            const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
            capacity.scaleOnRequestCount('ScaleOnRequests', {
                requestsPerTarget: 1000,
                targetGroup,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
                MaxCapacity: 10,
                MinCapacity: 1,
                ResourceId: {
                    'Fn::Join': [
                        '',
                        [
                            'service/',
                            {
                                Ref: 'EcsCluster97242B84',
                            },
                            '/',
                            {
                                'Fn::GetAtt': [
                                    'ServiceD69D759B',
                                    'Name',
                                ],
                            },
                        ],
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
                TargetTrackingScalingPolicyConfiguration: {
                    PredefinedMetricSpecification: {
                        PredefinedMetricType: 'ALBRequestCountPerTarget',
                        ResourceLabel: {
                            'Fn::Join': ['', [
                                    { 'Fn::Select': [1, { 'Fn::Split': ['/', { Ref: 'lblistener657ADDEC' }] }] }, '/',
                                    { 'Fn::Select': [2, { 'Fn::Split': ['/', { Ref: 'lblistener657ADDEC' }] }] }, '/',
                                    { 'Fn::Select': [3, { 'Fn::Split': ['/', { Ref: 'lblistener657ADDEC' }] }] }, '/',
                                    { 'Fn::GetAtt': ['lblistenertargetGroupC7489D1E', 'TargetGroupFullName'] },
                                ]],
                        },
                    },
                    TargetValue: 1000,
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                // if any load balancer is configured and healthCheckGracePeriodSeconds is not
                // set, then it should default to 60 seconds.
                HealthCheckGracePeriodSeconds: 60,
            }));
            test.done();
        },
        'allows auto scaling by ALB with new service arn format'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            const service = new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
            });
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
            const listener = lb.addListener('listener', { port: 80 });
            const targetGroup = listener.addTargets('target', {
                port: 80,
                targets: [service],
            });
            // WHEN
            const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
            capacity.scaleOnRequestCount('ScaleOnRequests', {
                requestsPerTarget: 1000,
                targetGroup,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
                MaxCapacity: 10,
                MinCapacity: 1,
                ResourceId: {
                    'Fn::Join': [
                        '',
                        [
                            'service/',
                            {
                                Ref: 'EcsCluster97242B84',
                            },
                            '/',
                            {
                                'Fn::GetAtt': [
                                    'ServiceD69D759B',
                                    'Name',
                                ],
                            },
                        ],
                    ],
                },
            }));
            test.done();
        },
        'allows specify any existing container name and port in a service': {
            'with default setting'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                listener.addTargets('target', {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: 'MainContainer',
                        })],
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                    LoadBalancers: [
                        {
                            ContainerName: 'MainContainer',
                            ContainerPort: 8000,
                            TargetGroupArn: {
                                Ref: 'lblistenertargetGroupC7489D1E',
                            },
                        },
                    ],
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                    Description: 'Load balancer to target',
                    FromPort: 8000,
                    ToPort: 8000,
                }));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                    Description: 'Load balancer to target',
                    FromPort: 8000,
                    ToPort: 8000,
                }));
                test.done();
            },
            'with TCP protocol'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.TCP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                // THEN
                listener.addTargets('target', {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: 'MainContainer',
                            containerPort: 8001,
                            protocol: ecs.Protocol.TCP,
                        })],
                });
                test.done();
            },
            'with UDP protocol'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.UDP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                // THEN
                listener.addTargets('target', {
                    port: 80,
                    targets: [service.loadBalancerTarget({
                            containerName: 'MainContainer',
                            containerPort: 8001,
                            protocol: ecs.Protocol.UDP,
                        })],
                });
                test.done();
            },
            'throws when protocol does not match'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001, protocol: ecs.Protocol.UDP });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets('target', {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: 'MainContainer',
                                containerPort: 8001,
                                protocol: ecs.Protocol.TCP,
                            })],
                    });
                }, /Container 'Default\/FargateTaskDef\/MainContainer' has no mapping for port 8001 and protocol tcp. Did you call "container.addPortMappings\(\)"\?/);
                test.done();
            },
            'throws when port does not match'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets('target', {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: 'MainContainer',
                                containerPort: 8002,
                            })],
                    });
                }, /Container 'Default\/FargateTaskDef\/MainContainer' has no mapping for port 8002 and protocol tcp. Did you call "container.addPortMappings\(\)"\?/);
                test.done();
            },
            'throws when container does not exist'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                const container = taskDefinition.addContainer('MainContainer', {
                    image: ecs.ContainerImage.fromRegistry('hello'),
                });
                container.addPortMappings({ containerPort: 8000 });
                container.addPortMappings({ containerPort: 8001 });
                const service = new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                });
                // WHEN
                const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                const listener = lb.addListener('listener', { port: 80 });
                // THEN
                test.throws(() => {
                    listener.addTargets('target', {
                        port: 80,
                        targets: [service.loadBalancerTarget({
                                containerName: 'SideContainer',
                                containerPort: 8001,
                            })],
                    });
                }, /No container named 'SideContainer'. Did you call "addContainer()"?/);
                test.done();
            },
        },
        'allows load balancing to any container and port of service': {
            'with application load balancers': {
                'with default target group port and protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: 'HTTP',
                    }));
                    test.done();
                },
                'with default target group port and HTTP protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            protocol: elbv2.ApplicationProtocol.HTTP,
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: 'HTTP',
                    }));
                    test.done();
                },
                'with default target group port and HTTPS protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            protocol: elbv2.ApplicationProtocol.HTTPS,
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 443,
                        Protocol: 'HTTPS',
                    }));
                    test.done();
                },
                'with any target group port and protocol'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.ApplicationLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.applicationListener(listener, {
                            port: 83,
                            protocol: elbv2.ApplicationProtocol.HTTP,
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 83,
                        Protocol: 'HTTP',
                    }));
                    test.done();
                },
            },
            'with network load balancers': {
                'with default target group port'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.NetworkLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.networkListener(listener),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 80,
                        Protocol: 'TCP',
                    }));
                    test.done();
                },
                'with any target group port'(test) {
                    // GIVEN
                    const stack = new cdk.Stack();
                    const vpc = new ec2.Vpc(stack, 'MyVpc', {});
                    const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
                    const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
                    const container = taskDefinition.addContainer('MainContainer', {
                        image: ecs.ContainerImage.fromRegistry('hello'),
                    });
                    container.addPortMappings({ containerPort: 8000 });
                    const service = new ecs.FargateService(stack, 'Service', {
                        cluster,
                        taskDefinition,
                    });
                    // WHEN
                    const lb = new elbv2.NetworkLoadBalancer(stack, 'lb', { vpc });
                    const listener = lb.addListener('listener', { port: 80 });
                    service.registerLoadBalancerTargets({
                        containerName: 'MainContainer',
                        containerPort: 8000,
                        listener: ecs.ListenerConfig.networkListener(listener, {
                            port: 81,
                        }),
                        newTargetGroupId: 'target1',
                    });
                    // THEN
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service', {
                        LoadBalancers: [
                            {
                                ContainerName: 'MainContainer',
                                ContainerPort: 8000,
                                TargetGroupArn: {
                                    Ref: 'lblistenertarget1Group1A1A5C9E',
                                },
                            },
                        ],
                    }));
                    assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
                        Port: 81,
                        Protocol: 'TCP',
                    }));
                    test.done();
                },
            },
        },
    },
    'allows scaling on a specified scheduled time'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnSchedule('ScaleOnSchedule', {
            schedule: appscaling.Schedule.cron({ hour: '8', minute: '0' }),
            minCapacity: 10,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: {
                        MinCapacity: 10,
                    },
                    Schedule: 'cron(0 8 * * ? *)',
                    ScheduledActionName: 'ScaleOnSchedule',
                },
            ],
        }));
        test.done();
    },
    'allows scaling on a specified metric value'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnMetric('ScaleOnMetric', {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            scalingSteps: [
                { upper: 0, change: -1 },
                { lower: 100, change: +1 },
                { lower: 500, change: +5 },
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'StepScaling',
            ScalingTargetId: {
                Ref: 'ServiceTaskCountTarget23E25614',
            },
            StepScalingPolicyConfiguration: {
                AdjustmentType: 'ChangeInCapacity',
                MetricAggregationType: 'Average',
                StepAdjustments: [
                    {
                        MetricIntervalUpperBound: 0,
                        ScalingAdjustment: -1,
                    },
                ],
            },
        }));
        test.done();
    },
    'allows scaling on a target CPU utilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnCpuUtilization('ScaleOnCpu', {
            targetUtilizationPercent: 30,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'ECSServiceAverageCPUUtilization' },
                TargetValue: 30,
            },
        }));
        test.done();
    },
    'allows scaling on memory utilization'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleOnMemoryUtilization('ScaleOnMemory', {
            targetUtilizationPercent: 30,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'ECSServiceAverageMemoryUtilization' },
                TargetValue: 30,
            },
        }));
        test.done();
    },
    'allows scaling on custom CloudWatch metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        const container = taskDefinition.addContainer('MainContainer', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        container.addPortMappings({ containerPort: 8000 });
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // WHEN
        const capacity = service.autoScaleTaskCount({ maxCapacity: 10, minCapacity: 1 });
        capacity.scaleToTrackCustomMetric('ScaleOnCustomMetric', {
            metric: new cloudwatch.Metric({ namespace: 'Test', metricName: 'Metric' }),
            targetValue: 5,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                CustomizedMetricSpecification: {
                    MetricName: 'Metric',
                    Namespace: 'Test',
                    Statistic: 'Average',
                },
                TargetValue: 5,
            },
        }));
        test.done();
    },
    'When enabling service discovery': {
        'throws if namespace has not been added to cluster'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // THEN
            test.throws(() => {
                new ecs.FargateService(stack, 'Service', {
                    cluster,
                    taskDefinition,
                    cloudMapOptions: {
                        name: 'myApp',
                    },
                });
            }, /Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster./);
            test.done();
        },
        'creates cloud map service for Private DNS namespace'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'A',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with SRV records with proper defaults'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    dnsRecordType: cloudmap.DnsRecordType.SRV,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 60,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
        'creates AWS Cloud Map service for Private DNS namespace with SRV records with overriden defaults'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
            cluster.addCapacity('DefaultAutoScalingGroup', { instanceType: new ec2.InstanceType('t2.micro') });
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            const container = taskDefinition.addContainer('MainContainer', {
                image: ecs.ContainerImage.fromRegistry('hello'),
                memoryLimitMiB: 512,
            });
            container.addPortMappings({ containerPort: 8000 });
            // WHEN
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
                type: cloudmap.NamespaceType.DNS_PRIVATE,
            });
            new ecs.FargateService(stack, 'Service', {
                cluster,
                taskDefinition,
                cloudMapOptions: {
                    name: 'myApp',
                    dnsRecordType: cloudmap.DnsRecordType.SRV,
                    dnsTtl: cdk.Duration.seconds(10),
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::Service', {
                DnsConfig: {
                    DnsRecords: [
                        {
                            TTL: 10,
                            Type: 'SRV',
                        },
                    ],
                    NamespaceId: {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                            'Id',
                        ],
                    },
                    RoutingPolicy: 'MULTIVALUE',
                },
                HealthCheckCustomConfig: {
                    FailureThreshold: 1,
                },
                Name: 'myApp',
                NamespaceId: {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultServiceDiscoveryNamespaceB0971B2F',
                        'Id',
                    ],
                },
            }));
            test.done();
        },
    },
    'Metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
        taskDefinition.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('hello'),
        });
        // WHEN
        const service = new ecs.FargateService(stack, 'Service', {
            cluster,
            taskDefinition,
        });
        // THEN
        test.deepEqual(stack.resolve(service.metricCpuUtilization()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
                ServiceName: { 'Fn::GetAtt': ['ServiceD69D759B', 'Name'] },
            },
            namespace: 'AWS/ECS',
            metricName: 'CPUUtilization',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'When import a Fargate Service': {
        'with serviceArn'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            // WHEN
            const service = ecs.FargateService.fromFargateServiceAttributes(stack, 'EcsService', {
                serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
                cluster,
            });
            // THEN
            test.equal(service.serviceArn, 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service');
            test.equal(service.serviceName, 'my-http-service');
            test.done();
        },
        'with serviceName'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const pseudo = new cdk.ScopedAws(stack);
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            // WHEN
            const service = ecs.FargateService.fromFargateServiceAttributes(stack, 'EcsService', {
                serviceName: 'my-http-service',
                cluster,
            });
            // THEN
            test.deepEqual(stack.resolve(service.serviceArn), stack.resolve(`arn:${pseudo.partition}:ecs:${pseudo.region}:${pseudo.accountId}:service/my-http-service`));
            test.equal(service.serviceName, 'my-http-service');
            test.done();
        },
        'throws an exception if both serviceArn and serviceName were provided for fromEc2ServiceAttributes'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            test.throws(() => {
                ecs.FargateService.fromFargateServiceAttributes(stack, 'EcsService', {
                    serviceArn: 'arn:aws:ecs:us-west-2:123456789012:service/my-http-service',
                    serviceName: 'my-http-service',
                    cluster,
                });
            }, /only specify either serviceArn or serviceName/);
            test.done();
        },
        'throws an exception if neither serviceArn nor serviceName were provided for fromEc2ServiceAttributes'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            test.throws(() => {
                ecs.FargateService.fromFargateServiceAttributes(stack, 'EcsService', {
                    cluster,
                });
            }, /only specify either serviceArn or serviceName/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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