"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const ecs = require("../../lib");
module.exports = {
    'When creating an Fargate TaskDefinition': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Family: 'FargateTaskDef',
                NetworkMode: ecs.NetworkMode.AWS_VPC,
                RequiresCompatibilities: ['FARGATE'],
                Cpu: '256',
                Memory: '512',
            }));
            test.done();
        },
        'support lazy cpu and memory values'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            new ecs.FargateTaskDefinition(stack, 'FargateTaskDef', {
                cpu: cdk.Lazy.numberValue({ produce: () => 128 }),
                memoryLimitMiB: cdk.Lazy.numberValue({ produce: () => 1024 }),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Cpu: '128',
                Memory: '1024',
            }));
            test.done();
        },
        'with all properties set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef', {
                cpu: 128,
                executionRole: new iam.Role(stack, 'ExecutionRole', {
                    path: '/',
                    assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('ecs.amazonaws.com'), new iam.ServicePrincipal('ecs-tasks.amazonaws.com')),
                }),
                family: 'myApp',
                memoryLimitMiB: 1024,
                taskRole: new iam.Role(stack, 'TaskRole', {
                    assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
                }),
            });
            taskDefinition.addVolume({
                host: {
                    sourcePath: '/tmp/cache',
                },
                name: 'scratch',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                Cpu: '128',
                ExecutionRoleArn: {
                    'Fn::GetAtt': [
                        'ExecutionRole605A040B',
                        'Arn',
                    ],
                },
                Family: 'myApp',
                Memory: '1024',
                NetworkMode: 'awsvpc',
                RequiresCompatibilities: [
                    ecs.LaunchType.FARGATE,
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        'TaskRole30FC0FBB',
                        'Arn',
                    ],
                },
                Volumes: [
                    {
                        Host: {
                            SourcePath: '/tmp/cache',
                        },
                        Name: 'scratch',
                    },
                ],
            }));
            test.done();
        },
        'throws when adding placement constraint'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.FargateTaskDefinition(stack, 'FargateTaskDef');
            // THEN
            test.throws(() => {
                taskDefinition.addPlacementConstraint(ecs.PlacementConstraint.memberOf('attribute:ecs.instance-type =~ t2.*'));
            }, /Cannot set placement constraints on tasks that run on Fargate/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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