"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
module.exports = {
    'correctly sets all appMeshProxyConfiguration'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: 'envoy',
                properties: {
                    ignoredUID: 1337,
                    ignoredGID: 1338,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81,
                    egressIgnoredPorts: [8081],
                    egressIgnoredIPs: ['169.254.170.2', '169.254.169.254'],
                },
            }),
        });
        taskDefinition.addContainer('web', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
        });
        taskDefinition.addContainer('envoy', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('envoyproxy/envoy'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ProxyConfiguration: {
                ContainerName: 'envoy',
                ProxyConfigurationProperties: [
                    {
                        Name: 'IgnoredUID',
                        Value: '1337',
                    },
                    {
                        Name: 'IgnoredGID',
                        Value: '1338',
                    },
                    {
                        Name: 'AppPorts',
                        Value: '80,81',
                    },
                    {
                        Name: 'ProxyIngressPort',
                        Value: '80',
                    },
                    {
                        Name: 'ProxyEgressPort',
                        Value: '81',
                    },
                    {
                        Name: 'EgressIgnoredPorts',
                        Value: '8081',
                    },
                    {
                        Name: 'EgressIgnoredIPs',
                        Value: '169.254.170.2,169.254.169.254',
                    },
                ],
                Type: 'APPMESH',
            },
        }));
        test.done();
    },
    'correctly sets appMeshProxyConfiguration with default properties set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: 'envoy',
                properties: {
                    ignoredUID: 1337,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81,
                },
            }),
        });
        taskDefinition.addContainer('web', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
        });
        taskDefinition.addContainer('envoy', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('envoyproxy/envoy'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ProxyConfiguration: {
                ContainerName: 'envoy',
                ProxyConfigurationProperties: [
                    {
                        Name: 'IgnoredUID',
                        Value: '1337',
                    },
                    {
                        Name: 'AppPorts',
                        Value: '80,81',
                    },
                    {
                        Name: 'ProxyIngressPort',
                        Value: '80',
                    },
                    {
                        Name: 'ProxyEgressPort',
                        Value: '81',
                    },
                ],
                Type: 'APPMESH',
            },
        }));
        test.done();
    },
    'correctly sets appMeshProxyConfiguration with empty egressIgnoredPorts and egressIgnoredIPs'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
            networkMode: ecs.NetworkMode.AWS_VPC,
            proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                containerName: 'envoy',
                properties: {
                    ignoredUID: 1337,
                    appPorts: [80, 81],
                    proxyIngressPort: 80,
                    proxyEgressPort: 81,
                    egressIgnoredIPs: [],
                    egressIgnoredPorts: [],
                },
            }),
        });
        taskDefinition.addContainer('web', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('amazon/amazon-ecs-sample'),
        });
        taskDefinition.addContainer('envoy', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromRegistry('envoyproxy/envoy'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ProxyConfiguration: {
                ContainerName: 'envoy',
                ProxyConfigurationProperties: [
                    {
                        Name: 'IgnoredUID',
                        Value: '1337',
                    },
                    {
                        Name: 'AppPorts',
                        Value: '80,81',
                    },
                    {
                        Name: 'ProxyIngressPort',
                        Value: '80',
                    },
                    {
                        Name: 'ProxyEgressPort',
                        Value: '81',
                    },
                ],
                Type: 'APPMESH',
            },
        }));
        test.done();
    },
    'throws when neither of IgnoredUID and IgnoredGID is set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // THEN
        test.throws(() => {
            new ecs.Ec2TaskDefinition(stack, 'Ec2TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
                proxyConfiguration: ecs.ProxyConfigurations.appMeshProxyConfiguration({
                    containerName: 'envoy',
                    properties: {
                        appPorts: [80, 81],
                        proxyIngressPort: 80,
                        proxyEgressPort: 81,
                    },
                }),
            });
        }, /At least one of ignoredUID or ignoredGID should be specified./);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5hcHAtbWVzaC1wcm94eS1jb25maWd1cmF0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5hcHAtbWVzaC1wcm94eS1jb25maWd1cmF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw0Q0FBMkQ7QUFDM0QscUNBQXFDO0FBRXJDLDhCQUE4QjtBQUU5QixpQkFBUztJQUNQLDhDQUE4QyxDQUFDLElBQVU7UUFDdkQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87WUFDcEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLHlCQUF5QixDQUFDO2dCQUNwRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLFVBQVUsRUFBRSxJQUFJO29CQUNoQixVQUFVLEVBQUUsSUFBSTtvQkFDaEIsUUFBUSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQztvQkFDbEIsZ0JBQWdCLEVBQUUsRUFBRTtvQkFDcEIsZUFBZSxFQUFFLEVBQUU7b0JBQ25CLGtCQUFrQixFQUFFLENBQUMsSUFBSSxDQUFDO29CQUMxQixnQkFBZ0IsRUFBRSxDQUFDLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQztpQkFDdkQ7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUU7WUFDakMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLDBCQUEwQixDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1lBQ25DLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxJQUFJLEVBQUUsWUFBWTt3QkFDbEIsS0FBSyxFQUFFLE1BQU07cUJBQ2Q7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLFlBQVk7d0JBQ2xCLEtBQUssRUFBRSxNQUFNO3FCQUNkO29CQUNEO3dCQUNFLElBQUksRUFBRSxVQUFVO3dCQUNoQixLQUFLLEVBQUUsT0FBTztxQkFDZjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsa0JBQWtCO3dCQUN4QixLQUFLLEVBQUUsSUFBSTtxQkFDWjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsaUJBQWlCO3dCQUN2QixLQUFLLEVBQUUsSUFBSTtxQkFDWjtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsb0JBQW9CO3dCQUMxQixLQUFLLEVBQUUsTUFBTTtxQkFDZDtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsa0JBQWtCO3dCQUN4QixLQUFLLEVBQUUsK0JBQStCO3FCQUN2QztpQkFDRjtnQkFDRCxJQUFJLEVBQUUsU0FBUzthQUNoQjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHNFQUFzRSxDQUFDLElBQVU7UUFDL0UsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87WUFDcEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLG1CQUFtQixDQUFDLHlCQUF5QixDQUFDO2dCQUNwRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLFVBQVUsRUFBRSxJQUFJO29CQUNoQixRQUFRLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDO29CQUNsQixnQkFBZ0IsRUFBRSxFQUFFO29CQUNwQixlQUFlLEVBQUUsRUFBRTtpQkFDcEI7YUFDRixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUU7WUFDakMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLDBCQUEwQixDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUNILGNBQWMsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1lBQ25DLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQztTQUMzRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxrQkFBa0IsRUFBRTtnQkFDbEIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxJQUFJLEVBQUUsWUFBWTt3QkFDbEIsS0FBSyxFQUFFLE1BQU07cUJBQ2Q7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLFVBQVU7d0JBQ2hCLEtBQUssRUFBRSxPQUFPO3FCQUNmO29CQUNEO3dCQUNFLElBQUksRUFBRSxrQkFBa0I7d0JBQ3hCLEtBQUssRUFBRSxJQUFJO3FCQUNaO29CQUNEO3dCQUNFLElBQUksRUFBRSxpQkFBaUI7d0JBQ3ZCLEtBQUssRUFBRSxJQUFJO3FCQUNaO2lCQUNGO2dCQUNELElBQUksRUFBRSxTQUFTO2FBQ2hCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkZBQTZGLENBQUMsSUFBVTtRQUN0RyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDcEUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsT0FBTztZQUNwQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMseUJBQXlCLENBQUM7Z0JBQ3BFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixVQUFVLEVBQUU7b0JBQ1YsVUFBVSxFQUFFLElBQUk7b0JBQ2hCLFFBQVEsRUFBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUM7b0JBQ2xCLGdCQUFnQixFQUFFLEVBQUU7b0JBQ3BCLGVBQWUsRUFBRSxFQUFFO29CQUNuQixnQkFBZ0IsRUFBRSxFQUFFO29CQUNwQixrQkFBa0IsRUFBRSxFQUFFO2lCQUN2QjthQUNGLENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxjQUFjLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRTtZQUNqQyxjQUFjLEVBQUUsSUFBSTtZQUNwQixLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsMEJBQTBCLENBQUM7U0FDbkUsQ0FBQyxDQUFDO1FBQ0gsY0FBYyxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUU7WUFDbkMsY0FBYyxFQUFFLElBQUk7WUFDcEIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFDO1NBQzNELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO1lBQzVELGtCQUFrQixFQUFFO2dCQUNsQixhQUFhLEVBQUUsT0FBTztnQkFDdEIsNEJBQTRCLEVBQUU7b0JBQzVCO3dCQUNFLElBQUksRUFBRSxZQUFZO3dCQUNsQixLQUFLLEVBQUUsTUFBTTtxQkFDZDtvQkFDRDt3QkFDRSxJQUFJLEVBQUUsVUFBVTt3QkFDaEIsS0FBSyxFQUFFLE9BQU87cUJBQ2Y7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLGtCQUFrQjt3QkFDeEIsS0FBSyxFQUFFLElBQUk7cUJBQ1o7b0JBQ0Q7d0JBQ0UsSUFBSSxFQUFFLGlCQUFpQjt3QkFDdkIsS0FBSyxFQUFFLElBQUk7cUJBQ1o7aUJBQ0Y7Z0JBQ0QsSUFBSSxFQUFFLFNBQVM7YUFDaEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5REFBeUQsQ0FBQyxJQUFVO1FBQ2xFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUU5QixPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDZixJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFO2dCQUM3QyxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2dCQUNwQyxrQkFBa0IsRUFBRSxHQUFHLENBQUMsbUJBQW1CLENBQUMseUJBQXlCLENBQUM7b0JBQ3BFLGFBQWEsRUFBRSxPQUFPO29CQUN0QixVQUFVLEVBQUU7d0JBQ1YsUUFBUSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQzt3QkFDbEIsZ0JBQWdCLEVBQUUsRUFBRTt3QkFDcEIsZUFBZSxFQUFFLEVBQUU7cUJBQ3BCO2lCQUNGLENBQUM7YUFDSCxDQUFDLENBQUM7UUFDTCxDQUFDLEVBQUUsK0RBQStELENBQUMsQ0FBQztRQUVwRSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGV4cGVjdCwgaGF2ZVJlc291cmNlTGlrZSB9IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ2NvcnJlY3RseSBzZXRzIGFsbCBhcHBNZXNoUHJveHlDb25maWd1cmF0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdFYzJUYXNrRGVmJywge1xuICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5BV1NfVlBDLFxuICAgICAgcHJveHlDb25maWd1cmF0aW9uOiBlY3MuUHJveHlDb25maWd1cmF0aW9ucy5hcHBNZXNoUHJveHlDb25maWd1cmF0aW9uKHtcbiAgICAgICAgY29udGFpbmVyTmFtZTogJ2Vudm95JyxcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGlnbm9yZWRVSUQ6IDEzMzcsXG4gICAgICAgICAgaWdub3JlZEdJRDogMTMzOCxcbiAgICAgICAgICBhcHBQb3J0czogWzgwLCA4MV0sXG4gICAgICAgICAgcHJveHlJbmdyZXNzUG9ydDogODAsXG4gICAgICAgICAgcHJveHlFZ3Jlc3NQb3J0OiA4MSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkUG9ydHM6IFs4MDgxXSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkSVBzOiBbJzE2OS4yNTQuMTcwLjInLCAnMTY5LjI1NC4xNjkuMjU0J10sXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ3dlYicsIHtcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ2FtYXpvbi9hbWF6b24tZWNzLXNhbXBsZScpLFxuICAgIH0pO1xuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignZW52b3knLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdlbnZveXByb3h5L2Vudm95JyksXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICBQcm94eUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgQ29udGFpbmVyTmFtZTogJ2Vudm95JyxcbiAgICAgICAgUHJveHlDb25maWd1cmF0aW9uUHJvcGVydGllczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6ICdJZ25vcmVkVUlEJyxcbiAgICAgICAgICAgIFZhbHVlOiAnMTMzNycsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnSWdub3JlZEdJRCcsXG4gICAgICAgICAgICBWYWx1ZTogJzEzMzgnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ0FwcFBvcnRzJyxcbiAgICAgICAgICAgIFZhbHVlOiAnODAsODEnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ1Byb3h5SW5ncmVzc1BvcnQnLFxuICAgICAgICAgICAgVmFsdWU6ICc4MCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnUHJveHlFZ3Jlc3NQb3J0JyxcbiAgICAgICAgICAgIFZhbHVlOiAnODEnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ0VncmVzc0lnbm9yZWRQb3J0cycsXG4gICAgICAgICAgICBWYWx1ZTogJzgwODEnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ0VncmVzc0lnbm9yZWRJUHMnLFxuICAgICAgICAgICAgVmFsdWU6ICcxNjkuMjU0LjE3MC4yLDE2OS4yNTQuMTY5LjI1NCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVHlwZTogJ0FQUE1FU0gnLFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvcnJlY3RseSBzZXRzIGFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24gd2l0aCBkZWZhdWx0IHByb3BlcnRpZXMgc2V0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdFYzJUYXNrRGVmJywge1xuICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5BV1NfVlBDLFxuICAgICAgcHJveHlDb25maWd1cmF0aW9uOiBlY3MuUHJveHlDb25maWd1cmF0aW9ucy5hcHBNZXNoUHJveHlDb25maWd1cmF0aW9uKHtcbiAgICAgICAgY29udGFpbmVyTmFtZTogJ2Vudm95JyxcbiAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgIGlnbm9yZWRVSUQ6IDEzMzcsXG4gICAgICAgICAgYXBwUG9ydHM6IFs4MCwgODFdLFxuICAgICAgICAgIHByb3h5SW5ncmVzc1BvcnQ6IDgwLFxuICAgICAgICAgIHByb3h5RWdyZXNzUG9ydDogODEsXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ3dlYicsIHtcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ2FtYXpvbi9hbWF6b24tZWNzLXNhbXBsZScpLFxuICAgIH0pO1xuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignZW52b3knLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdlbnZveXByb3h5L2Vudm95JyksXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICBQcm94eUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgQ29udGFpbmVyTmFtZTogJ2Vudm95JyxcbiAgICAgICAgUHJveHlDb25maWd1cmF0aW9uUHJvcGVydGllczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6ICdJZ25vcmVkVUlEJyxcbiAgICAgICAgICAgIFZhbHVlOiAnMTMzNycsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnQXBwUG9ydHMnLFxuICAgICAgICAgICAgVmFsdWU6ICc4MCw4MScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnUHJveHlJbmdyZXNzUG9ydCcsXG4gICAgICAgICAgICBWYWx1ZTogJzgwJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIE5hbWU6ICdQcm94eUVncmVzc1BvcnQnLFxuICAgICAgICAgICAgVmFsdWU6ICc4MScsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVHlwZTogJ0FQUE1FU0gnLFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvcnJlY3RseSBzZXRzIGFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24gd2l0aCBlbXB0eSBlZ3Jlc3NJZ25vcmVkUG9ydHMgYW5kIGVncmVzc0lnbm9yZWRJUHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ0VjMlRhc2tEZWYnLCB7XG4gICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkFXU19WUEMsXG4gICAgICBwcm94eUNvbmZpZ3VyYXRpb246IGVjcy5Qcm94eUNvbmZpZ3VyYXRpb25zLmFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24oe1xuICAgICAgICBjb250YWluZXJOYW1lOiAnZW52b3knLFxuICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgaWdub3JlZFVJRDogMTMzNyxcbiAgICAgICAgICBhcHBQb3J0czogWzgwLCA4MV0sXG4gICAgICAgICAgcHJveHlJbmdyZXNzUG9ydDogODAsXG4gICAgICAgICAgcHJveHlFZ3Jlc3NQb3J0OiA4MSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkSVBzOiBbXSxcbiAgICAgICAgICBlZ3Jlc3NJZ25vcmVkUG9ydHM6IFtdLFxuICAgICAgICB9LFxuICAgICAgfSksXG4gICAgfSk7XG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCd3ZWInLCB7XG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdhbWF6b24vYW1hem9uLWVjcy1zYW1wbGUnKSxcbiAgICB9KTtcbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2Vudm95Jywge1xuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnZW52b3lwcm94eS9lbnZveScpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgUHJveHlDb25maWd1cmF0aW9uOiB7XG4gICAgICAgIENvbnRhaW5lck5hbWU6ICdlbnZveScsXG4gICAgICAgIFByb3h5Q29uZmlndXJhdGlvblByb3BlcnRpZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnSWdub3JlZFVJRCcsXG4gICAgICAgICAgICBWYWx1ZTogJzEzMzcnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ0FwcFBvcnRzJyxcbiAgICAgICAgICAgIFZhbHVlOiAnODAsODEnLFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgTmFtZTogJ1Byb3h5SW5ncmVzc1BvcnQnLFxuICAgICAgICAgICAgVmFsdWU6ICc4MCcsXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiAnUHJveHlFZ3Jlc3NQb3J0JyxcbiAgICAgICAgICAgIFZhbHVlOiAnODEnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIFR5cGU6ICdBUFBNRVNIJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0aHJvd3Mgd2hlbiBuZWl0aGVyIG9mIElnbm9yZWRVSUQgYW5kIElnbm9yZWRHSUQgaXMgc2V0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnRWMyVGFza0RlZicsIHtcbiAgICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5BV1NfVlBDLFxuICAgICAgICBwcm94eUNvbmZpZ3VyYXRpb246IGVjcy5Qcm94eUNvbmZpZ3VyYXRpb25zLmFwcE1lc2hQcm94eUNvbmZpZ3VyYXRpb24oe1xuICAgICAgICAgIGNvbnRhaW5lck5hbWU6ICdlbnZveScsXG4gICAgICAgICAgcHJvcGVydGllczoge1xuICAgICAgICAgICAgYXBwUG9ydHM6IFs4MCwgODFdLFxuICAgICAgICAgICAgcHJveHlJbmdyZXNzUG9ydDogODAsXG4gICAgICAgICAgICBwcm94eUVncmVzc1BvcnQ6IDgxLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgfSk7XG4gICAgfSwgL0F0IGxlYXN0IG9uZSBvZiBpZ25vcmVkVUlEIG9yIGlnbm9yZWRHSUQgc2hvdWxkIGJlIHNwZWNpZmllZC4vKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTtcbiJdfQ==