"use strict";
const assert_1 = require("@aws-cdk/assert");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
let stack;
let td;
const image = ecs.ContainerImage.fromRegistry('test-image');
module.exports = {
    'setUp'(cb) {
        stack = new cdk.Stack();
        td = new ecs.FargateTaskDefinition(stack, 'TaskDefinition');
        cb();
    },
    'create an aws log driver'(test) {
        // WHEN
        td.addContainer('Container', {
            image,
            logging: new ecs.AwsLogDriver({
                datetimeFormat: 'format',
                logRetention: logs.RetentionDays.ONE_MONTH,
                multilinePattern: 'pattern',
                streamPrefix: 'hello',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: logs.RetentionDays.ONE_MONTH,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'TaskDefinitionContainerLogGroup4D0A87C1' },
                            'awslogs-stream-prefix': 'hello',
                            'awslogs-region': { Ref: 'AWS::Region' },
                            'awslogs-datetime-format': 'format',
                            'awslogs-multiline-pattern': 'pattern',
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'create an aws log driver using awsLogs'(test) {
        // WHEN
        td.addContainer('Container', {
            image,
            logging: ecs.AwsLogDriver.awsLogs({
                datetimeFormat: 'format',
                logRetention: logs.RetentionDays.ONE_MONTH,
                multilinePattern: 'pattern',
                streamPrefix: 'hello',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: logs.RetentionDays.ONE_MONTH,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'TaskDefinitionContainerLogGroup4D0A87C1' },
                            'awslogs-stream-prefix': 'hello',
                            'awslogs-region': { Ref: 'AWS::Region' },
                            'awslogs-datetime-format': 'format',
                            'awslogs-multiline-pattern': 'pattern',
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'with a defined log group'(test) {
        // GIVEN
        const logGroup = new logs.LogGroup(stack, 'LogGroup');
        // WHEN
        td.addContainer('Container', {
            image,
            logging: new ecs.AwsLogDriver({
                logGroup,
                streamPrefix: 'hello',
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Logs::LogGroup', {
            RetentionInDays: logs.RetentionDays.TWO_YEARS,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'LogGroupF5B46931' },
                            'awslogs-stream-prefix': 'hello',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        test.done();
    },
    'without a defined log group'(test) {
        // GIVEN
        td.addContainer('Container', {
            image,
            logging: new ecs.AwsLogDriver({
                streamPrefix: 'hello',
            }),
        });
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Logs::LogGroup', {}));
        test.done();
    },
    'throws when specifying log retention and log group'(test) {
        // GIVEN
        const logGroup = new logs.LogGroup(stack, 'LogGroup');
        // THEN
        test.throws(() => new ecs.AwsLogDriver({
            logGroup,
            logRetention: logs.RetentionDays.FIVE_DAYS,
            streamPrefix: 'hello',
        }), /`logGroup`.*`logRetentionDays`/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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