"use strict";
const assert_1 = require("@aws-cdk/assert");
const ecr_assets = require("@aws-cdk/aws-ecr-assets");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const ssm = require("@aws-cdk/aws-ssm");
const cdk = require("@aws-cdk/core");
const path = require("path");
const ecs = require("../lib");
module.exports = {
    'When creating a Task Definition': {
        'add a container using default props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            new ecs.ContainerDefinition(stack, 'Container', {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                taskDefinition,
                memoryLimitMiB: 2048,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                        Image: '/aws/aws-example-app',
                        Memory: 2048,
                        Name: 'Container',
                    },
                ],
            }));
            test.done();
        },
        'add a container using all props'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const secret = new secretsmanager.Secret(stack, 'Secret');
            new ecs.ContainerDefinition(stack, 'Container', {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                taskDefinition,
                memoryLimitMiB: 1024,
                memoryReservationMiB: 512,
                command: ['CMD-SHELL'],
                cpu: 128,
                disableNetworking: true,
                dnsSearchDomains: ['example.com'],
                dnsServers: ['host.com'],
                dockerLabels: {
                    key: 'fooLabel',
                    value: 'barLabel',
                },
                dockerSecurityOptions: ['ECS_SELINUX_CAPABLE=true'],
                entryPoint: ['top', '-b'],
                environment: {
                    key: 'foo',
                    value: 'bar',
                },
                essential: true,
                extraHosts: {
                    name: 'dev-db.hostname.pvt',
                },
                gpuCount: 256,
                hostname: 'host.example.com',
                privileged: true,
                readonlyRootFilesystem: true,
                startTimeout: cdk.Duration.millis(2000),
                stopTimeout: cdk.Duration.millis(5000),
                user: 'rootUser',
                workingDirectory: 'a/b/c',
                healthCheck: {
                    command: ['curl localhost:8000'],
                },
                linuxParameters: new ecs.LinuxParameters(stack, 'LinuxParameters'),
                logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' }),
                secrets: {
                    SECRET: ecs.Secret.fromSecretsManager(secret),
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Command: [
                            'CMD-SHELL',
                        ],
                        Cpu: 128,
                        DisableNetworking: true,
                        DnsSearchDomains: [
                            'example.com',
                        ],
                        DnsServers: [
                            'host.com',
                        ],
                        DockerLabels: {
                            key: 'fooLabel',
                            value: 'barLabel',
                        },
                        DockerSecurityOptions: [
                            'ECS_SELINUX_CAPABLE=true',
                        ],
                        EntryPoint: [
                            'top',
                            '-b',
                        ],
                        Environment: [
                            {
                                Name: 'key',
                                Value: 'foo',
                            },
                            {
                                Name: 'value',
                                Value: 'bar',
                            },
                        ],
                        Essential: true,
                        ExtraHosts: [
                            {
                                Hostname: 'name',
                                IpAddress: 'dev-db.hostname.pvt',
                            },
                        ],
                        HealthCheck: {
                            Command: [
                                'CMD-SHELL',
                                'curl localhost:8000',
                            ],
                            Interval: 30,
                            Retries: 3,
                            Timeout: 5,
                        },
                        Hostname: 'host.example.com',
                        Image: '/aws/aws-example-app',
                        LinuxParameters: {
                            Capabilities: {},
                        },
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {
                                    Ref: 'ContainerLogGroupE6FD74A4',
                                },
                                'awslogs-stream-prefix': 'prefix',
                                'awslogs-region': {
                                    Ref: 'AWS::Region',
                                },
                            },
                        },
                        Memory: 1024,
                        MemoryReservation: 512,
                        Name: 'Container',
                        Privileged: true,
                        ReadonlyRootFilesystem: true,
                        ResourceRequirements: [
                            {
                                Type: 'GPU',
                                Value: '256',
                            },
                        ],
                        Secrets: [
                            {
                                Name: 'SECRET',
                                ValueFrom: {
                                    Ref: 'SecretA720EF05',
                                },
                            },
                        ],
                        StartTimeout: 2,
                        StopTimeout: 5,
                        User: 'rootUser',
                        WorkingDirectory: 'a/b/c',
                    },
                ],
            }));
            test.done();
        },
        'throws when MemoryLimit is less than MemoryReservationLimit'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // THEN
            test.throws(() => {
                new ecs.ContainerDefinition(stack, 'Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    taskDefinition,
                    memoryLimitMiB: 512,
                    memoryReservationMiB: 1024,
                });
            }, /MemoryLimitMiB should not be less than MemoryReservationMiB./);
            test.done();
        },
        'With network mode AwsVpc': {
            'throws when Host port is different from container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // THEN
                test.throws(() => {
                    container.addPortMappings({
                        containerPort: 8080,
                        hostPort: 8081,
                    });
                });
                test.done();
            },
            'Host port is the same as container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
            'Host port can be empty '(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
        },
        'With network mode Host ': {
            'throws when Host port is different from container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // THEN
                test.throws(() => {
                    container.addPortMappings({
                        containerPort: 8080,
                        hostPort: 8081,
                    });
                });
                test.done();
            },
            'when host port is the same as container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
            'Host port can be empty '(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raised
                test.done();
            },
            'errors when adding links'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                const logger = taskDefinition.addContainer('LoggingContainer', {
                    image: ecs.ContainerImage.fromRegistry('myLogger'),
                    memoryLimitMiB: 1024,
                });
                // THEN
                test.throws(() => {
                    container.addLink(logger);
                });
                test.done();
            },
        },
        'With network mode Bridge': {
            'when Host port is empty '(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                });
                // THEN no exception raises
                test.done();
            },
            'when Host port is not empty '(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8084,
                });
                // THEN no exception raises
                test.done();
            },
            'allows adding links'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                const logger = taskDefinition.addContainer('LoggingContainer', {
                    image: ecs.ContainerImage.fromRegistry('myLogger'),
                    memoryLimitMiB: 1024,
                });
                // THEN
                container.addLink(logger);
                test.done();
            },
        },
        'With network mode NAT': {
            'produces undefined CF networkMode property'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                // WHEN
                new ecs.TaskDefinition(stack, 'TD', {
                    compatibility: ecs.Compatibility.EC2,
                    networkMode: ecs.NetworkMode.NAT,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', (props, inspection) => {
                    if (props.NetworkMode === undefined) {
                        return true;
                    }
                    inspection.failureReason = 'CF template should not have NetworkMode defined for a task definition that relies on NAT network mode.';
                    return false;
                }));
                test.done();
            },
        },
    },
    'Container Port': {
        'should return the first container port in PortMappings'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer('Container', {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                memoryLimitMiB: 2048,
            });
            // WHEN
            container.addPortMappings({
                containerPort: 8080,
            });
            container.addPortMappings({
                containerPort: 8081,
            });
            const actual = container.containerPort;
            // THEN
            const expected = 8080;
            test.equal(actual, expected, 'containerPort should return the first container port in PortMappings');
            test.done();
        },
        'throws when calling containerPort with no PortMappings'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                networkMode: ecs.NetworkMode.AWS_VPC,
            });
            const container = taskDefinition.addContainer('MyContainer', {
                image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                memoryLimitMiB: 2048,
            });
            // THEN
            test.throws(() => {
                const actual = container.containerPort;
                const expected = 8080;
                test.equal(actual, expected);
            }, /Container MyContainer hasn't defined any ports. Call addPortMappings()./);
            test.done();
        },
    },
    'Ingress Port': {
        'With network mode AwsVpc': {
            'Ingress port should be the same as container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8080;
                test.equal(actual, expected, 'Ingress port should be the same as container port');
                test.done();
            },
            'throws when calling ingressPort with no PortMappings'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.AWS_VPC,
                });
                const container = taskDefinition.addContainer('MyContainer', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // THEN
                test.throws(() => {
                    const actual = container.ingressPort;
                    const expected = 8080;
                    test.equal(actual, expected);
                }, /Container MyContainer hasn't defined any ports. Call addPortMappings()./);
                test.done();
            },
        },
        'With network mode Host ': {
            'Ingress port should be the same as container port'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.HOST,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8080;
                test.equal(actual, expected);
                test.done();
            },
        },
        'With network mode Bridge': {
            'Ingress port should be the same as host port if supplied'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8080,
                    hostPort: 8081,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 8081;
                test.equal(actual, expected);
                test.done();
            },
            'Ingress port should be 0 if not supplied'(test) {
                // GIVEN
                const stack = new cdk.Stack();
                const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef', {
                    networkMode: ecs.NetworkMode.BRIDGE,
                });
                const container = taskDefinition.addContainer('Container', {
                    image: ecs.ContainerImage.fromRegistry('/aws/aws-example-app'),
                    memoryLimitMiB: 2048,
                });
                // WHEN
                container.addPortMappings({
                    containerPort: 8081,
                });
                const actual = container.ingressPort;
                // THEN
                const expected = 0;
                test.equal(actual, expected);
                test.done();
            },
        },
    },
    'can add environment variables to the container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            environment: {
                TEST_ENVIRONMENT_VARIABLE: 'test environment variable value',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Environment: [{
                            Name: 'TEST_ENVIRONMENT_VARIABLE',
                            Value: 'test environment variable value',
                        }],
                },
            ],
        }));
        test.done();
    },
    'Given GPU count parameter': {
        'will add resource requirements to container definition'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                gpuCount: 4,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        ResourceRequirements: [
                            {
                                Type: 'GPU',
                                Value: '4',
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
    'can add secret environment variables to the container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        const parameter = ssm.StringParameter.fromSecureStringParameterAttributes(stack, 'Parameter', {
            parameterName: '/name',
            version: 1,
        });
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            secrets: {
                SECRET: ecs.Secret.fromSecretsManager(secret),
                PARAMETER: ecs.Secret.fromSsmParameter(parameter),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Secrets: [
                        {
                            Name: 'SECRET',
                            ValueFrom: {
                                Ref: 'SecretA720EF05',
                            },
                        },
                        {
                            Name: 'PARAMETER',
                            ValueFrom: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':ssm:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':parameter/name',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'SecretA720EF05',
                        },
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':ssm:',
                                    {
                                        Ref: 'AWS::Region',
                                    },
                                    ':',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':parameter/name',
                                ],
                            ],
                        },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'use a specific secret JSON key as environment variable'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            secrets: {
                SECRET_KEY: ecs.Secret.fromSecretsManager(secret, 'specificKey'),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Secrets: [
                        {
                            Name: 'SECRET_KEY',
                            ValueFrom: {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            Ref: 'SecretA720EF05',
                                        },
                                        ':specificKey::',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            ],
        }));
        test.done();
    },
    'throws when using a specific secret JSON field as environment variable for a Fargate task'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.FargateTaskDefinition(stack, 'TaskDef');
        const secret = new secretsmanager.Secret(stack, 'Secret');
        // THEN
        test.throws(() => taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            secrets: {
                SECRET_KEY: ecs.Secret.fromSecretsManager(secret, 'specificKey'),
            },
        }), /Cannot specify secret JSON field for a task using the FARGATE launch type/);
        test.done();
    },
    'can add AWS logging to container definition'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            logging: new ecs.AwsLogDriver({ streamPrefix: 'prefix' }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': { Ref: 'TaskDefcontLogGroup4E10DCBF' },
                            'awslogs-stream-prefix': 'prefix',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: ['logs:CreateLogStream', 'logs:PutLogEvents'],
                        Effect: 'Allow',
                        Resource: { 'Fn::GetAtt': ['TaskDefcontLogGroup4E10DCBF', 'Arn'] },
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'can set Health Check with defaults'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = 'curl localhost:8000';
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: [hcCommand],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ['CMD-SHELL', hcCommand],
                        Interval: 30,
                        Retries: 3,
                        Timeout: 5,
                    },
                },
            ],
        }));
        test.done();
    },
    'throws when setting Health Check with no commands'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: [],
            },
        });
        // THEN
        test.throws(() => {
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        HealthCheck: {
                            Command: [],
                            Interval: 30,
                            Retries: 3,
                            Timeout: 5,
                        },
                    },
                ],
            }));
        }, /At least one argument must be supplied for health check command./);
        test.done();
    },
    'can specify Health Check values in shell form'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = 'curl localhost:8000';
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: [hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ['CMD-SHELL', hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10,
                    },
                },
            ],
        }));
        test.done();
    },
    'can specify Health Check values in array form starting with CMD-SHELL'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = 'curl localhost:8000';
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: ['CMD-SHELL', hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ['CMD-SHELL', hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10,
                    },
                },
            ],
        }));
        test.done();
    },
    'can specify Health Check values in array form starting with CMD'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const hcCommand = 'curl localhost:8000';
        // WHEN
        taskDefinition.addContainer('cont', {
            image: ecs.ContainerImage.fromRegistry('test'),
            memoryLimitMiB: 1024,
            healthCheck: {
                command: ['CMD', hcCommand],
                interval: cdk.Duration.seconds(20),
                retries: 5,
                startPeriod: cdk.Duration.seconds(10),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    HealthCheck: {
                        Command: ['CMD', hcCommand],
                        Interval: 20,
                        Retries: 5,
                        Timeout: 5,
                        StartPeriod: 10,
                    },
                },
            ],
        }));
        test.done();
    },
    'can specify private registry credentials'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const mySecretArn = 'arn:aws:secretsmanager:region:1234567890:secret:MyRepoSecret-6f8hj3';
        const repoCreds = secretsmanager.Secret.fromSecretArn(stack, 'MyRepoSecret', mySecretArn);
        // WHEN
        taskDefinition.addContainer('Container', {
            image: ecs.ContainerImage.fromRegistry('user-x/my-app', {
                credentials: repoCreds,
            }),
            memoryLimitMiB: 2048,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Image: 'user-x/my-app',
                    RepositoryCredentials: {
                        CredentialsParameter: mySecretArn,
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: mySecretArn,
                    },
                ],
            },
        }));
        test.done();
    },
    '_linkContainer works properly': {
        'when the props passed in is an essential container'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            const container = taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                essential: true,
            });
            // THEN
            test.equal(taskDefinition.defaultContainer, container);
            test.done();
        },
        'when the props passed in is not an essential container'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                essential: false,
            });
            // THEN
            test.equal(taskDefinition.defaultContainer, undefined);
            test.done();
        },
    },
    'Can specify linux parameters': {
        'with only required properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters');
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {},
                        },
                    },
                ],
            }));
            test.done();
        },
        'before calling addContainer'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            linuxParameters.addCapabilities(ecs.Capability.ALL);
            linuxParameters.dropCapabilities(ecs.Capability.KILL);
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {
                                Add: ['ALL'],
                                Drop: ['KILL'],
                            },
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    },
                ],
            }));
            test.done();
        },
        'after calling addContainer'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            linuxParameters.addCapabilities(ecs.Capability.ALL);
            // WHEN
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // Mutate linuxParameter after added to a container
            linuxParameters.dropCapabilities(ecs.Capability.SETUID);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Capabilities: {
                                Add: ['ALL'],
                                Drop: ['SETUID'],
                            },
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    },
                ],
            }));
            test.done();
        },
        'with one or more host devices'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            // WHEN
            linuxParameters.addDevices({
                hostPath: 'a/b/c',
            });
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Devices: [
                                {
                                    HostPath: 'a/b/c',
                                },
                            ],
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    },
                ],
            }));
            test.done();
        },
        'with the tmpfs mount for a container'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
            const linuxParameters = new ecs.LinuxParameters(stack, 'LinuxParameters', {
                initProcessEnabled: true,
                sharedMemorySize: 1024,
            });
            // WHEN
            linuxParameters.addTmpfs({
                containerPath: 'a/b/c',
                size: 1024,
            });
            taskDefinition.addContainer('cont', {
                image: ecs.ContainerImage.fromRegistry('test'),
                memoryLimitMiB: 1024,
                linuxParameters,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Image: 'test',
                        LinuxParameters: {
                            Tmpfs: [
                                {
                                    ContainerPath: 'a/b/c',
                                    Size: 1024,
                                },
                            ],
                            InitProcessEnabled: true,
                            SharedMemorySize: 1024,
                        },
                    },
                ],
            }));
            test.done();
        },
    },
    'can use a DockerImageAsset directly for a container image'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        const asset = new ecr_assets.DockerImageAsset(stack, 'MyDockerImage', {
            directory: path.join(__dirname, 'demo-image'),
        });
        // WHEN
        taskDefinition.addContainer('default', {
            image: ecs.ContainerImage.fromDockerImageAsset(asset),
            memoryLimitMiB: 1024,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    Essential: true,
                    Image: {
                        'Fn::Join': [
                            '',
                            [
                                { Ref: 'AWS::AccountId' },
                                '.dkr.ecr.',
                                { Ref: 'AWS::Region' },
                                '.',
                                { Ref: 'AWS::URLSuffix' },
                                '/aws-cdk/assets:baa2d6eb2a17c75424df631c8c70ff39f2d5f3bee8b9e1a109ee24ca17300540',
                            ],
                        ],
                    },
                    Memory: 1024,
                    Name: 'default',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'ecr:BatchCheckLayerAvailability',
                            'ecr:GetDownloadUrlForLayer',
                            'ecr:BatchGetImage',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                ['arn:', { Ref: 'AWS::Partition' }, ':ecr:', { Ref: 'AWS::Region' }, ':', { Ref: 'AWS::AccountId' }, ':repository/aws-cdk/assets'],
                            ],
                        },
                    },
                    {
                        Action: 'ecr:GetAuthorizationToken',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
        }));
        test.done();
    },
    'docker image asset options can be used when using container image'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'MyStack');
        const taskDefinition = new ecs.Ec2TaskDefinition(stack, 'TaskDef');
        // WHEN
        taskDefinition.addContainer('default', {
            memoryLimitMiB: 1024,
            image: ecs.ContainerImage.fromAsset(path.join(__dirname, 'demo-image'), {
                file: 'index.py',
                target: 'build-target',
            }),
        });
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackArtifact(stack.artifactId).assets[0], {
            repositoryName: 'aws-cdk/assets',
            imageTag: 'f9014d1df7c8f5a5e7abaf18eb5bc895e82f8b06eeed6f75a40cf1bc2a78955a',
            id: 'f9014d1df7c8f5a5e7abaf18eb5bc895e82f8b06eeed6f75a40cf1bc2a78955a',
            packaging: 'container-image',
            path: 'asset.f9014d1df7c8f5a5e7abaf18eb5bc895e82f8b06eeed6f75a40cf1bc2a78955a',
            sourceHash: 'f9014d1df7c8f5a5e7abaf18eb5bc895e82f8b06eeed6f75a40cf1bc2a78955a',
            target: 'build-target',
            file: 'index.py',
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb250YWluZXItZGVmaW5pdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuY29udGFpbmVyLWRlZmluaXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUE0RjtBQUM1RixzREFBc0Q7QUFDdEQsOERBQThEO0FBQzlELHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFFckMsNkJBQTZCO0FBQzdCLDhCQUE4QjtBQUU5QixpQkFBUztJQUNQLGlDQUFpQyxFQUFFO1FBQ2pDLHFDQUFxQyxDQUFDLElBQVU7WUFDOUMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUVuRSxJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUM5QyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7Z0JBQzlELGNBQWM7Z0JBQ2QsY0FBYyxFQUFFLElBQUk7YUFDckIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQzVELG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxTQUFTLEVBQUUsSUFBSTt3QkFDZixLQUFLLEVBQUUsc0JBQXNCO3dCQUM3QixNQUFNLEVBQUUsSUFBSTt3QkFDWixJQUFJLEVBQUUsV0FBVztxQkFDbEI7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxpQ0FBaUMsQ0FBQyxJQUFVO1lBQzFDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFDbkUsTUFBTSxNQUFNLEdBQUcsSUFBSSxjQUFjLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztZQUMxRCxJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO2dCQUM5QyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7Z0JBQzlELGNBQWM7Z0JBQ2QsY0FBYyxFQUFFLElBQUk7Z0JBQ3BCLG9CQUFvQixFQUFFLEdBQUc7Z0JBQ3pCLE9BQU8sRUFBRSxDQUFDLFdBQVcsQ0FBQztnQkFDdEIsR0FBRyxFQUFFLEdBQUc7Z0JBQ1IsaUJBQWlCLEVBQUUsSUFBSTtnQkFDdkIsZ0JBQWdCLEVBQUUsQ0FBQyxhQUFhLENBQUM7Z0JBQ2pDLFVBQVUsRUFBRSxDQUFDLFVBQVUsQ0FBQztnQkFDeEIsWUFBWSxFQUFFO29CQUNaLEdBQUcsRUFBRSxVQUFVO29CQUNmLEtBQUssRUFBRSxVQUFVO2lCQUNsQjtnQkFDRCxxQkFBcUIsRUFBRSxDQUFDLDBCQUEwQixDQUFDO2dCQUNuRCxVQUFVLEVBQUUsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDO2dCQUN6QixXQUFXLEVBQUU7b0JBQ1gsR0FBRyxFQUFFLEtBQUs7b0JBQ1YsS0FBSyxFQUFFLEtBQUs7aUJBQ2I7Z0JBQ0QsU0FBUyxFQUFFLElBQUk7Z0JBQ2YsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxxQkFBcUI7aUJBQzVCO2dCQUNELFFBQVEsRUFBRSxHQUFHO2dCQUNiLFFBQVEsRUFBRSxrQkFBa0I7Z0JBQzVCLFVBQVUsRUFBRSxJQUFJO2dCQUNoQixzQkFBc0IsRUFBRSxJQUFJO2dCQUM1QixZQUFZLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO2dCQUN2QyxXQUFXLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO2dCQUN0QyxJQUFJLEVBQUUsVUFBVTtnQkFDaEIsZ0JBQWdCLEVBQUUsT0FBTztnQkFDekIsV0FBVyxFQUFFO29CQUNYLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2lCQUNqQztnQkFDRCxlQUFlLEVBQUUsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQztnQkFDbEUsT0FBTyxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUUsQ0FBQztnQkFDekQsT0FBTyxFQUFFO29CQUNQLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQztpQkFDOUM7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLE9BQU8sRUFBRTs0QkFDUCxXQUFXO3lCQUNaO3dCQUNELEdBQUcsRUFBRSxHQUFHO3dCQUNSLGlCQUFpQixFQUFFLElBQUk7d0JBQ3ZCLGdCQUFnQixFQUFFOzRCQUNoQixhQUFhO3lCQUNkO3dCQUNELFVBQVUsRUFBRTs0QkFDVixVQUFVO3lCQUNYO3dCQUNELFlBQVksRUFBRTs0QkFDWixHQUFHLEVBQUUsVUFBVTs0QkFDZixLQUFLLEVBQUUsVUFBVTt5QkFDbEI7d0JBQ0QscUJBQXFCLEVBQUU7NEJBQ3JCLDBCQUEwQjt5QkFDM0I7d0JBQ0QsVUFBVSxFQUFFOzRCQUNWLEtBQUs7NEJBQ0wsSUFBSTt5QkFDTDt3QkFDRCxXQUFXLEVBQUU7NEJBQ1g7Z0NBQ0UsSUFBSSxFQUFFLEtBQUs7Z0NBQ1gsS0FBSyxFQUFFLEtBQUs7NkJBQ2I7NEJBQ0Q7Z0NBQ0UsSUFBSSxFQUFFLE9BQU87Z0NBQ2IsS0FBSyxFQUFFLEtBQUs7NkJBQ2I7eUJBQ0Y7d0JBQ0QsU0FBUyxFQUFFLElBQUk7d0JBQ2YsVUFBVSxFQUFFOzRCQUNWO2dDQUNFLFFBQVEsRUFBRSxNQUFNO2dDQUNoQixTQUFTLEVBQUUscUJBQXFCOzZCQUNqQzt5QkFDRjt3QkFDRCxXQUFXLEVBQUU7NEJBQ1gsT0FBTyxFQUFFO2dDQUNQLFdBQVc7Z0NBQ1gscUJBQXFCOzZCQUN0Qjs0QkFDRCxRQUFRLEVBQUUsRUFBRTs0QkFDWixPQUFPLEVBQUUsQ0FBQzs0QkFDVixPQUFPLEVBQUUsQ0FBQzt5QkFDWDt3QkFDRCxRQUFRLEVBQUUsa0JBQWtCO3dCQUM1QixLQUFLLEVBQUUsc0JBQXNCO3dCQUM3QixlQUFlLEVBQUU7NEJBQ2YsWUFBWSxFQUFFLEVBQUU7eUJBQ2pCO3dCQUNELGdCQUFnQixFQUFFOzRCQUNoQixTQUFTLEVBQUUsU0FBUzs0QkFDcEIsT0FBTyxFQUFFO2dDQUNQLGVBQWUsRUFBRTtvQ0FDZixHQUFHLEVBQUUsMkJBQTJCO2lDQUNqQztnQ0FDRCx1QkFBdUIsRUFBRSxRQUFRO2dDQUNqQyxnQkFBZ0IsRUFBRTtvQ0FDaEIsR0FBRyxFQUFFLGFBQWE7aUNBQ25COzZCQUNGO3lCQUNGO3dCQUNELE1BQU0sRUFBRSxJQUFJO3dCQUNaLGlCQUFpQixFQUFFLEdBQUc7d0JBQ3RCLElBQUksRUFBRSxXQUFXO3dCQUNqQixVQUFVLEVBQUUsSUFBSTt3QkFDaEIsc0JBQXNCLEVBQUUsSUFBSTt3QkFDNUIsb0JBQW9CLEVBQUU7NEJBQ3BCO2dDQUNFLElBQUksRUFBRSxLQUFLO2dDQUNYLEtBQUssRUFBRSxLQUFLOzZCQUNiO3lCQUNGO3dCQUNELE9BQU8sRUFBRTs0QkFDUDtnQ0FDRSxJQUFJLEVBQUUsUUFBUTtnQ0FDZCxTQUFTLEVBQUU7b0NBQ1QsR0FBRyxFQUFFLGdCQUFnQjtpQ0FDdEI7NkJBQ0Y7eUJBQ0Y7d0JBQ0QsWUFBWSxFQUFFLENBQUM7d0JBQ2YsV0FBVyxFQUFFLENBQUM7d0JBQ2QsSUFBSSxFQUFFLFVBQVU7d0JBQ2hCLGdCQUFnQixFQUFFLE9BQU87cUJBQzFCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsNkRBQTZELENBQUMsSUFBVTtZQUN0RSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRW5FLE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixJQUFJLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO29CQUM5QyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7b0JBQzlELGNBQWM7b0JBQ2QsY0FBYyxFQUFFLEdBQUc7b0JBQ25CLG9CQUFvQixFQUFFLElBQUk7aUJBQzNCLENBQUMsQ0FBQztZQUNMLENBQUMsRUFBRSw4REFBOEQsQ0FBQyxDQUFDO1lBRW5FLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwwQkFBMEIsRUFBRTtZQUMxQix3REFBd0QsQ0FBQyxJQUFVO2dCQUNqRSxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2lCQUNyQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO29CQUNmLFNBQVMsQ0FBQyxlQUFlLENBQUM7d0JBQ3hCLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixRQUFRLEVBQUUsSUFBSTtxQkFDZixDQUFDLENBQUM7Z0JBQ0wsQ0FBQyxDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELHlDQUF5QyxDQUFDLElBQVU7Z0JBQ2xELFFBQVE7Z0JBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7b0JBQ2pFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87aUJBQ3JDLENBQUMsQ0FBQztnQkFFSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtvQkFDekQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDO29CQUM5RCxjQUFjLEVBQUUsSUFBSTtpQkFDckIsQ0FBQyxDQUFDO2dCQUVILFNBQVMsQ0FBQyxlQUFlLENBQUM7b0JBQ3hCLGFBQWEsRUFBRSxJQUFJO29CQUNuQixRQUFRLEVBQUUsSUFBSTtpQkFDZixDQUFDLENBQUM7Z0JBRUgsMkJBQTJCO2dCQUMzQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1lBRUQseUJBQXlCLENBQUMsSUFBVTtnQkFDbEMsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtvQkFDakUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsT0FBTztpQkFDckMsQ0FBQyxDQUFDO2dCQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO29CQUN6RCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7b0JBQzlELGNBQWMsRUFBRSxJQUFJO2lCQUNyQixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUN4QixhQUFhLEVBQUUsSUFBSTtpQkFDcEIsQ0FBQyxDQUFDO2dCQUVILDJCQUEyQjtnQkFDM0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBRUQseUJBQXlCLEVBQUU7WUFDekIsd0RBQXdELENBQUMsSUFBVTtnQkFDakUsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtvQkFDakUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSTtpQkFDbEMsQ0FBQyxDQUFDO2dCQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO29CQUN6RCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7b0JBQzlELGNBQWMsRUFBRSxJQUFJO2lCQUNyQixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtvQkFDZixTQUFTLENBQUMsZUFBZSxDQUFDO3dCQUN4QixhQUFhLEVBQUUsSUFBSTt3QkFDbkIsUUFBUSxFQUFFLElBQUk7cUJBQ2YsQ0FBQyxDQUFDO2dCQUNMLENBQUMsQ0FBQyxDQUFDO2dCQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO2dCQUN2RCxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJO2lCQUNsQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUN4QixhQUFhLEVBQUUsSUFBSTtvQkFDbkIsUUFBUSxFQUFFLElBQUk7aUJBQ2YsQ0FBQyxDQUFDO2dCQUVILDJCQUEyQjtnQkFDM0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELHlCQUF5QixDQUFDLElBQVU7Z0JBQ2xDLFFBQVE7Z0JBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7b0JBQ2pFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLElBQUk7aUJBQ2xDLENBQUMsQ0FBQztnQkFFSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtvQkFDekQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDO29CQUM5RCxjQUFjLEVBQUUsSUFBSTtpQkFDckIsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsU0FBUyxDQUFDLGVBQWUsQ0FBQztvQkFDeEIsYUFBYSxFQUFFLElBQUk7aUJBQ3BCLENBQUMsQ0FBQztnQkFFSCwyQkFBMkI7Z0JBQzNCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCwwQkFBMEIsQ0FBQyxJQUFVO2dCQUNuQyxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJO2lCQUNsQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxNQUFNLE1BQU0sR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLGtCQUFrQixFQUFFO29CQUM3RCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDO29CQUNsRCxjQUFjLEVBQUUsSUFBSTtpQkFDckIsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7b0JBQ2YsU0FBUyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDNUIsQ0FBQyxDQUFDLENBQUM7Z0JBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO1FBRUQsMEJBQTBCLEVBQUU7WUFDMUIsMEJBQTBCLENBQUMsSUFBVTtnQkFDbkMsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtvQkFDakUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsTUFBTTtpQkFDcEMsQ0FBQyxDQUFDO2dCQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO29CQUN6RCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7b0JBQzlELGNBQWMsRUFBRSxJQUFJO2lCQUNyQixDQUFDLENBQUM7Z0JBRUgsU0FBUyxDQUFDLGVBQWUsQ0FBQztvQkFDeEIsYUFBYSxFQUFFLElBQUk7aUJBQ3BCLENBQUMsQ0FBQztnQkFFSCwyQkFBMkI7Z0JBQzNCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCw4QkFBOEIsQ0FBQyxJQUFVO2dCQUN2QyxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxNQUFNO2lCQUNwQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUN4QixhQUFhLEVBQUUsSUFBSTtvQkFDbkIsUUFBUSxFQUFFLElBQUk7aUJBQ2YsQ0FBQyxDQUFDO2dCQUVILDJCQUEyQjtnQkFDM0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztZQUVELHFCQUFxQixDQUFDLElBQVU7Z0JBQzlCLFFBQVE7Z0JBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7b0JBQ2pFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE1BQU07aUJBQ3BDLENBQUMsQ0FBQztnQkFFSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtvQkFDekQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDO29CQUM5RCxjQUFjLEVBQUUsSUFBSTtpQkFDckIsQ0FBQyxDQUFDO2dCQUVILE1BQU0sTUFBTSxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsa0JBQWtCLEVBQUU7b0JBQzdELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUM7b0JBQ2xELGNBQWMsRUFBRSxJQUFJO2lCQUNyQixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxTQUFTLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUUxQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0Y7UUFFRCx1QkFBdUIsRUFBRTtZQUN2Qiw0Q0FBNEMsQ0FBQyxJQUFVO2dCQUNyRCxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUU5QixPQUFPO2dCQUNQLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO29CQUNsQyxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHO29CQUNwQyxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxHQUFHO2lCQUNqQyxDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsMEJBQTBCLEVBQUUsQ0FBQyxLQUFVLEVBQUUsVUFBNkIsRUFBRSxFQUFFO29CQUN0RyxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssU0FBUyxFQUFFO3dCQUNuQyxPQUFPLElBQUksQ0FBQztxQkFDYjtvQkFFRCxVQUFVLENBQUMsYUFBYSxHQUFHLHdHQUF3RyxDQUFDO29CQUNwSSxPQUFPLEtBQUssQ0FBQztnQkFDZixDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7U0FDRjtLQUNGO0lBRUQsZ0JBQWdCLEVBQUU7UUFDaEIsd0RBQXdELENBQUMsSUFBVTtZQUNqRSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtnQkFDakUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsT0FBTzthQUNyQyxDQUFDLENBQUM7WUFFSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtnQkFDekQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDO2dCQUM5RCxjQUFjLEVBQUUsSUFBSTthQUNyQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsU0FBUyxDQUFDLGVBQWUsQ0FBQztnQkFDeEIsYUFBYSxFQUFFLElBQUk7YUFDcEIsQ0FBQyxDQUFDO1lBRUgsU0FBUyxDQUFDLGVBQWUsQ0FBQztnQkFDeEIsYUFBYSxFQUFFLElBQUk7YUFDcEIsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxNQUFNLEdBQUcsU0FBUyxDQUFDLGFBQWEsQ0FBQztZQUV2QyxPQUFPO1lBQ1AsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDO1lBQ3RCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRSxzRUFBc0UsQ0FBQyxDQUFDO1lBQ3JHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx3REFBd0QsQ0FBQyxJQUFVO1lBQ2pFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO2dCQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2FBQ3JDLENBQUMsQ0FBQztZQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsYUFBYSxFQUFFO2dCQUMzRCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7Z0JBQzlELGNBQWMsRUFBRSxJQUFJO2FBQ3JCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixNQUFNLE1BQU0sR0FBRyxTQUFTLENBQUMsYUFBYSxDQUFDO2dCQUN2QyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUM7Z0JBQ3RCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQy9CLENBQUMsRUFBRSx5RUFBeUUsQ0FBQyxDQUFDO1lBRTlFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELGNBQWMsRUFBRTtRQUNkLDBCQUEwQixFQUFFO1lBQzFCLG1EQUFtRCxDQUFDLElBQVU7Z0JBQzVELFFBQVE7Z0JBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7b0JBQ2pFLFdBQVcsRUFBRSxHQUFHLENBQUMsV0FBVyxDQUFDLE9BQU87aUJBQ3JDLENBQUMsQ0FBQztnQkFFSCxNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtvQkFDekQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLHNCQUFzQixDQUFDO29CQUM5RCxjQUFjLEVBQUUsSUFBSTtpQkFDckIsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsU0FBUyxDQUFDLGVBQWUsQ0FBQztvQkFDeEIsYUFBYSxFQUFFLElBQUk7aUJBQ3BCLENBQUMsQ0FBQztnQkFDSCxNQUFNLE1BQU0sR0FBRyxTQUFTLENBQUMsV0FBVyxDQUFDO2dCQUVyQyxPQUFPO2dCQUNQLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQztnQkFDdEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsUUFBUSxFQUFFLG1EQUFtRCxDQUFDLENBQUM7Z0JBQ2xGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCxzREFBc0QsQ0FBQyxJQUFVO2dCQUMvRCxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2lCQUNyQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUU7b0JBQzNELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO29CQUNmLE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUM7b0JBQ3JDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQztvQkFDdEIsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7Z0JBQy9CLENBQUMsRUFBRSx5RUFBeUUsQ0FBQyxDQUFDO2dCQUU5RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0Y7UUFFRCx5QkFBeUIsRUFBRTtZQUN6QixtREFBbUQsQ0FBQyxJQUFVO2dCQUM1RCxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJO2lCQUNsQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLFNBQVMsQ0FBQyxlQUFlLENBQUM7b0JBQ3hCLGFBQWEsRUFBRSxJQUFJO2lCQUNwQixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxNQUFNLEdBQUcsU0FBUyxDQUFDLFdBQVcsQ0FBQztnQkFFckMsT0FBTztnQkFDUCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUM7Z0JBQ3RCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO2dCQUM3QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0Y7UUFFRCwwQkFBMEIsRUFBRTtZQUMxQiwwREFBMEQsQ0FBQyxJQUFVO2dCQUNuRSxRQUFRO2dCQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxNQUFNO2lCQUNwQyxDQUFDLENBQUM7Z0JBRUgsTUFBTSxTQUFTLEdBQUcsY0FBYyxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUU7b0JBQ3pELEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxzQkFBc0IsQ0FBQztvQkFDOUQsY0FBYyxFQUFFLElBQUk7aUJBQ3JCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLFNBQVMsQ0FBQyxlQUFlLENBQUM7b0JBQ3hCLGFBQWEsRUFBRSxJQUFJO29CQUNuQixRQUFRLEVBQUUsSUFBSTtpQkFDZixDQUFDLENBQUM7Z0JBQ0gsTUFBTSxNQUFNLEdBQUcsU0FBUyxDQUFDLFdBQVcsQ0FBQztnQkFFckMsT0FBTztnQkFDUCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUM7Z0JBQ3RCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO2dCQUM3QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1lBRUQsMENBQTBDLENBQUMsSUFBVTtnQkFDbkQsUUFBUTtnQkFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtvQkFDakUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsTUFBTTtpQkFDcEMsQ0FBQyxDQUFDO2dCQUVILE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFO29CQUN6RCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsc0JBQXNCLENBQUM7b0JBQzlELGNBQWMsRUFBRSxJQUFJO2lCQUNyQixDQUFDLENBQUM7Z0JBRUgsT0FBTztnQkFDUCxTQUFTLENBQUMsZUFBZSxDQUFDO29CQUN4QixhQUFhLEVBQUUsSUFBSTtpQkFDcEIsQ0FBQyxDQUFDO2dCQUNILE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxXQUFXLENBQUM7Z0JBRXJDLE9BQU87Z0JBQ1AsTUFBTSxRQUFRLEdBQUcsQ0FBQyxDQUFDO2dCQUNuQixJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztnQkFDN0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2QsQ0FBQztTQUNGO0tBQ0Y7SUFFRCwyREFBMkQsQ0FBQyxJQUFVO1FBQ3BFLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFbkUsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLHlCQUF5QixFQUFFLGlDQUFpQzthQUM3RDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO1lBQzVELG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxXQUFXLEVBQUUsQ0FBQzs0QkFDWixJQUFJLEVBQUUsMkJBQTJCOzRCQUNqQyxLQUFLLEVBQUUsaUNBQWlDO3lCQUN6QyxDQUFDO2lCQUNIO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUVkLENBQUM7SUFFRCwyQkFBMkIsRUFBRTtRQUMzQix3REFBd0QsQ0FBQyxJQUFVO1lBQ2pFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsT0FBTztZQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUM5QyxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsUUFBUSxFQUFFLENBQUM7YUFDWixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLEtBQUssRUFBRSxNQUFNO3dCQUNiLG9CQUFvQixFQUFFOzRCQUNwQjtnQ0FDRSxJQUFJLEVBQUUsS0FBSztnQ0FDWCxLQUFLLEVBQUUsR0FBRzs2QkFDWDt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsa0VBQWtFLENBQUMsSUFBVTtRQUMzRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRW5FLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDMUQsTUFBTSxTQUFTLEdBQUcsR0FBRyxDQUFDLGVBQWUsQ0FBQyxtQ0FBbUMsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQzVGLGFBQWEsRUFBRSxPQUFPO1lBQ3RCLE9BQU8sRUFBRSxDQUFDO1NBQ1gsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQztnQkFDN0MsU0FBUyxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDO2FBQ2xEO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7WUFDNUQsb0JBQW9CLEVBQUU7Z0JBQ3BCO29CQUNFLE9BQU8sRUFBRTt3QkFDUDs0QkFDRSxJQUFJLEVBQUUsUUFBUTs0QkFDZCxTQUFTLEVBQUU7Z0NBQ1QsR0FBRyxFQUFFLGdCQUFnQjs2QkFDdEI7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLFdBQVc7NEJBQ2pCLFNBQVMsRUFBRTtnQ0FDVCxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELE9BQU87d0NBQ1A7NENBQ0UsR0FBRyxFQUFFLGFBQWE7eUNBQ25CO3dDQUNELEdBQUc7d0NBQ0g7NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsaUJBQWlCO3FDQUNsQjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGtCQUFrQixFQUFFO1lBQ3BELGNBQWMsRUFBRTtnQkFDZCxTQUFTLEVBQUU7b0JBQ1Q7d0JBQ0UsTUFBTSxFQUFFOzRCQUNOLCtCQUErQjs0QkFDL0IsK0JBQStCO3lCQUNoQzt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUU7NEJBQ1IsR0FBRyxFQUFFLGdCQUFnQjt5QkFDdEI7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFOzRCQUNOLHdCQUF3Qjs0QkFDeEIsbUJBQW1COzRCQUNuQixrQkFBa0I7NEJBQ2xCLHlCQUF5Qjt5QkFDMUI7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFOzRCQUNSLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGO29DQUNFLE1BQU07b0NBQ047d0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtxQ0FDdEI7b0NBQ0QsT0FBTztvQ0FDUDt3Q0FDRSxHQUFHLEVBQUUsYUFBYTtxQ0FDbkI7b0NBQ0QsR0FBRztvQ0FDSDt3Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3FDQUN0QjtvQ0FDRCxpQkFBaUI7aUNBQ2xCOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxZQUFZO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFFZCxDQUFDO0lBRUQsd0RBQXdELENBQUMsSUFBVTtRQUNqRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRW5FLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFMUQsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsT0FBTyxFQUFFO2dCQUNQLFVBQVUsRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sRUFBRSxhQUFhLENBQUM7YUFDakU7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxvQkFBb0IsRUFBRTtnQkFDcEI7b0JBQ0UsT0FBTyxFQUFFO3dCQUNQOzRCQUNFLElBQUksRUFBRSxZQUFZOzRCQUNsQixTQUFTLEVBQUU7Z0NBQ1QsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0U7NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsZ0JBQWdCO3FDQUNqQjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFFZCxDQUFDO0lBRUQsMkZBQTJGLENBQUMsSUFBVTtRQUNwRyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRXZFLE1BQU0sTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFFMUQsT0FBTztRQUNQLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7WUFDcEQsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztZQUM5QyxjQUFjLEVBQUUsSUFBSTtZQUNwQixPQUFPLEVBQUU7Z0JBQ1AsVUFBVSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUMsTUFBTSxFQUFFLGFBQWEsQ0FBQzthQUNqRTtTQUNGLENBQUMsRUFBRSwyRUFBMkUsQ0FBQyxDQUFDO1FBRWpGLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2Q0FBNkMsQ0FBQyxJQUFVO1FBQ3RELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFbkUsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsT0FBTyxFQUFFLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUUsQ0FBQztTQUMxRCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxvQkFBb0IsRUFBRTtnQkFDcEI7b0JBQ0UsZ0JBQWdCLEVBQUU7d0JBQ2hCLFNBQVMsRUFBRSxTQUFTO3dCQUNwQixPQUFPLEVBQUU7NEJBQ1AsZUFBZSxFQUFFLEVBQUUsR0FBRyxFQUFFLDZCQUE2QixFQUFFOzRCQUN2RCx1QkFBdUIsRUFBRSxRQUFROzRCQUNqQyxnQkFBZ0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7eUJBQ3pDO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyxrQkFBa0IsRUFBRTtZQUNoRCxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxDQUFDLHNCQUFzQixFQUFFLG1CQUFtQixDQUFDO3dCQUNyRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyw2QkFBNkIsRUFBRSxLQUFLLENBQUMsRUFBRTtxQkFDbkU7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxvQ0FBb0MsQ0FBQyxJQUFVO1FBQzdDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDbkUsTUFBTSxTQUFTLEdBQUcscUJBQXFCLENBQUM7UUFFeEMsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLE9BQU8sRUFBRSxDQUFDLFNBQVMsQ0FBQzthQUNyQjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO1lBQzVELG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxXQUFXLEVBQUU7d0JBQ1gsT0FBTyxFQUFFLENBQUMsV0FBVyxFQUFFLFNBQVMsQ0FBQzt3QkFDakMsUUFBUSxFQUFFLEVBQUU7d0JBQ1osT0FBTyxFQUFFLENBQUM7d0JBQ1YsT0FBTyxFQUFFLENBQUM7cUJBQ1g7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1EQUFtRCxDQUFDLElBQVU7UUFDNUQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUVuRSxPQUFPO1FBQ1AsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7WUFDbEMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztZQUM5QyxjQUFjLEVBQUUsSUFBSTtZQUNwQixXQUFXLEVBQUU7Z0JBQ1gsT0FBTyxFQUFFLEVBQUU7YUFDWjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7Z0JBQzVELG9CQUFvQixFQUFFO29CQUNwQjt3QkFDRSxXQUFXLEVBQUU7NEJBQ1gsT0FBTyxFQUFFLEVBQUU7NEJBQ1gsUUFBUSxFQUFFLEVBQUU7NEJBQ1osT0FBTyxFQUFFLENBQUM7NEJBQ1YsT0FBTyxFQUFFLENBQUM7eUJBQ1g7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztRQUNOLENBQUMsRUFBRSxrRUFBa0UsQ0FBQyxDQUFDO1FBRXZFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDbkUsTUFBTSxTQUFTLEdBQUcscUJBQXFCLENBQUM7UUFFeEMsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLE9BQU8sRUFBRSxDQUFDLFNBQVMsQ0FBQztnQkFDcEIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztnQkFDbEMsT0FBTyxFQUFFLENBQUM7Z0JBQ1YsV0FBVyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQzthQUN0QztTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO1lBQzVELG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxXQUFXLEVBQUU7d0JBQ1gsT0FBTyxFQUFFLENBQUMsV0FBVyxFQUFFLFNBQVMsQ0FBQzt3QkFDakMsUUFBUSxFQUFFLEVBQUU7d0JBQ1osT0FBTyxFQUFFLENBQUM7d0JBQ1YsT0FBTyxFQUFFLENBQUM7d0JBQ1YsV0FBVyxFQUFFLEVBQUU7cUJBQ2hCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx1RUFBdUUsQ0FBQyxJQUFVO1FBQ2hGLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDbkUsTUFBTSxTQUFTLEdBQUcscUJBQXFCLENBQUM7UUFFeEMsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO1lBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsY0FBYyxFQUFFLElBQUk7WUFDcEIsV0FBVyxFQUFFO2dCQUNYLE9BQU8sRUFBRSxDQUFDLFdBQVcsRUFBRSxTQUFTLENBQUM7Z0JBQ2pDLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7Z0JBQ2xDLE9BQU8sRUFBRSxDQUFDO2dCQUNWLFdBQVcsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7YUFDdEM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtZQUM1RCxvQkFBb0IsRUFBRTtnQkFDcEI7b0JBQ0UsV0FBVyxFQUFFO3dCQUNYLE9BQU8sRUFBRSxDQUFDLFdBQVcsRUFBRSxTQUFTLENBQUM7d0JBQ2pDLFFBQVEsRUFBRSxFQUFFO3dCQUNaLE9BQU8sRUFBRSxDQUFDO3dCQUNWLE9BQU8sRUFBRSxDQUFDO3dCQUNWLFdBQVcsRUFBRSxFQUFFO3FCQUNoQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsaUVBQWlFLENBQUMsSUFBVTtRQUMxRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ25FLE1BQU0sU0FBUyxHQUFHLHFCQUFxQixDQUFDO1FBRXhDLE9BQU87UUFDUCxjQUFjLENBQUMsWUFBWSxDQUFDLE1BQU0sRUFBRTtZQUNsQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO1lBQzlDLGNBQWMsRUFBRSxJQUFJO1lBQ3BCLFdBQVcsRUFBRTtnQkFDWCxPQUFPLEVBQUUsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDO2dCQUMzQixRQUFRLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2dCQUNsQyxPQUFPLEVBQUUsQ0FBQztnQkFDVixXQUFXLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO2FBQ3RDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7WUFDNUQsb0JBQW9CLEVBQUU7Z0JBQ3BCO29CQUNFLFdBQVcsRUFBRTt3QkFDWCxPQUFPLEVBQUUsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDO3dCQUMzQixRQUFRLEVBQUUsRUFBRTt3QkFDWixPQUFPLEVBQUUsQ0FBQzt3QkFDVixPQUFPLEVBQUUsQ0FBQzt3QkFDVixXQUFXLEVBQUUsRUFBRTtxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBDQUEwQyxDQUFDLElBQVU7UUFDbkQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNuRSxNQUFNLFdBQVcsR0FBRyxxRUFBcUUsQ0FBQztRQUUxRixNQUFNLFNBQVMsR0FBRyxjQUFjLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBRTFGLE9BQU87UUFDUCxjQUFjLENBQUMsWUFBWSxDQUFDLFdBQVcsRUFBRTtZQUN2QyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsZUFBZSxFQUFFO2dCQUN0RCxXQUFXLEVBQUUsU0FBUzthQUN2QixDQUFDO1lBQ0YsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsMEJBQTBCLEVBQUU7WUFDNUQsb0JBQW9CLEVBQUU7Z0JBQ3BCO29CQUNFLEtBQUssRUFBRSxlQUFlO29CQUN0QixxQkFBcUIsRUFBRTt3QkFDckIsb0JBQW9CLEVBQUUsV0FBVztxQkFDbEM7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxrQkFBa0IsRUFBRTtZQUNwRCxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRTs0QkFDTiwrQkFBK0I7NEJBQy9CLCtCQUErQjt5QkFDaEM7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFLFdBQVc7cUJBQ3RCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQkFBK0IsRUFBRTtRQUMvQixvREFBb0QsQ0FBQyxJQUFVO1lBQzdELFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO2dCQUNwRCxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUM5QyxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsU0FBUyxFQUFFLElBQUk7YUFDaEIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRXZELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCx3REFBd0QsQ0FBQyxJQUFVO1lBQ2pFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsT0FBTztZQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUM5QyxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsU0FBUyxFQUFFLEtBQUs7YUFDakIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRXZELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELDhCQUE4QixFQUFFO1FBQzlCLHlFQUF5RSxDQUFDLElBQVU7WUFDbEYsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUVuRSxNQUFNLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7WUFFMUUsT0FBTztZQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUM5QyxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsZUFBZTthQUNoQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLEtBQUssRUFBRSxNQUFNO3dCQUNiLGVBQWUsRUFBRTs0QkFDZixZQUFZLEVBQUUsRUFBRTt5QkFDakI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2QkFBNkIsQ0FBQyxJQUFVO1lBQ3RDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtnQkFDeEUsa0JBQWtCLEVBQUUsSUFBSTtnQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTthQUN2QixDQUFDLENBQUM7WUFFSCxlQUFlLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDcEQsZUFBZSxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFdEQsT0FBTztZQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxFQUFFO2dCQUNsQyxLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUM5QyxjQUFjLEVBQUUsSUFBSTtnQkFDcEIsZUFBZTthQUNoQixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLEtBQUssRUFBRSxNQUFNO3dCQUNiLGVBQWUsRUFBRTs0QkFDZixZQUFZLEVBQUU7Z0NBQ1osR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDO2dDQUNaLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQzs2QkFDZjs0QkFDRCxrQkFBa0IsRUFBRSxJQUFJOzRCQUN4QixnQkFBZ0IsRUFBRSxJQUFJO3lCQUN2QjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELDRCQUE0QixDQUFDLElBQVU7WUFDckMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztZQUVuRSxNQUFNLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUN4RSxrQkFBa0IsRUFBRSxJQUFJO2dCQUN4QixnQkFBZ0IsRUFBRSxJQUFJO2FBQ3ZCLENBQUMsQ0FBQztZQUVILGVBQWUsQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUVwRCxPQUFPO1lBQ1AsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQzlDLGNBQWMsRUFBRSxJQUFJO2dCQUNwQixlQUFlO2FBQ2hCLENBQUMsQ0FBQztZQUVILG1EQUFtRDtZQUNuRCxlQUFlLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUV4RCxPQUFPO1lBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQywwQkFBMEIsRUFBRTtnQkFDNUQsb0JBQW9CLEVBQUU7b0JBQ3BCO3dCQUNFLEtBQUssRUFBRSxNQUFNO3dCQUNiLGVBQWUsRUFBRTs0QkFDZixZQUFZLEVBQUU7Z0NBQ1osR0FBRyxFQUFFLENBQUMsS0FBSyxDQUFDO2dDQUNaLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQzs2QkFDakI7NEJBQ0Qsa0JBQWtCLEVBQUUsSUFBSTs0QkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTt5QkFDdkI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwrQkFBK0IsQ0FBQyxJQUFVO1lBQ3hDLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtnQkFDeEUsa0JBQWtCLEVBQUUsSUFBSTtnQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTthQUN2QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDLFVBQVUsQ0FBQztnQkFDekIsUUFBUSxFQUFFLE9BQU87YUFDbEIsQ0FBQyxDQUFDO1lBRUgsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQzlDLGNBQWMsRUFBRSxJQUFJO2dCQUNwQixlQUFlO2FBQ2hCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO2dCQUM1RCxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsS0FBSyxFQUFFLE1BQU07d0JBQ2IsZUFBZSxFQUFFOzRCQUNmLE9BQU8sRUFBRTtnQ0FDUDtvQ0FDRSxRQUFRLEVBQUUsT0FBTztpQ0FDbEI7NkJBQ0Y7NEJBQ0Qsa0JBQWtCLEVBQUUsSUFBSTs0QkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTt5QkFDdkI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1lBQy9DLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFFbkUsTUFBTSxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBRTtnQkFDeEUsa0JBQWtCLEVBQUUsSUFBSTtnQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTthQUN2QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsZUFBZSxDQUFDLFFBQVEsQ0FBQztnQkFDdkIsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLElBQUksRUFBRSxJQUFJO2FBQ1gsQ0FBQyxDQUFDO1lBRUgsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQzlDLGNBQWMsRUFBRSxJQUFJO2dCQUNwQixlQUFlO2FBQ2hCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDBCQUEwQixFQUFFO2dCQUM1RCxvQkFBb0IsRUFBRTtvQkFDcEI7d0JBQ0UsS0FBSyxFQUFFLE1BQU07d0JBQ2IsZUFBZSxFQUFFOzRCQUNmLEtBQUssRUFBRTtnQ0FDTDtvQ0FDRSxhQUFhLEVBQUUsT0FBTztvQ0FDdEIsSUFBSSxFQUFFLElBQUk7aUNBQ1g7NkJBQ0Y7NEJBQ0Qsa0JBQWtCLEVBQUUsSUFBSTs0QkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTt5QkFDdkI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUMsQ0FBQztZQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtJQUVELDJEQUEyRCxDQUFDLElBQVU7UUFDcEUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNuRSxNQUFNLEtBQUssR0FBRyxJQUFJLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO1lBQ3BFLFNBQVMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUM7U0FDOUMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGNBQWMsQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFO1lBQ3JDLEtBQUssRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQztZQUNyRCxjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDBCQUEwQixFQUFFO1lBQ3hELG9CQUFvQixFQUFFO2dCQUNwQjtvQkFDRSxTQUFTLEVBQUUsSUFBSTtvQkFDZixLQUFLLEVBQUU7d0JBQ0wsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUU7Z0NBQ3pCLFdBQVc7Z0NBQ1gsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO2dDQUN0QixHQUFHO2dDQUNILEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO2dDQUN6QixrRkFBa0Y7NkJBQ25GO3lCQUNGO3FCQUNGO29CQUNELE1BQU0sRUFBRSxJQUFJO29CQUNaLElBQUksRUFBRSxTQUFTO2lCQUNoQjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7WUFDaEQsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUU7NEJBQ04saUNBQWlDOzRCQUNqQyw0QkFBNEI7NEJBQzVCLG1CQUFtQjt5QkFDcEI7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFOzRCQUNSLFVBQVUsRUFBRTtnQ0FDVixFQUFFO2dDQUNGLENBQUUsTUFBTSxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFLEVBQUUsT0FBTyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLDRCQUE0QixDQUFFOzZCQUNySTt5QkFDRjtxQkFDRjtvQkFDRDt3QkFDRSxNQUFNLEVBQUUsMkJBQTJCO3dCQUNuQyxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsR0FBRztxQkFDZDtpQkFDRjtnQkFDRCxPQUFPLEVBQUUsWUFBWTthQUN0QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1FQUFtRSxDQUFDLElBQVU7UUFDNUUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQzFCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDNUMsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBRW5FLE9BQU87UUFDUCxjQUFjLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRTtZQUNyQyxjQUFjLEVBQUUsSUFBSTtZQUNwQixLQUFLLEVBQUUsR0FBRyxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLEVBQUU7Z0JBQ3RFLElBQUksRUFBRSxVQUFVO2dCQUNoQixNQUFNLEVBQUUsY0FBYzthQUN2QixDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQy9ELGNBQWMsRUFBRSxnQkFBZ0I7WUFDaEMsUUFBUSxFQUFFLGtFQUFrRTtZQUM1RSxFQUFFLEVBQUUsa0VBQWtFO1lBQ3RFLFNBQVMsRUFBRSxpQkFBaUI7WUFDNUIsSUFBSSxFQUFFLHdFQUF3RTtZQUM5RSxVQUFVLEVBQUUsa0VBQWtFO1lBQzlFLE1BQU0sRUFBRSxjQUFjO1lBQ3RCLElBQUksRUFBRSxVQUFVO1NBQ2pCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgZXhwZWN0LCBoYXZlUmVzb3VyY2UsIGhhdmVSZXNvdXJjZUxpa2UsIEluc3BlY3Rpb25GYWlsdXJlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGVjcl9hc3NldHMgZnJvbSAnQGF3cy1jZGsvYXdzLWVjci1hc3NldHMnO1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCAqIGFzIHNzbSBmcm9tICdAYXdzLWNkay9hd3Mtc3NtJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJy4uL2xpYic7XG5cbmV4cG9ydCA9IHtcbiAgJ1doZW4gY3JlYXRpbmcgYSBUYXNrIERlZmluaXRpb24nOiB7XG4gICAgJ2FkZCBhIGNvbnRhaW5lciB1c2luZyBkZWZhdWx0IHByb3BzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAgIG5ldyBlY3MuQ29udGFpbmVyRGVmaW5pdGlvbihzdGFjaywgJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgIHRhc2tEZWZpbml0aW9uLFxuICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgICAgICBJbWFnZTogJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyxcbiAgICAgICAgICAgIE1lbW9yeTogMjA0OCxcbiAgICAgICAgICAgIE5hbWU6ICdDb250YWluZXInLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYWRkIGEgY29udGFpbmVyIHVzaW5nIGFsbCBwcm9wcycodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicpO1xuICAgICAgY29uc3Qgc2VjcmV0ID0gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldChzdGFjaywgJ1NlY3JldCcpO1xuICAgICAgbmV3IGVjcy5Db250YWluZXJEZWZpbml0aW9uKHN0YWNrLCAnQ29udGFpbmVyJywge1xuICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgdGFza0RlZmluaXRpb24sXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgICBtZW1vcnlSZXNlcnZhdGlvbk1pQjogNTEyLFxuICAgICAgICBjb21tYW5kOiBbJ0NNRC1TSEVMTCddLFxuICAgICAgICBjcHU6IDEyOCxcbiAgICAgICAgZGlzYWJsZU5ldHdvcmtpbmc6IHRydWUsXG4gICAgICAgIGRuc1NlYXJjaERvbWFpbnM6IFsnZXhhbXBsZS5jb20nXSxcbiAgICAgICAgZG5zU2VydmVyczogWydob3N0LmNvbSddLFxuICAgICAgICBkb2NrZXJMYWJlbHM6IHtcbiAgICAgICAgICBrZXk6ICdmb29MYWJlbCcsXG4gICAgICAgICAgdmFsdWU6ICdiYXJMYWJlbCcsXG4gICAgICAgIH0sXG4gICAgICAgIGRvY2tlclNlY3VyaXR5T3B0aW9uczogWydFQ1NfU0VMSU5VWF9DQVBBQkxFPXRydWUnXSxcbiAgICAgICAgZW50cnlQb2ludDogWyd0b3AnLCAnLWInXSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBrZXk6ICdmb28nLFxuICAgICAgICAgIHZhbHVlOiAnYmFyJyxcbiAgICAgICAgfSxcbiAgICAgICAgZXNzZW50aWFsOiB0cnVlLFxuICAgICAgICBleHRyYUhvc3RzOiB7XG4gICAgICAgICAgbmFtZTogJ2Rldi1kYi5ob3N0bmFtZS5wdnQnLFxuICAgICAgICB9LFxuICAgICAgICBncHVDb3VudDogMjU2LFxuICAgICAgICBob3N0bmFtZTogJ2hvc3QuZXhhbXBsZS5jb20nLFxuICAgICAgICBwcml2aWxlZ2VkOiB0cnVlLFxuICAgICAgICByZWFkb25seVJvb3RGaWxlc3lzdGVtOiB0cnVlLFxuICAgICAgICBzdGFydFRpbWVvdXQ6IGNkay5EdXJhdGlvbi5taWxsaXMoMjAwMCksXG4gICAgICAgIHN0b3BUaW1lb3V0OiBjZGsuRHVyYXRpb24ubWlsbGlzKDUwMDApLFxuICAgICAgICB1c2VyOiAncm9vdFVzZXInLFxuICAgICAgICB3b3JraW5nRGlyZWN0b3J5OiAnYS9iL2MnLFxuICAgICAgICBoZWFsdGhDaGVjazoge1xuICAgICAgICAgIGNvbW1hbmQ6IFsnY3VybCBsb2NhbGhvc3Q6ODAwMCddLFxuICAgICAgICB9LFxuICAgICAgICBsaW51eFBhcmFtZXRlcnM6IG5ldyBlY3MuTGludXhQYXJhbWV0ZXJzKHN0YWNrLCAnTGludXhQYXJhbWV0ZXJzJyksXG4gICAgICAgIGxvZ2dpbmc6IG5ldyBlY3MuQXdzTG9nRHJpdmVyKHsgc3RyZWFtUHJlZml4OiAncHJlZml4JyB9KSxcbiAgICAgICAgc2VjcmV0czoge1xuICAgICAgICAgIFNFQ1JFVDogZWNzLlNlY3JldC5mcm9tU2VjcmV0c01hbmFnZXIoc2VjcmV0KSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBDb21tYW5kOiBbXG4gICAgICAgICAgICAgICdDTUQtU0hFTEwnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIENwdTogMTI4LFxuICAgICAgICAgICAgRGlzYWJsZU5ldHdvcmtpbmc6IHRydWUsXG4gICAgICAgICAgICBEbnNTZWFyY2hEb21haW5zOiBbXG4gICAgICAgICAgICAgICdleGFtcGxlLmNvbScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRG5zU2VydmVyczogW1xuICAgICAgICAgICAgICAnaG9zdC5jb20nLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIERvY2tlckxhYmVsczoge1xuICAgICAgICAgICAgICBrZXk6ICdmb29MYWJlbCcsXG4gICAgICAgICAgICAgIHZhbHVlOiAnYmFyTGFiZWwnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIERvY2tlclNlY3VyaXR5T3B0aW9uczogW1xuICAgICAgICAgICAgICAnRUNTX1NFTElOVVhfQ0FQQUJMRT10cnVlJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFbnRyeVBvaW50OiBbXG4gICAgICAgICAgICAgICd0b3AnLFxuICAgICAgICAgICAgICAnLWInLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEVudmlyb25tZW50OiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBOYW1lOiAna2V5JyxcbiAgICAgICAgICAgICAgICBWYWx1ZTogJ2ZvbycsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBOYW1lOiAndmFsdWUnLFxuICAgICAgICAgICAgICAgIFZhbHVlOiAnYmFyJyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgICAgICBFeHRyYUhvc3RzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBIb3N0bmFtZTogJ25hbWUnLFxuICAgICAgICAgICAgICAgIElwQWRkcmVzczogJ2Rldi1kYi5ob3N0bmFtZS5wdnQnLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEhlYWx0aENoZWNrOiB7XG4gICAgICAgICAgICAgIENvbW1hbmQ6IFtcbiAgICAgICAgICAgICAgICAnQ01ELVNIRUxMJyxcbiAgICAgICAgICAgICAgICAnY3VybCBsb2NhbGhvc3Q6ODAwMCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIEludGVydmFsOiAzMCxcbiAgICAgICAgICAgICAgUmV0cmllczogMyxcbiAgICAgICAgICAgICAgVGltZW91dDogNSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBIb3N0bmFtZTogJ2hvc3QuZXhhbXBsZS5jb20nLFxuICAgICAgICAgICAgSW1hZ2U6ICcvYXdzL2F3cy1leGFtcGxlLWFwcCcsXG4gICAgICAgICAgICBMaW51eFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgICAgQ2FwYWJpbGl0aWVzOiB7fSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBMb2dDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICAgIExvZ0RyaXZlcjogJ2F3c2xvZ3MnLFxuICAgICAgICAgICAgICBPcHRpb25zOiB7XG4gICAgICAgICAgICAgICAgJ2F3c2xvZ3MtZ3JvdXAnOiB7XG4gICAgICAgICAgICAgICAgICBSZWY6ICdDb250YWluZXJMb2dHcm91cEU2RkQ3NEE0JyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICdhd3Nsb2dzLXN0cmVhbS1wcmVmaXgnOiAncHJlZml4JyxcbiAgICAgICAgICAgICAgICAnYXdzbG9ncy1yZWdpb24nOiB7XG4gICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlJlZ2lvbicsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBNZW1vcnk6IDEwMjQsXG4gICAgICAgICAgICBNZW1vcnlSZXNlcnZhdGlvbjogNTEyLFxuICAgICAgICAgICAgTmFtZTogJ0NvbnRhaW5lcicsXG4gICAgICAgICAgICBQcml2aWxlZ2VkOiB0cnVlLFxuICAgICAgICAgICAgUmVhZG9ubHlSb290RmlsZXN5c3RlbTogdHJ1ZSxcbiAgICAgICAgICAgIFJlc291cmNlUmVxdWlyZW1lbnRzOiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBUeXBlOiAnR1BVJyxcbiAgICAgICAgICAgICAgICBWYWx1ZTogJzI1NicsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgU2VjcmV0czogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgTmFtZTogJ1NFQ1JFVCcsXG4gICAgICAgICAgICAgICAgVmFsdWVGcm9tOiB7XG4gICAgICAgICAgICAgICAgICBSZWY6ICdTZWNyZXRBNzIwRUYwNScsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBTdGFydFRpbWVvdXQ6IDIsXG4gICAgICAgICAgICBTdG9wVGltZW91dDogNSxcbiAgICAgICAgICAgIFVzZXI6ICdyb290VXNlcicsXG4gICAgICAgICAgICBXb3JraW5nRGlyZWN0b3J5OiAnYS9iL2MnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhyb3dzIHdoZW4gTWVtb3J5TGltaXQgaXMgbGVzcyB0aGFuIE1lbW9yeVJlc2VydmF0aW9uTGltaXQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBuZXcgZWNzLkNvbnRhaW5lckRlZmluaXRpb24oc3RhY2ssICdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgdGFza0RlZmluaXRpb24sXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDUxMixcbiAgICAgICAgICBtZW1vcnlSZXNlcnZhdGlvbk1pQjogMTAyNCxcbiAgICAgICAgfSk7XG4gICAgICB9LCAvTWVtb3J5TGltaXRNaUIgc2hvdWxkIG5vdCBiZSBsZXNzIHRoYW4gTWVtb3J5UmVzZXJ2YXRpb25NaUIuLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnV2l0aCBuZXR3b3JrIG1vZGUgQXdzVnBjJzoge1xuICAgICAgJ3Rocm93cyB3aGVuIEhvc3QgcG9ydCBpcyBkaWZmZXJlbnQgZnJvbSBjb250YWluZXIgcG9ydCcodGVzdDogVGVzdCkge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicsIHtcbiAgICAgICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkFXU19WUEMsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignQ29udGFpbmVyJywge1xuICAgICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCcvYXdzL2F3cy1leGFtcGxlLWFwcCcpLFxuICAgICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgICBjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwODAsXG4gICAgICAgICAgICBob3N0UG9ydDogODA4MSxcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuXG4gICAgICAnSG9zdCBwb3J0IGlzIHRoZSBzYW1lIGFzIGNvbnRhaW5lciBwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDIwNDgsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwODAsXG4gICAgICAgICAgaG9zdFBvcnQ6IDgwODAsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU4gbm8gZXhjZXB0aW9uIHJhaXNlZFxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICdIb3N0IHBvcnQgY2FuIGJlIGVtcHR5ICcodGVzdDogVGVzdCkge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicsIHtcbiAgICAgICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkFXU19WUEMsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignQ29udGFpbmVyJywge1xuICAgICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCcvYXdzL2F3cy1leGFtcGxlLWFwcCcpLFxuICAgICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwODAsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU4gbm8gZXhjZXB0aW9uIHJhaXNlZFxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcblxuICAgICdXaXRoIG5ldHdvcmsgbW9kZSBIb3N0ICc6IHtcbiAgICAgICd0aHJvd3Mgd2hlbiBIb3N0IHBvcnQgaXMgZGlmZmVyZW50IGZyb20gY29udGFpbmVyIHBvcnQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5IT1NULFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgICAgY29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgICAgICBjb250YWluZXJQb3J0OiA4MDgwLFxuICAgICAgICAgICAgaG9zdFBvcnQ6IDgwODEsXG4gICAgICAgICAgfSk7XG4gICAgICAgIH0pO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ3doZW4gaG9zdCBwb3J0IGlzIHRoZSBzYW1lIGFzIGNvbnRhaW5lciBwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuSE9TVCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDIwNDgsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwODAsXG4gICAgICAgICAgaG9zdFBvcnQ6IDgwODAsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU4gbm8gZXhjZXB0aW9uIHJhaXNlZFxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICdIb3N0IHBvcnQgY2FuIGJlIGVtcHR5ICcodGVzdDogVGVzdCkge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicsIHtcbiAgICAgICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkhPU1QsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignQ29udGFpbmVyJywge1xuICAgICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCcvYXdzL2F3cy1leGFtcGxlLWFwcCcpLFxuICAgICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICAgIGNvbnRhaW5lclBvcnQ6IDgwODAsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFRIRU4gbm8gZXhjZXB0aW9uIHJhaXNlZFxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICdlcnJvcnMgd2hlbiBhZGRpbmcgbGlua3MnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5IT1NULFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgbG9nZ2VyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdMb2dnaW5nQ29udGFpbmVyJywge1xuICAgICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdteUxvZ2dlcicpLFxuICAgICAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgICBjb250YWluZXIuYWRkTGluayhsb2dnZXIpO1xuICAgICAgICB9KTtcblxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcblxuICAgICdXaXRoIG5ldHdvcmsgbW9kZSBCcmlkZ2UnOiB7XG4gICAgICAnd2hlbiBIb3N0IHBvcnQgaXMgZW1wdHkgJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQlJJREdFLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgICAgY29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTiBubyBleGNlcHRpb24gcmFpc2VzXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ3doZW4gSG9zdCBwb3J0IGlzIG5vdCBlbXB0eSAnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgbmV0d29ya01vZGU6IGVjcy5OZXR3b3JrTW9kZS5CUklER0UsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGNvbnRhaW5lciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignQ29udGFpbmVyJywge1xuICAgICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCcvYXdzL2F3cy1leGFtcGxlLWFwcCcpLFxuICAgICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgICB9KTtcblxuICAgICAgICBjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgICAgICBjb250YWluZXJQb3J0OiA4MDgwLFxuICAgICAgICAgIGhvc3RQb3J0OiA4MDg0LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOIG5vIGV4Y2VwdGlvbiByYWlzZXNcbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuXG4gICAgICAnYWxsb3dzIGFkZGluZyBsaW5rcycodGVzdDogVGVzdCkge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicsIHtcbiAgICAgICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkJSSURHRSxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDIwNDgsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIGNvbnN0IGxvZ2dlciA9IHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignTG9nZ2luZ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnbXlMb2dnZXInKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBjb250YWluZXIuYWRkTGluayhsb2dnZXIpO1xuXG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuXG4gICAgJ1dpdGggbmV0d29yayBtb2RlIE5BVCc6IHtcbiAgICAgICdwcm9kdWNlcyB1bmRlZmluZWQgQ0YgbmV0d29ya01vZGUgcHJvcGVydHknKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBuZXcgZWNzLlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVEQnLCB7XG4gICAgICAgICAgY29tcGF0aWJpbGl0eTogZWNzLkNvbXBhdGliaWxpdHkuRUMyLFxuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuTkFULFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCAocHJvcHM6IGFueSwgaW5zcGVjdGlvbjogSW5zcGVjdGlvbkZhaWx1cmUpID0+IHtcbiAgICAgICAgICBpZiAocHJvcHMuTmV0d29ya01vZGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaW5zcGVjdGlvbi5mYWlsdXJlUmVhc29uID0gJ0NGIHRlbXBsYXRlIHNob3VsZCBub3QgaGF2ZSBOZXR3b3JrTW9kZSBkZWZpbmVkIGZvciBhIHRhc2sgZGVmaW5pdGlvbiB0aGF0IHJlbGllcyBvbiBOQVQgbmV0d29yayBtb2RlLic7XG4gICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICB9KSk7XG5cbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuICAgIH0sXG4gIH0sXG5cbiAgJ0NvbnRhaW5lciBQb3J0Jzoge1xuICAgICdzaG91bGQgcmV0dXJuIHRoZSBmaXJzdCBjb250YWluZXIgcG9ydCBpbiBQb3J0TWFwcGluZ3MnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICBjb250YWluZXJQb3J0OiA4MDgwLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbnRhaW5lci5hZGRQb3J0TWFwcGluZ3Moe1xuICAgICAgICBjb250YWluZXJQb3J0OiA4MDgxLFxuICAgICAgfSk7XG4gICAgICBjb25zdCBhY3R1YWwgPSBjb250YWluZXIuY29udGFpbmVyUG9ydDtcblxuICAgICAgLy8gVEhFTlxuICAgICAgY29uc3QgZXhwZWN0ZWQgPSA4MDgwO1xuICAgICAgdGVzdC5lcXVhbChhY3R1YWwsIGV4cGVjdGVkLCAnY29udGFpbmVyUG9ydCBzaG91bGQgcmV0dXJuIHRoZSBmaXJzdCBjb250YWluZXIgcG9ydCBpbiBQb3J0TWFwcGluZ3MnKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAndGhyb3dzIHdoZW4gY2FsbGluZyBjb250YWluZXJQb3J0IHdpdGggbm8gUG9ydE1hcHBpbmdzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICBuZXR3b3JrTW9kZTogZWNzLk5ldHdvcmtNb2RlLkFXU19WUEMsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdNeUNvbnRhaW5lcicsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgY29uc3QgYWN0dWFsID0gY29udGFpbmVyLmNvbnRhaW5lclBvcnQ7XG4gICAgICAgIGNvbnN0IGV4cGVjdGVkID0gODA4MDtcbiAgICAgICAgdGVzdC5lcXVhbChhY3R1YWwsIGV4cGVjdGVkKTtcbiAgICAgIH0sIC9Db250YWluZXIgTXlDb250YWluZXIgaGFzbid0IGRlZmluZWQgYW55IHBvcnRzLiBDYWxsIGFkZFBvcnRNYXBwaW5ncygpLi8pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICB9LFxuXG4gICdJbmdyZXNzIFBvcnQnOiB7XG4gICAgJ1dpdGggbmV0d29yayBtb2RlIEF3c1ZwYyc6IHtcbiAgICAgICdJbmdyZXNzIHBvcnQgc2hvdWxkIGJlIHRoZSBzYW1lIGFzIGNvbnRhaW5lciBwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDIwNDgsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgICAgY29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGFjdHVhbCA9IGNvbnRhaW5lci5pbmdyZXNzUG9ydDtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGNvbnN0IGV4cGVjdGVkID0gODA4MDtcbiAgICAgICAgdGVzdC5lcXVhbChhY3R1YWwsIGV4cGVjdGVkLCAnSW5ncmVzcyBwb3J0IHNob3VsZCBiZSB0aGUgc2FtZSBhcyBjb250YWluZXIgcG9ydCcpO1xuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICd0aHJvd3Mgd2hlbiBjYWxsaW5nIGluZ3Jlc3NQb3J0IHdpdGggbm8gUG9ydE1hcHBpbmdzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQVdTX1ZQQyxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdNeUNvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgICAgY29uc3QgYWN0dWFsID0gY29udGFpbmVyLmluZ3Jlc3NQb3J0O1xuICAgICAgICAgIGNvbnN0IGV4cGVjdGVkID0gODA4MDtcbiAgICAgICAgICB0ZXN0LmVxdWFsKGFjdHVhbCwgZXhwZWN0ZWQpO1xuICAgICAgICB9LCAvQ29udGFpbmVyIE15Q29udGFpbmVyIGhhc24ndCBkZWZpbmVkIGFueSBwb3J0cy4gQ2FsbCBhZGRQb3J0TWFwcGluZ3MoKS4vKTtcblxuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcblxuICAgICdXaXRoIG5ldHdvcmsgbW9kZSBIb3N0ICc6IHtcbiAgICAgICdJbmdyZXNzIHBvcnQgc2hvdWxkIGJlIHRoZSBzYW1lIGFzIGNvbnRhaW5lciBwb3J0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuSE9TVCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdDb250YWluZXInLCB7XG4gICAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJy9hd3MvYXdzLWV4YW1wbGUtYXBwJyksXG4gICAgICAgICAgbWVtb3J5TGltaXRNaUI6IDIwNDgsXG4gICAgICAgIH0pO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29udGFpbmVyLmFkZFBvcnRNYXBwaW5ncyh7XG4gICAgICAgICAgY29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGFjdHVhbCA9IGNvbnRhaW5lci5pbmdyZXNzUG9ydDtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGNvbnN0IGV4cGVjdGVkID0gODA4MDtcbiAgICAgICAgdGVzdC5lcXVhbChhY3R1YWwsIGV4cGVjdGVkKTtcbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuICAgIH0sXG5cbiAgICAnV2l0aCBuZXR3b3JrIG1vZGUgQnJpZGdlJzoge1xuICAgICAgJ0luZ3Jlc3MgcG9ydCBzaG91bGQgYmUgdGhlIHNhbWUgYXMgaG9zdCBwb3J0IGlmIHN1cHBsaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQlJJREdFLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgICAgICBjb250YWluZXJQb3J0OiA4MDgwLFxuICAgICAgICAgIGhvc3RQb3J0OiA4MDgxLFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgYWN0dWFsID0gY29udGFpbmVyLmluZ3Jlc3NQb3J0O1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgY29uc3QgZXhwZWN0ZWQgPSA4MDgxO1xuICAgICAgICB0ZXN0LmVxdWFsKGFjdHVhbCwgZXhwZWN0ZWQpO1xuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG5cbiAgICAgICdJbmdyZXNzIHBvcnQgc2hvdWxkIGJlIDAgaWYgbm90IHN1cHBsaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJywge1xuICAgICAgICAgIG5ldHdvcmtNb2RlOiBlY3MuTmV0d29ya01vZGUuQlJJREdFLFxuICAgICAgICB9KTtcblxuICAgICAgICBjb25zdCBjb250YWluZXIgPSB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ0NvbnRhaW5lcicsIHtcbiAgICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgnL2F3cy9hd3MtZXhhbXBsZS1hcHAnKSxcbiAgICAgICAgICBtZW1vcnlMaW1pdE1pQjogMjA0OCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBjb250YWluZXIuYWRkUG9ydE1hcHBpbmdzKHtcbiAgICAgICAgICBjb250YWluZXJQb3J0OiA4MDgxLFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgYWN0dWFsID0gY29udGFpbmVyLmluZ3Jlc3NQb3J0O1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgY29uc3QgZXhwZWN0ZWQgPSAwO1xuICAgICAgICB0ZXN0LmVxdWFsKGFjdHVhbCwgZXhwZWN0ZWQpO1xuICAgICAgICB0ZXN0LmRvbmUoKTtcbiAgICAgIH0sXG4gICAgfSxcbiAgfSxcblxuICAnY2FuIGFkZCBlbnZpcm9ubWVudCB2YXJpYWJsZXMgdG8gdGhlIGNvbnRhaW5lciBkZWZpbml0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgVEVTVF9FTlZJUk9OTUVOVF9WQVJJQUJMRTogJ3Rlc3QgZW52aXJvbm1lbnQgdmFyaWFibGUgdmFsdWUnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBFbnZpcm9ubWVudDogW3tcbiAgICAgICAgICAgIE5hbWU6ICdURVNUX0VOVklST05NRU5UX1ZBUklBQkxFJyxcbiAgICAgICAgICAgIFZhbHVlOiAndGVzdCBlbnZpcm9ubWVudCB2YXJpYWJsZSB2YWx1ZScsXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuXG4gIH0sXG5cbiAgJ0dpdmVuIEdQVSBjb3VudCBwYXJhbWV0ZXInOiB7XG4gICAgJ3dpbGwgYWRkIHJlc291cmNlIHJlcXVpcmVtZW50cyB0byBjb250YWluZXIgZGVmaW5pdGlvbicodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2NvbnQnLCB7XG4gICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCd0ZXN0JyksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgICBncHVDb3VudDogNCxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbWFnZTogJ3Rlc3QnLFxuICAgICAgICAgICAgUmVzb3VyY2VSZXF1aXJlbWVudHM6IFtcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFR5cGU6ICdHUFUnLFxuICAgICAgICAgICAgICAgIFZhbHVlOiAnNCcsXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgJ2NhbiBhZGQgc2VjcmV0IGVudmlyb25tZW50IHZhcmlhYmxlcyB0byB0aGUgY29udGFpbmVyIGRlZmluaXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgIGNvbnN0IHNlY3JldCA9IG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc3RhY2ssICdTZWNyZXQnKTtcbiAgICBjb25zdCBwYXJhbWV0ZXIgPSBzc20uU3RyaW5nUGFyYW1ldGVyLmZyb21TZWN1cmVTdHJpbmdQYXJhbWV0ZXJBdHRyaWJ1dGVzKHN0YWNrLCAnUGFyYW1ldGVyJywge1xuICAgICAgcGFyYW1ldGVyTmFtZTogJy9uYW1lJyxcbiAgICAgIHZlcnNpb246IDEsXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgc2VjcmV0czoge1xuICAgICAgICBTRUNSRVQ6IGVjcy5TZWNyZXQuZnJvbVNlY3JldHNNYW5hZ2VyKHNlY3JldCksXG4gICAgICAgIFBBUkFNRVRFUjogZWNzLlNlY3JldC5mcm9tU3NtUGFyYW1ldGVyKHBhcmFtZXRlciksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFNlY3JldHM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgTmFtZTogJ1NFQ1JFVCcsXG4gICAgICAgICAgICAgIFZhbHVlRnJvbToge1xuICAgICAgICAgICAgICAgIFJlZjogJ1NlY3JldEE3MjBFRjA1JyxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIE5hbWU6ICdQQVJBTUVURVInLFxuICAgICAgICAgICAgICBWYWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAgJ2FybjonLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnOnNzbTonLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpSZWdpb24nLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OkFjY291bnRJZCcsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICc6cGFyYW1ldGVyL25hbWUnLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkdldFNlY3JldFZhbHVlJyxcbiAgICAgICAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkRlc2NyaWJlU2VjcmV0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBSZWY6ICdTZWNyZXRBNzIwRUYwNScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzc206RGVzY3JpYmVQYXJhbWV0ZXJzJyxcbiAgICAgICAgICAgICAgJ3NzbTpHZXRQYXJhbWV0ZXJzJyxcbiAgICAgICAgICAgICAgJ3NzbTpHZXRQYXJhbWV0ZXInLFxuICAgICAgICAgICAgICAnc3NtOkdldFBhcmFtZXRlckhpc3RvcnknLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOnNzbTonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlJlZ2lvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OkFjY291bnRJZCcsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpwYXJhbWV0ZXIvbmFtZScsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcblxuICB9LFxuXG4gICd1c2UgYSBzcGVjaWZpYyBzZWNyZXQgSlNPTiBrZXkgYXMgZW52aXJvbm1lbnQgdmFyaWFibGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgIGNvbnN0IHNlY3JldCA9IG5ldyBzZWNyZXRzbWFuYWdlci5TZWNyZXQoc3RhY2ssICdTZWNyZXQnKTtcblxuICAgIC8vIFdIRU5cbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2NvbnQnLCB7XG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgndGVzdCcpLFxuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBzZWNyZXRzOiB7XG4gICAgICAgIFNFQ1JFVF9LRVk6IGVjcy5TZWNyZXQuZnJvbVNlY3JldHNNYW5hZ2VyKHNlY3JldCwgJ3NwZWNpZmljS2V5JyksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFNlY3JldHM6IFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgTmFtZTogJ1NFQ1JFVF9LRVknLFxuICAgICAgICAgICAgICBWYWx1ZUZyb206IHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ1NlY3JldEE3MjBFRjA1JyxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgJzpzcGVjaWZpY0tleTo6JyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG5cbiAgfSxcblxuICAndGhyb3dzIHdoZW4gdXNpbmcgYSBzcGVjaWZpYyBzZWNyZXQgSlNPTiBmaWVsZCBhcyBlbnZpcm9ubWVudCB2YXJpYWJsZSBmb3IgYSBGYXJnYXRlIHRhc2snKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICBjb25zdCBzZWNyZXQgPSBuZXcgc2VjcmV0c21hbmFnZXIuU2VjcmV0KHN0YWNrLCAnU2VjcmV0Jyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgc2VjcmV0czoge1xuICAgICAgICBTRUNSRVRfS0VZOiBlY3MuU2VjcmV0LmZyb21TZWNyZXRzTWFuYWdlcihzZWNyZXQsICdzcGVjaWZpY0tleScpLFxuICAgICAgfSxcbiAgICB9KSwgL0Nhbm5vdCBzcGVjaWZ5IHNlY3JldCBKU09OIGZpZWxkIGZvciBhIHRhc2sgdXNpbmcgdGhlIEZBUkdBVEUgbGF1bmNoIHR5cGUvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gYWRkIEFXUyBsb2dnaW5nIHRvIGNvbnRhaW5lciBkZWZpbml0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgbG9nZ2luZzogbmV3IGVjcy5Bd3NMb2dEcml2ZXIoeyBzdHJlYW1QcmVmaXg6ICdwcmVmaXgnIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIExvZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIExvZ0RyaXZlcjogJ2F3c2xvZ3MnLFxuICAgICAgICAgICAgT3B0aW9uczoge1xuICAgICAgICAgICAgICAnYXdzbG9ncy1ncm91cCc6IHsgUmVmOiAnVGFza0RlZmNvbnRMb2dHcm91cDRFMTBEQ0JGJyB9LFxuICAgICAgICAgICAgICAnYXdzbG9ncy1zdHJlYW0tcHJlZml4JzogJ3ByZWZpeCcsXG4gICAgICAgICAgICAgICdhd3Nsb2dzLXJlZ2lvbic6IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJywgJ2xvZ3M6UHV0TG9nRXZlbnRzJ10sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogeyAnRm46OkdldEF0dCc6IFsnVGFza0RlZmNvbnRMb2dHcm91cDRFMTBEQ0JGJywgJ0FybiddIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY2FuIHNldCBIZWFsdGggQ2hlY2sgd2l0aCBkZWZhdWx0cycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFza0RlZmluaXRpb24gPSBuZXcgZWNzLkVjMlRhc2tEZWZpbml0aW9uKHN0YWNrLCAnVGFza0RlZicpO1xuICAgIGNvbnN0IGhjQ29tbWFuZCA9ICdjdXJsIGxvY2FsaG9zdDo4MDAwJztcblxuICAgIC8vIFdIRU5cbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2NvbnQnLCB7XG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgndGVzdCcpLFxuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBoZWFsdGhDaGVjazoge1xuICAgICAgICBjb21tYW5kOiBbaGNDb21tYW5kXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICB7XG4gICAgICAgICAgSGVhbHRoQ2hlY2s6IHtcbiAgICAgICAgICAgIENvbW1hbmQ6IFsnQ01ELVNIRUxMJywgaGNDb21tYW5kXSxcbiAgICAgICAgICAgIEludGVydmFsOiAzMCxcbiAgICAgICAgICAgIFJldHJpZXM6IDMsXG4gICAgICAgICAgICBUaW1lb3V0OiA1LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd0aHJvd3Mgd2hlbiBzZXR0aW5nIEhlYWx0aCBDaGVjayB3aXRoIG5vIGNvbW1hbmRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaGVhbHRoQ2hlY2s6IHtcbiAgICAgICAgY29tbWFuZDogW10sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEhlYWx0aENoZWNrOiB7XG4gICAgICAgICAgICAgIENvbW1hbmQ6IFtdLFxuICAgICAgICAgICAgICBJbnRlcnZhbDogMzAsXG4gICAgICAgICAgICAgIFJldHJpZXM6IDMsXG4gICAgICAgICAgICAgIFRpbWVvdXQ6IDUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG4gICAgfSwgL0F0IGxlYXN0IG9uZSBhcmd1bWVudCBtdXN0IGJlIHN1cHBsaWVkIGZvciBoZWFsdGggY2hlY2sgY29tbWFuZC4vKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gc3BlY2lmeSBIZWFsdGggQ2hlY2sgdmFsdWVzIGluIHNoZWxsIGZvcm0nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcbiAgICBjb25zdCBoY0NvbW1hbmQgPSAnY3VybCBsb2NhbGhvc3Q6ODAwMCc7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaGVhbHRoQ2hlY2s6IHtcbiAgICAgICAgY29tbWFuZDogW2hjQ29tbWFuZF0sXG4gICAgICAgIGludGVydmFsOiBjZGsuRHVyYXRpb24uc2Vjb25kcygyMCksXG4gICAgICAgIHJldHJpZXM6IDUsXG4gICAgICAgIHN0YXJ0UGVyaW9kOiBjZGsuRHVyYXRpb24uc2Vjb25kcygxMCksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEhlYWx0aENoZWNrOiB7XG4gICAgICAgICAgICBDb21tYW5kOiBbJ0NNRC1TSEVMTCcsIGhjQ29tbWFuZF0sXG4gICAgICAgICAgICBJbnRlcnZhbDogMjAsXG4gICAgICAgICAgICBSZXRyaWVzOiA1LFxuICAgICAgICAgICAgVGltZW91dDogNSxcbiAgICAgICAgICAgIFN0YXJ0UGVyaW9kOiAxMCxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY2FuIHNwZWNpZnkgSGVhbHRoIENoZWNrIHZhbHVlcyBpbiBhcnJheSBmb3JtIHN0YXJ0aW5nIHdpdGggQ01ELVNIRUxMJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG4gICAgY29uc3QgaGNDb21tYW5kID0gJ2N1cmwgbG9jYWxob3N0OjgwMDAnO1xuXG4gICAgLy8gV0hFTlxuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignY29udCcsIHtcbiAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCd0ZXN0JyksXG4gICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgIGhlYWx0aENoZWNrOiB7XG4gICAgICAgIGNvbW1hbmQ6IFsnQ01ELVNIRUxMJywgaGNDb21tYW5kXSxcbiAgICAgICAgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDIwKSxcbiAgICAgICAgcmV0cmllczogNSxcbiAgICAgICAgc3RhcnRQZXJpb2Q6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDEwKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICB7XG4gICAgICAgICAgSGVhbHRoQ2hlY2s6IHtcbiAgICAgICAgICAgIENvbW1hbmQ6IFsnQ01ELVNIRUxMJywgaGNDb21tYW5kXSxcbiAgICAgICAgICAgIEludGVydmFsOiAyMCxcbiAgICAgICAgICAgIFJldHJpZXM6IDUsXG4gICAgICAgICAgICBUaW1lb3V0OiA1LFxuICAgICAgICAgICAgU3RhcnRQZXJpb2Q6IDEwLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW4gc3BlY2lmeSBIZWFsdGggQ2hlY2sgdmFsdWVzIGluIGFycmF5IGZvcm0gc3RhcnRpbmcgd2l0aCBDTUQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcbiAgICBjb25zdCBoY0NvbW1hbmQgPSAnY3VybCBsb2NhbGhvc3Q6ODAwMCc7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgaGVhbHRoQ2hlY2s6IHtcbiAgICAgICAgY29tbWFuZDogWydDTUQnLCBoY0NvbW1hbmRdLFxuICAgICAgICBpbnRlcnZhbDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMjApLFxuICAgICAgICByZXRyaWVzOiA1LFxuICAgICAgICBzdGFydFBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMTApLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBIZWFsdGhDaGVjazoge1xuICAgICAgICAgICAgQ29tbWFuZDogWydDTUQnLCBoY0NvbW1hbmRdLFxuICAgICAgICAgICAgSW50ZXJ2YWw6IDIwLFxuICAgICAgICAgICAgUmV0cmllczogNSxcbiAgICAgICAgICAgIFRpbWVvdXQ6IDUsXG4gICAgICAgICAgICBTdGFydFBlcmlvZDogMTAsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NhbiBzcGVjaWZ5IHByaXZhdGUgcmVnaXN0cnkgY3JlZGVudGlhbHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcbiAgICBjb25zdCBteVNlY3JldEFybiA9ICdhcm46YXdzOnNlY3JldHNtYW5hZ2VyOnJlZ2lvbjoxMjM0NTY3ODkwOnNlY3JldDpNeVJlcG9TZWNyZXQtNmY4aGozJztcblxuICAgIGNvbnN0IHJlcG9DcmVkcyA9IHNlY3JldHNtYW5hZ2VyLlNlY3JldC5mcm9tU2VjcmV0QXJuKHN0YWNrLCAnTXlSZXBvU2VjcmV0JywgbXlTZWNyZXRBcm4pO1xuXG4gICAgLy8gV0hFTlxuICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignQ29udGFpbmVyJywge1xuICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3VzZXIteC9teS1hcHAnLCB7XG4gICAgICAgIGNyZWRlbnRpYWxzOiByZXBvQ3JlZHMsXG4gICAgICB9KSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAyMDQ4LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEltYWdlOiAndXNlci14L215LWFwcCcsXG4gICAgICAgICAgUmVwb3NpdG9yeUNyZWRlbnRpYWxzOiB7XG4gICAgICAgICAgICBDcmVkZW50aWFsc1BhcmFtZXRlcjogbXlTZWNyZXRBcm4sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IG15U2VjcmV0QXJuLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdfbGlua0NvbnRhaW5lciB3b3JrcyBwcm9wZXJseSc6IHtcbiAgICAnd2hlbiB0aGUgcHJvcHMgcGFzc2VkIGluIGlzIGFuIGVzc2VudGlhbCBjb250YWluZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgY29udGFpbmVyID0gdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgndGVzdCcpLFxuICAgICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgICAgZXNzZW50aWFsOiB0cnVlLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIHRlc3QuZXF1YWwodGFza0RlZmluaXRpb24uZGVmYXVsdENvbnRhaW5lciwgY29udGFpbmVyKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICd3aGVuIHRoZSBwcm9wcyBwYXNzZWQgaW4gaXMgbm90IGFuIGVzc2VudGlhbCBjb250YWluZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdjb250Jywge1xuICAgICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21SZWdpc3RyeSgndGVzdCcpLFxuICAgICAgICBtZW1vcnlMaW1pdE1pQjogMTAyNCxcbiAgICAgICAgZXNzZW50aWFsOiBmYWxzZSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICB0ZXN0LmVxdWFsKHRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXIsIHVuZGVmaW5lZCk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG5cbiAgJ0NhbiBzcGVjaWZ5IGxpbnV4IHBhcmFtZXRlcnMnOiB7XG4gICAgJ3dpdGggb25seSByZXF1aXJlZCBwcm9wZXJ0aWVzIHNldCwgaXQgY29ycmVjdGx5IHNldHMgZGVmYXVsdCBwcm9wZXJ0aWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAgIGNvbnN0IGxpbnV4UGFyYW1ldGVycyA9IG5ldyBlY3MuTGludXhQYXJhbWV0ZXJzKHN0YWNrLCAnTGludXhQYXJhbWV0ZXJzJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignY29udCcsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICAgIGxpbnV4UGFyYW1ldGVycyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbWFnZTogJ3Rlc3QnLFxuICAgICAgICAgICAgTGludXhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgIENhcGFiaWxpdGllczoge30sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYmVmb3JlIGNhbGxpbmcgYWRkQ29udGFpbmVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAgIGNvbnN0IGxpbnV4UGFyYW1ldGVycyA9IG5ldyBlY3MuTGludXhQYXJhbWV0ZXJzKHN0YWNrLCAnTGludXhQYXJhbWV0ZXJzJywge1xuICAgICAgICBpbml0UHJvY2Vzc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIHNoYXJlZE1lbW9yeVNpemU6IDEwMjQsXG4gICAgICB9KTtcblxuICAgICAgbGludXhQYXJhbWV0ZXJzLmFkZENhcGFiaWxpdGllcyhlY3MuQ2FwYWJpbGl0eS5BTEwpO1xuICAgICAgbGludXhQYXJhbWV0ZXJzLmRyb3BDYXBhYmlsaXRpZXMoZWNzLkNhcGFiaWxpdHkuS0lMTCk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignY29udCcsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICAgIGxpbnV4UGFyYW1ldGVycyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbWFnZTogJ3Rlc3QnLFxuICAgICAgICAgICAgTGludXhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgIENhcGFiaWxpdGllczoge1xuICAgICAgICAgICAgICAgIEFkZDogWydBTEwnXSxcbiAgICAgICAgICAgICAgICBEcm9wOiBbJ0tJTEwnXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgSW5pdFByb2Nlc3NFbmFibGVkOiB0cnVlLFxuICAgICAgICAgICAgICBTaGFyZWRNZW1vcnlTaXplOiAxMDI0LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSkpO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2FmdGVyIGNhbGxpbmcgYWRkQ29udGFpbmVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAgIGNvbnN0IGxpbnV4UGFyYW1ldGVycyA9IG5ldyBlY3MuTGludXhQYXJhbWV0ZXJzKHN0YWNrLCAnTGludXhQYXJhbWV0ZXJzJywge1xuICAgICAgICBpbml0UHJvY2Vzc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIHNoYXJlZE1lbW9yeVNpemU6IDEwMjQsXG4gICAgICB9KTtcblxuICAgICAgbGludXhQYXJhbWV0ZXJzLmFkZENhcGFiaWxpdGllcyhlY3MuQ2FwYWJpbGl0eS5BTEwpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2NvbnQnLCB7XG4gICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCd0ZXN0JyksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgICBsaW51eFBhcmFtZXRlcnMsXG4gICAgICB9KTtcblxuICAgICAgLy8gTXV0YXRlIGxpbnV4UGFyYW1ldGVyIGFmdGVyIGFkZGVkIHRvIGEgY29udGFpbmVyXG4gICAgICBsaW51eFBhcmFtZXRlcnMuZHJvcENhcGFiaWxpdGllcyhlY3MuQ2FwYWJpbGl0eS5TRVRVSUQpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbWFnZTogJ3Rlc3QnLFxuICAgICAgICAgICAgTGludXhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgIENhcGFiaWxpdGllczoge1xuICAgICAgICAgICAgICAgIEFkZDogWydBTEwnXSxcbiAgICAgICAgICAgICAgICBEcm9wOiBbJ1NFVFVJRCddLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBJbml0UHJvY2Vzc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICAgIFNoYXJlZE1lbW9yeVNpemU6IDEwMjQsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnd2l0aCBvbmUgb3IgbW9yZSBob3N0IGRldmljZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IGVjcy5FYzJUYXNrRGVmaW5pdGlvbihzdGFjaywgJ1Rhc2tEZWYnKTtcblxuICAgICAgY29uc3QgbGludXhQYXJhbWV0ZXJzID0gbmV3IGVjcy5MaW51eFBhcmFtZXRlcnMoc3RhY2ssICdMaW51eFBhcmFtZXRlcnMnLCB7XG4gICAgICAgIGluaXRQcm9jZXNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgc2hhcmVkTWVtb3J5U2l6ZTogMTAyNCxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBsaW51eFBhcmFtZXRlcnMuYWRkRGV2aWNlcyh7XG4gICAgICAgIGhvc3RQYXRoOiAnYS9iL2MnLFxuICAgICAgfSk7XG5cbiAgICAgIHRhc2tEZWZpbml0aW9uLmFkZENvbnRhaW5lcignY29udCcsIHtcbiAgICAgICAgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZS5mcm9tUmVnaXN0cnkoJ3Rlc3QnKSxcbiAgICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICAgIGxpbnV4UGFyYW1ldGVycyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBJbWFnZTogJ3Rlc3QnLFxuICAgICAgICAgICAgTGludXhQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgIERldmljZXM6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBIb3N0UGF0aDogJ2EvYi9jJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBJbml0UHJvY2Vzc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICAgIFNoYXJlZE1lbW9yeVNpemU6IDEwMjQsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnd2l0aCB0aGUgdG1wZnMgbW91bnQgZm9yIGEgY29udGFpbmVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAgIGNvbnN0IGxpbnV4UGFyYW1ldGVycyA9IG5ldyBlY3MuTGludXhQYXJhbWV0ZXJzKHN0YWNrLCAnTGludXhQYXJhbWV0ZXJzJywge1xuICAgICAgICBpbml0UHJvY2Vzc0VuYWJsZWQ6IHRydWUsXG4gICAgICAgIHNoYXJlZE1lbW9yeVNpemU6IDEwMjQsXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbGludXhQYXJhbWV0ZXJzLmFkZFRtcGZzKHtcbiAgICAgICAgY29udGFpbmVyUGF0aDogJ2EvYi9jJyxcbiAgICAgICAgc2l6ZTogMTAyNCxcbiAgICAgIH0pO1xuXG4gICAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2NvbnQnLCB7XG4gICAgICAgIGltYWdlOiBlY3MuQ29udGFpbmVySW1hZ2UuZnJvbVJlZ2lzdHJ5KCd0ZXN0JyksXG4gICAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgICAgICBsaW51eFBhcmFtZXRlcnMsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgSW1hZ2U6ICd0ZXN0JyxcbiAgICAgICAgICAgIExpbnV4UGFyYW1ldGVyczoge1xuICAgICAgICAgICAgICBUbXBmczogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIENvbnRhaW5lclBhdGg6ICdhL2IvYycsXG4gICAgICAgICAgICAgICAgICBTaXplOiAxMDI0LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIEluaXRQcm9jZXNzRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgICAgICAgU2hhcmVkTWVtb3J5U2l6ZTogMTAyNCxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAnY2FuIHVzZSBhIERvY2tlckltYWdlQXNzZXQgZGlyZWN0bHkgZm9yIGEgY29udGFpbmVyIGltYWdlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG4gICAgY29uc3QgYXNzZXQgPSBuZXcgZWNyX2Fzc2V0cy5Eb2NrZXJJbWFnZUFzc2V0KHN0YWNrLCAnTXlEb2NrZXJJbWFnZScsIHtcbiAgICAgIGRpcmVjdG9yeTogcGF0aC5qb2luKF9fZGlybmFtZSwgJ2RlbW8taW1hZ2UnKSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICB0YXNrRGVmaW5pdGlvbi5hZGRDb250YWluZXIoJ2RlZmF1bHQnLCB7XG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21Eb2NrZXJJbWFnZUFzc2V0KGFzc2V0KSxcbiAgICAgIG1lbW9yeUxpbWl0TWlCOiAxMDI0LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkVDUzo6VGFza0RlZmluaXRpb24nLCB7XG4gICAgICBDb250YWluZXJEZWZpbml0aW9uczogW1xuICAgICAgICB7XG4gICAgICAgICAgRXNzZW50aWFsOiB0cnVlLFxuICAgICAgICAgIEltYWdlOiB7XG4gICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSxcbiAgICAgICAgICAgICAgICAnLmRrci5lY3IuJyxcbiAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICAgICAgICcuJyxcbiAgICAgICAgICAgICAgICB7IFJlZjogJ0FXUzo6VVJMU3VmZml4JyB9LFxuICAgICAgICAgICAgICAgICcvYXdzLWNkay9hc3NldHM6YmFhMmQ2ZWIyYTE3Yzc1NDI0ZGY2MzFjOGM3MGZmMzlmMmQ1ZjNiZWU4YjllMWExMDllZTI0Y2ExNzMwMDU0MCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBdLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgTWVtb3J5OiAxMDI0LFxuICAgICAgICAgIE5hbWU6ICdkZWZhdWx0JyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgICAgICAgJ2VjcjpHZXREb3dubG9hZFVybEZvckxheWVyJyxcbiAgICAgICAgICAgICAgJ2VjcjpCYXRjaEdldEltYWdlJyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgJycsXG4gICAgICAgICAgICAgICAgWyAnYXJuOicsIHsgUmVmOiAnQVdTOjpQYXJ0aXRpb24nIH0sICc6ZWNyOicsIHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sICc6JywgeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSwgJzpyZXBvc2l0b3J5L2F3cy1jZGsvYXNzZXRzJyBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICB9LFxuICAgIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZG9ja2VyIGltYWdlIGFzc2V0IG9wdGlvbnMgY2FuIGJlIHVzZWQgd2hlbiB1c2luZyBjb250YWluZXIgaW1hZ2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBjZGsuQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ015U3RhY2snKTtcbiAgICBjb25zdCB0YXNrRGVmaW5pdGlvbiA9IG5ldyBlY3MuRWMyVGFza0RlZmluaXRpb24oc3RhY2ssICdUYXNrRGVmJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgdGFza0RlZmluaXRpb24uYWRkQ29udGFpbmVyKCdkZWZhdWx0Jywge1xuICAgICAgbWVtb3J5TGltaXRNaUI6IDEwMjQsXG4gICAgICBpbWFnZTogZWNzLkNvbnRhaW5lckltYWdlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnZGVtby1pbWFnZScpLCB7XG4gICAgICAgIGZpbGU6ICdpbmRleC5weScsIC8vIGp1c3QgYmVjYXVzZSBpdCdzIHRoZXJlIGFscmVhZHlcbiAgICAgICAgdGFyZ2V0OiAnYnVpbGQtdGFyZ2V0JyxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzbSA9IGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzbS5nZXRTdGFja0FydGlmYWN0KHN0YWNrLmFydGlmYWN0SWQpLmFzc2V0c1swXSwge1xuICAgICAgcmVwb3NpdG9yeU5hbWU6ICdhd3MtY2RrL2Fzc2V0cycsXG4gICAgICBpbWFnZVRhZzogJ2Y5MDE0ZDFkZjdjOGY1YTVlN2FiYWYxOGViNWJjODk1ZTgyZjhiMDZlZWVkNmY3NWE0MGNmMWJjMmE3ODk1NWEnLFxuICAgICAgaWQ6ICdmOTAxNGQxZGY3YzhmNWE1ZTdhYmFmMThlYjViYzg5NWU4MmY4YjA2ZWVlZDZmNzVhNDBjZjFiYzJhNzg5NTVhJyxcbiAgICAgIHBhY2thZ2luZzogJ2NvbnRhaW5lci1pbWFnZScsXG4gICAgICBwYXRoOiAnYXNzZXQuZjkwMTRkMWRmN2M4ZjVhNWU3YWJhZjE4ZWI1YmM4OTVlODJmOGIwNmVlZWQ2Zjc1YTQwY2YxYmMyYTc4OTU1YScsXG4gICAgICBzb3VyY2VIYXNoOiAnZjkwMTRkMWRmN2M4ZjVhNWU3YWJhZjE4ZWI1YmM4OTVlODJmOGIwNmVlZWQ2Zjc1YTQwY2YxYmMyYTc4OTU1YScsXG4gICAgICB0YXJnZXQ6ICdidWlsZC10YXJnZXQnLFxuICAgICAgZmlsZTogJ2luZGV4LnB5JyxcbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG4iXX0=