"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cloudmap = require("@aws-cdk/aws-servicediscovery");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
module.exports = {
    'When creating an ECS Cluster': {
        'with no properties set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const cluster = new ecs.Cluster(stack, 'EcsCluster');
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/EcsCluster/Vpc',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
                ImageId: {
                    Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
                },
                InstanceType: 't2.micro',
                IamInstanceProfile: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3',
                },
                SecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231',
                            'GroupId',
                        ],
                    },
                ],
                UserData: {
                    'Fn::Base64': {
                        'Fn::Join': [
                            '',
                            [
                                '#!/bin/bash\necho ECS_CLUSTER=',
                                {
                                    Ref: 'EcsCluster97242B84',
                                },
                                // eslint-disable-next-line max-len
                                ' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config',
                            ],
                        ],
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
                MaxSize: '1',
                MinSize: '1',
                LaunchConfigurationName: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1',
                },
                Tags: [
                    {
                        Key: 'Name',
                        PropagateAtLaunch: true,
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: 'EcsClusterVpcPrivateSubnet1SubnetFAB0E487',
                    },
                    {
                        Ref: 'EcsClusterVpcPrivateSubnet2SubnetC2B7B1BA',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Default/EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VpcId: {
                    Ref: 'EcsClusterVpc779914AB',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: { 'Fn::Join': ['', ['ec2.', { Ref: 'AWS::URLSuffix' }]] },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'ecs:DeregisterContainerInstance',
                                'ecs:RegisterContainerInstance',
                                'ecs:Submit*',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'EcsCluster97242B84',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ecs:Poll',
                                'ecs:StartTelemetrySession',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                            Condition: {
                                ArnEquals: {
                                    'ecs:cluster': {
                                        'Fn::GetAtt': [
                                            'EcsCluster97242B84',
                                            'Arn',
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            Action: [
                                'ecs:DiscoverPollEndpoint',
                                'ecr:GetAuthorizationToken',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'with only vpc set, it correctly sets default properties'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
            });
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/MyVpc',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
                ImageId: {
                    Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
                },
                InstanceType: 't2.micro',
                IamInstanceProfile: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3',
                },
                SecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231',
                            'GroupId',
                        ],
                    },
                ],
                UserData: {
                    'Fn::Base64': {
                        'Fn::Join': [
                            '',
                            [
                                '#!/bin/bash\necho ECS_CLUSTER=',
                                {
                                    Ref: 'EcsCluster97242B84',
                                },
                                // eslint-disable-next-line max-len
                                ' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config',
                            ],
                        ],
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
                MaxSize: '1',
                MinSize: '1',
                LaunchConfigurationName: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1',
                },
                Tags: [
                    {
                        Key: 'Name',
                        PropagateAtLaunch: true,
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                    },
                    {
                        Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Default/EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: { 'Fn::Join': ['', ['ec2.', { Ref: 'AWS::URLSuffix' }]] },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'ecs:DeregisterContainerInstance',
                                'ecs:RegisterContainerInstance',
                                'ecs:Submit*',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'EcsCluster97242B84',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ecs:Poll',
                                'ecs:StartTelemetrySession',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                            Condition: {
                                ArnEquals: {
                                    'ecs:cluster': {
                                        'Fn::GetAtt': [
                                            'EcsCluster97242B84',
                                            'Arn',
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            Action: [
                                'ecs:DiscoverPollEndpoint',
                                'ecr:GetAuthorizationToken',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
        'multiple clusters with default capacity'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            // WHEN
            for (let i = 0; i < 2; i++) {
                const cluster = new ecs.Cluster(stack, `EcsCluster${i}`, { vpc });
                cluster.addCapacity('MyCapacity', {
                    instanceType: new ec2.InstanceType('m3.medium'),
                });
            }
            test.done();
        },
        'lifecycle hook is automatically added'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            const cluster = new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
            });
            // WHEN
            cluster.addCapacity('DefaultAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LifecycleHook', {
                AutoScalingGroupName: { Ref: 'EcsClusterDefaultAutoScalingGroupASGC1A785DB' },
                LifecycleTransition: 'autoscaling:EC2_INSTANCE_TERMINATING',
                DefaultResult: 'CONTINUE',
                HeartbeatTimeout: 300,
                NotificationTargetARN: { Ref: 'EcsClusterDefaultAutoScalingGroupLifecycleHookDrainHookTopicACD2D4A4' },
                RoleARN: { 'Fn::GetAtt': ['EcsClusterDefaultAutoScalingGroupLifecycleHookDrainHookRoleA38EC83B', 'Arn'] },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Timeout: 310,
                Environment: {
                    Variables: {
                        CLUSTER: {
                            Ref: 'EcsCluster97242B84',
                        },
                    },
                },
                Handler: 'index.lambda_handler',
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'ec2:DescribeInstances',
                                'ec2:DescribeInstanceAttribute',
                                'ec2:DescribeInstanceStatus',
                                'ec2:DescribeHosts',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                        {
                            Action: 'autoscaling:CompleteLifecycleAction',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':autoscaling:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':autoScalingGroup:*:autoScalingGroupName/',
                                        {
                                            Ref: 'EcsClusterDefaultAutoScalingGroupASGC1A785DB',
                                        },
                                    ],
                                ],
                            },
                        },
                        {
                            Action: [
                                'ecs:DescribeContainerInstances',
                                'ecs:DescribeTasks',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                            Condition: {
                                ArnEquals: {
                                    'ecs:cluster': {
                                        'Fn::GetAtt': [
                                            'EcsCluster97242B84',
                                            'Arn',
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            Action: [
                                'ecs:ListContainerInstances',
                                'ecs:SubmitContainerStateChange',
                                'ecs:SubmitTaskStateChange',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'EcsCluster97242B84',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ecs:UpdateContainerInstancesState',
                                'ecs:ListTasks',
                            ],
                            Condition: {
                                ArnEquals: {
                                    'ecs:cluster': {
                                        'Fn::GetAtt': [
                                            'EcsCluster97242B84',
                                            'Arn',
                                        ],
                                    },
                                },
                            },
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'EcsClusterDefaultAutoScalingGroupDrainECSHookFunctionServiceRoleDefaultPolicyA45BF396',
                Roles: [
                    {
                        Ref: 'EcsClusterDefaultAutoScalingGroupDrainECSHookFunctionServiceRole94543EDA',
                    },
                ],
            }));
            test.done();
        },
        'with capacity and cloudmap namespace properties set'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'MyVpc', {});
            new ecs.Cluster(stack, 'EcsCluster', {
                vpc,
                capacity: {
                    instanceType: new ec2.InstanceType('t2.micro'),
                },
                defaultCloudMapNamespace: {
                    name: 'foo.com',
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PrivateDnsNamespace', {
                Name: 'foo.com',
                Vpc: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '10.0.0.0/16',
                EnableDnsHostnames: true,
                EnableDnsSupport: true,
                InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/MyVpc',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
                ImageId: {
                    Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
                },
                InstanceType: 't2.micro',
                IamInstanceProfile: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3',
                },
                SecurityGroups: [
                    {
                        'Fn::GetAtt': [
                            'EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231',
                            'GroupId',
                        ],
                    },
                ],
                UserData: {
                    'Fn::Base64': {
                        'Fn::Join': [
                            '',
                            [
                                '#!/bin/bash\necho ECS_CLUSTER=',
                                {
                                    Ref: 'EcsCluster97242B84',
                                },
                                // eslint-disable-next-line max-len
                                ' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config',
                            ],
                        ],
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
                MaxSize: '1',
                MinSize: '1',
                LaunchConfigurationName: {
                    Ref: 'EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1',
                },
                Tags: [
                    {
                        Key: 'Name',
                        PropagateAtLaunch: true,
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VPCZoneIdentifier: [
                    {
                        Ref: 'MyVpcPrivateSubnet1Subnet5057CF7E',
                    },
                    {
                        Ref: 'MyVpcPrivateSubnet2Subnet0040C983',
                    },
                ],
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'Default/EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup',
                SecurityGroupEgress: [
                    {
                        CidrIp: '0.0.0.0/0',
                        Description: 'Allow all outbound traffic by default',
                        IpProtocol: '-1',
                    },
                ],
                Tags: [
                    {
                        Key: 'Name',
                        Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                    },
                ],
                VpcId: {
                    Ref: 'MyVpcF9F0CA6F',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role', {
                AssumeRolePolicyDocument: {
                    Statement: [
                        {
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: { 'Fn::Join': ['', ['ec2.', { Ref: 'AWS::URLSuffix' }]] },
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'ecs:DeregisterContainerInstance',
                                'ecs:RegisterContainerInstance',
                                'ecs:Submit*',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'EcsCluster97242B84',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: [
                                'ecs:Poll',
                                'ecs:StartTelemetrySession',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                            Condition: {
                                ArnEquals: {
                                    'ecs:cluster': {
                                        'Fn::GetAtt': [
                                            'EcsCluster97242B84',
                                            'Arn',
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            Action: [
                                'ecs:DiscoverPollEndpoint',
                                'ecr:GetAuthorizationToken',
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
    },
    'allows specifying instance type'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('m3.large'),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'm3.large',
        }));
        test.done();
    },
    'allows specifying cluster size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            desiredCapacity: 3,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            MaxSize: '3',
        }));
        test.done();
    },
    /*
     * TODO:v2.0.0 BEGINNING OF OBSOLETE BLOCK
     */
    'allows specifying special HW AMI Type'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: new ecs.EcsOptimizedAmi({
                hardwareType: ecs.AmiHardwareType.GPU,
            }),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            ImageId: {
                Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
            },
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: 'AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>',
                Default: '/aws/service/ecs/optimized-ami/amazon-linux-2/gpu/recommended/image_id',
            },
        });
        test.done();
    },
    'errors if amazon linux given with special HW type'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('GpuAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
                    hardwareType: ecs.AmiHardwareType.GPU,
                }),
            });
        }, /Amazon Linux does not support special hardware type/);
        test.done();
    },
    'allows specifying windows image'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('WindowsAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: new ecs.EcsOptimizedAmi({
                windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
            }),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiwindowsserver2019englishfullrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: 'AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>',
                Default: '/aws/service/ecs/optimized-ami/windows_server/2019/english/full/recommended/image_id',
            },
        });
        test.done();
    },
    'errors if windows given with special HW type'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('WindowsGpuAutoScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
                    hardwareType: ecs.AmiHardwareType.GPU,
                }),
            });
        }, /Windows Server does not support special hardware type/);
        test.done();
    },
    'errors if windowsVersion and linux generation are set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.throws(() => {
            cluster.addCapacity('WindowsScalingGroup', {
                instanceType: new ec2.InstanceType('t2.micro'),
                machineImage: new ecs.EcsOptimizedAmi({
                    windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
                }),
            });
        }, /"windowsVersion" and Linux image "generation" cannot be both set/);
        test.done();
    },
    'allows returning the correct image for windows for EcsOptimizedAmi'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            windowsVersion: ecs.WindowsOptimizedVersion.SERVER_2019,
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.WINDOWS);
        test.done();
    },
    'allows returning the correct image for linux for EcsOptimizedAmi'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX,
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    'allows returning the correct image for linux 2 for EcsOptimizedAmi'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const ami = new ecs.EcsOptimizedAmi({
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
        test.equal(ami.getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    'allows returning the correct image for linux for EcsOptimizedImage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.amazonLinux().getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    'allows returning the correct image for linux 2 for EcsOptimizedImage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.amazonLinux2().getImage(stack).osType, ec2.OperatingSystemType.LINUX);
        test.done();
    },
    'allows returning the correct image for windows for EcsOptimizedImage'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        test.equal(ecs.EcsOptimizedImage.windows(ecs.WindowsOptimizedVersion.SERVER_2019).getImage(stack).osType, ec2.OperatingSystemType.WINDOWS);
        test.done();
    },
    /*
     * TODO:v2.0.0 END OF OBSOLETE BLOCK
     */
    'allows specifying special HW AMI Type v2'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.GPU),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            ImageId: {
                Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
            },
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinux2gpurecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: 'AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>',
                Default: '/aws/service/ecs/optimized-ami/amazon-linux-2/gpu/recommended/image_id',
            },
        });
        test.done();
    },
    'allows specifying Amazon Linux v1 AMI'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('GpuAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.amazonLinux(),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            ImageId: {
                Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinuxrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
            },
        }));
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiamazonlinuxrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: 'AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>',
                Default: '/aws/service/ecs/optimized-ami/amazon-linux/recommended/image_id',
            },
        });
        test.done();
    },
    'allows specifying windows image v2'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('WindowsAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            machineImage: ecs.EcsOptimizedImage.windows(ecs.WindowsOptimizedVersion.SERVER_2019),
        });
        // THEN
        const assembly = app.synth();
        const template = assembly.getStackByName(stack.stackName).template;
        test.deepEqual(template.Parameters, {
            SsmParameterValueawsserviceecsoptimizedamiwindowsserver2019englishfullrecommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter: {
                Type: 'AWS::SSM::Parameter::Value<AWS::EC2::Image::Id>',
                Default: '/aws/service/ecs/optimized-ami/windows_server/2019/english/full/recommended/image_id',
            },
        });
        test.done();
    },
    'allows specifying spot fleet'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            spotPrice: '0.31',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            SpotPrice: '0.31',
        }));
        test.done();
    },
    'allows specifying drain time'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            taskDrainTime: cdk.Duration.minutes(1),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LifecycleHook', {
            HeartbeatTimeout: 60,
        }));
        test.done();
    },
    'allows specifying automated spot draining'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('c5.xlarge'),
            spotPrice: '0.0735',
            spotInstanceDraining: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\necho ECS_CLUSTER=',
                            {
                                Ref: 'EcsCluster97242B84',
                            },
                            ' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config\necho ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'allows containers access to instance metadata service'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            canContainersAccessInstanceRole: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\necho ECS_CLUSTER=',
                            {
                                Ref: 'EcsCluster97242B84',
                            },
                            ' >> /etc/ecs/ecs.config',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'allows adding default service discovery namespace'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: 'foo.com',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PrivateDnsNamespace', {
            Name: 'foo.com',
            Vpc: {
                Ref: 'MyVpcF9F0CA6F',
            },
        }));
        test.done();
    },
    'allows adding public service discovery namespace'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: 'foo.com',
            type: cloudmap.NamespaceType.DNS_PUBLIC,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ServiceDiscovery::PublicDnsNamespace', {
            Name: 'foo.com',
        }));
        test.equal(cluster.defaultCloudMapNamespace.type, cloudmap.NamespaceType.DNS_PUBLIC);
        test.done();
    },
    'throws if default service discovery namespace added more than once'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
        });
        // WHEN
        cluster.addDefaultCloudMapNamespace({
            name: 'foo.com',
        });
        // THEN
        test.throws(() => {
            cluster.addDefaultCloudMapNamespace({
                name: 'foo.com',
            });
        }, /Can only add default namespace once./);
        test.done();
    },
    'export/import of a cluster with a namespace'(test) {
        // GIVEN
        const stack1 = new cdk.Stack();
        const vpc1 = new ec2.Vpc(stack1, 'Vpc');
        const cluster1 = new ecs.Cluster(stack1, 'Cluster', { vpc: vpc1 });
        cluster1.addDefaultCloudMapNamespace({
            name: 'hello.com',
        });
        const stack2 = new cdk.Stack();
        // WHEN
        const cluster2 = ecs.Cluster.fromClusterAttributes(stack2, 'Cluster', {
            vpc: vpc1,
            securityGroups: cluster1.connections.securityGroups,
            defaultCloudMapNamespace: cloudmap.PrivateDnsNamespace.fromPrivateDnsNamespaceAttributes(stack2, 'ns', {
                namespaceId: 'import-namespace-id',
                namespaceArn: 'import-namespace-arn',
                namespaceName: 'import-namespace-name',
            }),
            clusterName: 'cluster-name',
        });
        // THEN
        test.equal(cluster2.defaultCloudMapNamespace.type, cloudmap.NamespaceType.DNS_PRIVATE);
        test.deepEqual(stack2.resolve(cluster2.defaultCloudMapNamespace.namespaceId), 'import-namespace-id');
        // Can retrieve subnets from VPC - will throw 'There are no 'Private' subnets in this VPC. Use a different VPC subnet selection.' if broken.
        cluster2.vpc.selectSubnets();
        test.done();
    },
    'imported cluster with imported security groups honors allowAllOutbound'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Vpc');
        const importedSg1 = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-1', { allowAllOutbound: false });
        const importedSg2 = ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-2');
        const cluster = ecs.Cluster.fromClusterAttributes(stack, 'Cluster', {
            clusterName: 'cluster-name',
            securityGroups: [importedSg1, importedSg2],
            vpc,
        });
        // WHEN
        cluster.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'sg-1',
        }));
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroupEgress', 1));
        test.done();
    },
    'Metric'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyVpc', {});
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // THEN
        test.deepEqual(stack.resolve(cluster.metricCpuReservation()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'CPUReservation',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.deepEqual(stack.resolve(cluster.metricMemoryReservation()), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'MemoryReservation',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.deepEqual(stack.resolve(cluster.metric('myMetric')), {
            dimensions: {
                ClusterName: { Ref: 'EcsCluster97242B84' },
            },
            namespace: 'AWS/ECS',
            metricName: 'myMetric',
            period: cdk.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'ASG with a public VPC without NAT Gateways'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'MyPublicVpc', {
            natGateways: 0,
            subnetConfiguration: [
                { cidrMask: 24, name: 'ingress', subnetType: ec2.SubnetType.PUBLIC },
            ],
        });
        const cluster = new ecs.Cluster(stack, 'EcsCluster', { vpc });
        // WHEN
        cluster.addCapacity('DefaultAutoScalingGroup', {
            instanceType: new ec2.InstanceType('t2.micro'),
            associatePublicIpAddress: true,
            vpcSubnets: {
                onePerAz: true,
                subnetType: ec2.SubnetType.PUBLIC,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
            CidrBlock: '10.0.0.0/16',
            EnableDnsHostnames: true,
            EnableDnsSupport: true,
            InstanceTenancy: ec2.DefaultInstanceTenancy.DEFAULT,
            Tags: [
                {
                    Key: 'Name',
                    Value: 'Default/MyPublicVpc',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::LaunchConfiguration', {
            ImageId: {
                Ref: 'SsmParameterValueawsserviceecsoptimizedamiamazonlinux2recommendedimageidC96584B6F00A464EAD1953AFF4B05118Parameter',
            },
            InstanceType: 't2.micro',
            AssociatePublicIpAddress: true,
            IamInstanceProfile: {
                Ref: 'EcsClusterDefaultAutoScalingGroupInstanceProfile2CE606B3',
            },
            SecurityGroups: [
                {
                    'Fn::GetAtt': [
                        'EcsClusterDefaultAutoScalingGroupInstanceSecurityGroup912E1231',
                        'GroupId',
                    ],
                },
            ],
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            '#!/bin/bash\necho ECS_CLUSTER=',
                            {
                                Ref: 'EcsCluster97242B84',
                            },
                            // eslint-disable-next-line max-len
                            ' >> /etc/ecs/ecs.config\nsudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP\nsudo service iptables save\necho ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config',
                        ],
                    ],
                },
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            MaxSize: '1',
            MinSize: '1',
            LaunchConfigurationName: {
                Ref: 'EcsClusterDefaultAutoScalingGroupLaunchConfigB7E376C1',
            },
            Tags: [
                {
                    Key: 'Name',
                    PropagateAtLaunch: true,
                    Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                },
            ],
            VPCZoneIdentifier: [
                {
                    Ref: 'MyPublicVpcingressSubnet1Subnet9191044C',
                },
                {
                    Ref: 'MyPublicVpcingressSubnet2SubnetD2F2E034',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            GroupDescription: 'Default/EcsCluster/DefaultAutoScalingGroup/InstanceSecurityGroup',
            SecurityGroupEgress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Allow all outbound traffic by default',
                    IpProtocol: '-1',
                },
            ],
            Tags: [
                {
                    Key: 'Name',
                    Value: 'Default/EcsCluster/DefaultAutoScalingGroup',
                },
            ],
            VpcId: {
                Ref: 'MyPublicVpcA2BF6CDA',
            },
        }));
        // THEN
        test.done();
    },
    'enable container insights'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        new ecs.Cluster(stack, 'EcsCluster', { containerInsights: true });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster', {
            ClusterSettings: [
                {
                    Name: 'containerInsights',
                    Value: 'enabled',
                },
            ],
        }, assert_1.ResourcePart.Properties));
        test.done();
    },
    'default container insights undefined'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'test');
        new ecs.Cluster(stack, 'EcsCluster');
        // THEN
        const assembly = app.synth();
        const stackAssembly = assembly.getStackByName(stack.stackName);
        const template = stackAssembly.template;
        test.equal(template.Resources.EcsCluster97242B84.Properties === undefined ||
            template.Resources.EcsCluster97242B84.Properties.ClusterSettings === undefined, true, 'ClusterSettings should not be defined');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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