"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const ecs = require("../lib");
let stack;
let td;
const image = ecs.ContainerImage.fromRegistry('test-image');
module.exports = {
    'setUp'(cb) {
        stack = new cdk.Stack();
        td = new ecs.Ec2TaskDefinition(stack, 'TaskDefinition');
        cb();
    },
    'create a firelens log driver with default options'(test) {
        // WHEN
        td.addContainer('Container', {
            image,
            logging: ecs.LogDrivers.firelens({}),
            memoryLimitMiB: 128,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awsfirelens',
                    },
                },
                {
                    Essential: true,
                    FirelensConfiguration: {
                        Type: 'fluentbit',
                    },
                },
            ],
        }));
        test.done();
    },
    'create a firelens log driver to route logs to CloudWatch Logs with Fluent Bit'(test) {
        // WHEN
        td.addContainer('Container', {
            image,
            logging: ecs.LogDrivers.firelens({
                options: {
                    Name: 'cloudwatch',
                    region: 'us-west-2',
                    log_group_name: 'firelens-fluent-bit',
                    auto_create_group: 'true',
                    log_stream_prefix: 'from-fluent-bit',
                },
            }),
            memoryLimitMiB: 128,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awsfirelens',
                        Options: {
                            Name: 'cloudwatch',
                            region: 'us-west-2',
                            log_group_name: 'firelens-fluent-bit',
                            auto_create_group: 'true',
                            log_stream_prefix: 'from-fluent-bit',
                        },
                    },
                },
                {
                    Essential: true,
                    FirelensConfiguration: {
                        Type: 'fluentbit',
                    },
                },
            ],
        }));
        test.done();
    },
    'create a firelens log driver to route logs to kinesis firehose Logs with Fluent Bit'(test) {
        // WHEN
        td.addContainer('Container', {
            image,
            logging: ecs.LogDrivers.firelens({
                options: {
                    Name: 'firehose',
                    region: 'us-west-2',
                    delivery_stream: 'my-stream',
                },
            }),
            memoryLimitMiB: 128,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                {
                    LogConfiguration: {
                        LogDriver: 'awsfirelens',
                        Options: {
                            Name: 'firehose',
                            region: 'us-west-2',
                            delivery_stream: 'my-stream',
                        },
                    },
                },
                {
                    Essential: true,
                    FirelensConfiguration: {
                        Type: 'fluentbit',
                    },
                },
            ],
        }));
        test.done();
    },
    'Firelens Configuration': {
        'fluentd log router container'(test) {
            // GIVEN
            td.addFirelensLogRouter('log_router', {
                image: ecs.ContainerImage.fromRegistry('fluent/fluentd'),
                firelensConfig: {
                    type: ecs.FirelensLogRouterType.FLUENTD,
                },
                memoryReservationMiB: 50,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                        Image: 'fluent/fluentd',
                        MemoryReservation: 50,
                        Name: 'log_router',
                        FirelensConfiguration: {
                            Type: 'fluentd',
                        },
                    },
                ],
            }));
            test.done();
        },
        'fluent-bit log router container with options'(test) {
            // GIVEN
            const stack2 = new cdk.Stack(undefined, 'Stack2', { env: { region: 'us-east-1' } });
            const td2 = new ecs.Ec2TaskDefinition(stack2, 'TaskDefinition');
            td2.addFirelensLogRouter('log_router', {
                image: ecs.obtainDefaultFluentBitECRImage(td2, undefined, '2.1.0'),
                firelensConfig: {
                    type: ecs.FirelensLogRouterType.FLUENTBIT,
                    options: {
                        enableECSLogMetadata: false,
                        configFileValue: 'arn:aws:s3:::mybucket/fluent.conf',
                    },
                },
                logging: new ecs.AwsLogDriver({ streamPrefix: 'firelens' }),
                memoryReservationMiB: 50,
            });
            // THEN
            assert_1.expect(stack2).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                        MemoryReservation: 50,
                        Name: 'log_router',
                        FirelensConfiguration: {
                            Type: 'fluentbit',
                            Options: {
                                'enable-ecs-log-metadata': 'false',
                                'config-file-type': 's3',
                                'config-file-value': 'arn:aws:s3:::mybucket/fluent.conf',
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
        'fluent-bit log router with file config type'(test) {
            // GIVEN
            td.addFirelensLogRouter('log_router', {
                image: ecs.obtainDefaultFluentBitECRImage(td, undefined, '2.1.0'),
                firelensConfig: {
                    type: ecs.FirelensLogRouterType.FLUENTBIT,
                    options: {
                        enableECSLogMetadata: false,
                        configFileType: ecs.FirelensConfigFileType.FILE,
                        configFileValue: '/my/working/dir/firelens/config',
                    },
                },
                logging: new ecs.AwsLogDriver({ streamPrefix: 'firelens' }),
                memoryReservationMiB: 50,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                        MemoryReservation: 50,
                        Name: 'log_router',
                        FirelensConfiguration: {
                            Type: 'fluentbit',
                            Options: {
                                'enable-ecs-log-metadata': 'false',
                                'config-file-type': 'file',
                                'config-file-value': '/my/working/dir/firelens/config',
                            },
                        },
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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