"""Templates for OPENCLI.md generation.

These templates provide structure for the AI agent's output,
but the agent has full flexibility to adapt them based on
the specific project being analyzed.
"""

from datetime import datetime
from typing import Optional


class OCLITemplate:
    """Template generator for OPENCLI.md files."""

    @staticmethod
    def generate_header(project_name: str) -> str:
        """Generate header section.

        Args:
            project_name: Name of the project

        Returns:
            Markdown header
        """
        date = datetime.now().strftime("%Y-%m-%d")
        return f"""# {project_name}

> Generated by SWE-CLI /init on {date}

"""

    @staticmethod
    def generate_overview(
        language: Optional[str] = None,
        project_type: Optional[str] = None,
        description: Optional[str] = None,
    ) -> str:
        """Generate project overview section.

        Args:
            language: Primary programming language
            project_type: Type of project (e.g., "REST API", "CLI Tool")
            description: Brief description

        Returns:
            Markdown overview section
        """
        lines = ["## Project Overview\n"]

        if language:
            lines.append(f"**Language**: {language}")
        if project_type:
            lines.append(f"**Type**: {project_type}")
        if description:
            lines.append(f"**Description**: {description}")

        lines.append("")  # Empty line
        return "\n".join(lines)

    @staticmethod
    def generate_structure(tree_output: Optional[str] = None) -> str:
        """Generate folder structure section.

        Args:
            tree_output: Output from tree command

        Returns:
            Markdown structure section
        """
        if not tree_output:
            return ""

        return f"""## Folder Structure

```
{tree_output}
```

"""

    @staticmethod
    def generate_dependencies(
        dependencies: Optional[dict[str, list[str]]] = None
    ) -> str:
        """Generate dependencies section.

        Args:
            dependencies: Dictionary of dependency categories to lists

        Returns:
            Markdown dependencies section
        """
        if not dependencies:
            return ""

        lines = ["## Dependencies\n"]

        for category, deps in dependencies.items():
            lines.append(f"### {category}")
            for dep in deps:
                lines.append(f"- {dep}")
            lines.append("")

        return "\n".join(lines)

    @staticmethod
    def generate_architecture(
        pattern: Optional[str] = None,
        components: Optional[dict[str, str]] = None,
    ) -> str:
        """Generate architecture section.

        Args:
            pattern: Architecture pattern (e.g., "MVC", "Microservices")
            components: Dictionary of component names to descriptions

        Returns:
            Markdown architecture section
        """
        if not pattern and not components:
            return ""

        lines = ["## Architecture\n"]

        if pattern:
            lines.append(f"**Pattern**: {pattern}\n")

        if components:
            lines.append("**Key Components**:")
            for name, desc in components.items():
                lines.append(f"- **{name}**: {desc}")
            lines.append("")

        return "\n".join(lines)

    @staticmethod
    def generate_key_files(files: Optional[dict[str, str]] = None) -> str:
        """Generate key files section.

        Args:
            files: Dictionary of file roles to file paths

        Returns:
            Markdown key files section
        """
        if not files:
            return ""

        lines = ["## Key Files\n"]

        for role, filepath in files.items():
            lines.append(f"- **{role}**: `{filepath}`")

        lines.append("")
        return "\n".join(lines)

    @staticmethod
    def generate_workflow(
        commands: Optional[dict[str, str]] = None
    ) -> str:
        """Generate development workflow section.

        Args:
            commands: Dictionary of workflow steps to commands

        Returns:
            Markdown workflow section
        """
        if not commands:
            return ""

        lines = ["## Development Workflow\n"]

        for step, command in commands.items():
            lines.append(f"### {step}")
            lines.append("```bash")
            lines.append(command)
            lines.append("```")
            lines.append("")

        return "\n".join(lines)

    @staticmethod
    def generate_issues(issues: Optional[list[str]] = None) -> str:
        """Generate known issues section.

        Args:
            issues: List of TODOs/FIXMEs found

        Returns:
            Markdown issues section
        """
        if not issues:
            return ""

        lines = ["## Known Issues\n"]

        for issue in issues:
            lines.append(f"- {issue}")

        lines.append("")
        return "\n".join(lines)

    @staticmethod
    def generate_footer() -> str:
        """Generate footer with regeneration instructions.

        Returns:
            Markdown footer
        """
        return """---

*This file is auto-generated by SWE-CLI. Edit manually or run `/init` to refresh.*
"""

    @classmethod
    def generate_full_template(
        cls,
        project_name: str = "Project",
        language: Optional[str] = None,
        project_type: Optional[str] = None,
        description: Optional[str] = None,
        tree_output: Optional[str] = None,
        dependencies: Optional[dict[str, list[str]]] = None,
        architecture_pattern: Optional[str] = None,
        components: Optional[dict[str, str]] = None,
        key_files: Optional[dict[str, str]] = None,
        workflow_commands: Optional[dict[str, str]] = None,
        issues: Optional[list[str]] = None,
    ) -> str:
        """Generate complete OPENCLI.md template.

        Args:
            project_name: Name of the project
            language: Primary programming language
            project_type: Type of project
            description: Brief description
            tree_output: Directory tree output
            dependencies: Dependencies by category
            architecture_pattern: Architecture pattern name
            components: Architecture components
            key_files: Important files
            workflow_commands: Development commands
            issues: Known issues/TODOs

        Returns:
            Complete OPENCLI.md content
        """
        sections = [
            cls.generate_header(project_name),
            cls.generate_overview(language, project_type, description),
            cls.generate_structure(tree_output),
            cls.generate_dependencies(dependencies),
            cls.generate_architecture(architecture_pattern, components),
            cls.generate_key_files(key_files),
            cls.generate_workflow(workflow_commands),
            cls.generate_issues(issues),
            cls.generate_footer(),
        ]

        # Filter out empty sections and join
        return "\n".join(section for section in sections if section.strip())


# Example minimal template for agent reference
MINIMAL_TEMPLATE = """# {project_name}

> Generated by SWE-CLI /init on {date}

## Project Overview

**Language**: {language}
**Description**: {description}

## Folder Structure

```
{tree_output}
```

## Dependencies

{dependencies}

## Key Files

- **Entry Point**: `{entry_point}`
- **Configuration**: `{config_files}`

## Development Workflow

### Install
```bash
{install_command}
```

### Run
```bash
{run_command}
```

### Test
```bash
{test_command}
```

---

*Auto-generated by SWE-CLI. Run `/init` to refresh.*
"""
