![Samplomatic](assets/fig/samplomatic.svg)

_Serving all of your circuit sampling needs since 2025._

> [!NOTE]
> This library is in an early stage of development where things are changing fast and in breaking ways.
> While the major version is `0`, please expect breaking changes between minor versions and pin your dependencies accordingly. We do not issue deprecation warnings presently, but we will document breaking changes in the changelog.

Samplomatic is a library that helps you sample randomizations of your quantum circuits in exactly the way that you specify.
Pauli twirling a static circuit is the simplest example, but the types of randomization available are extensible by design—we hope that you will contribute your own weird groups!
Beyond twirling, which is a primary use-case, this library also supports other types of randomization, such as sampling-based noise injection.

## Installation

You can install Samplomatic via pip from PyPI:

```bash
pip install samplomatic
```

For visualization support, include the visualization dependencies:

```bash
pip install samplomatic[vis]
```

Or, you can clone this repo and install in editable mode, in this example with both visualization and development optional dependencies:

```bash
pip install -e ".[dev,vis]"
```

## Hello World

In samplomatic, twirling intent is specified declaratively with annotated box instructions within a Qiskit quantum circuit. Other randomization intent is available via configuring the attributes of annotations, or other annotation types like `InjectNoise`.
These boxes can be constructed manually, as in the following example, or automatically, using transpiler passes defined in `samplomatic.transpiler`.

```python
from samplomatic import build, Twirl
from qiskit.circuit import QuantumCircuit, Parameter
import numpy as np


circuit = QuantumCircuit(5)

with circuit.box([Twirl()]):
    # twirled boxes are always "dressed": putting your single-qubit gates into
    # the boxes will result in them being composed into the "dressing" layer
    # that also includes random (in this case) Paulis
    circuit.sx(0)
    circuit.t(0)
    # notice that twirl-annotated circuits can themselves be parametric
    circuit.rx(Parameter("x"), 3)
    circuit.rx(Parameter("y"), 4)
    circuit.x(2)

    circuit.cx(1, 0)
    circuit.cz(3, 4)

with circuit.box([Twirl(decomposition="rzrx")]):
    # this box Pauli-twirls measurement, folding hadamards into the dressing
    circuit.h(range(5))
    circuit.measure_all()

circuit.draw("mpl", scale=0.5)
```

![Base circuit with twirl-annotated boxes.](assets/fig/readme-circuit.png)

Next, the `build()` function is invoked to interpret the boxes into a circuit and samplex pair.
The template is structurally similar to the original circuit and contains sufficient parametric
gates to implement any specific randomization.
The samplex encodes all information about the randomization process itself.
In other words, it represents a probability distribution over arguments for the parameters
of the template circuit, and also over other classical quantities required for post-processing results.
It is represented as a DAG, where each graph node represents a procedure such as sampling from a virtual group, composing virtual group members, commuting gates past each other, converting virtual gates to parameter values, and so forth.

```python
template, samplex = build(circuit)

template.draw("mpl", scale=0.5)
samplex.draw()
```

![Template circuit generated by build().](assets/fig/readme-template.png)
![Samplex generated by build().](assets/fig/readme-samplex.png)

At this point, we are ready to generate randomizations by calling `samplex.sample(...)`.
Notice we must provide concrete values for the parameters `"x"` and `"y"` of the original circuit.
This process does not generate new quantum circuits, it instead generates circuit arguments that are valid for the template circuit.
It additionally generates values required during post-processing, which in this example are bit-flips for the `meas` classical register because we are Pauli-twirling measurements.

```python
# generate input for the samplex
samplex_input = samplex.inputs()
samplex_input.bind(parameter_values=np.array([0.1, 0.2]))

# sample 15 randomizations valid against the template circuit.
samples = samplex.sample(samplex_input, num_randomizations=15)

# measurement bitflips are available
samples["measurement_flips.meas"] # boolean array

# one can, for example, bind the template circuit against the 7th randomization.
template.assign_parameters(samples["parameter_values"][7])
```
