# This code is a Qiskit project.
#
# (C) Copyright IBM 2025.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""InjectNoiseNode"""

import numpy as np
from qiskit.quantum_info import PauliLindbladMap

from ...aliases import NumSubsystems, RegisterName, StrRef
from ...annotations import VirtualType
from ...virtual_registers import PauliRegister, Z2Register
from .sampling_node import SamplingNode


class InjectNoiseNode(SamplingNode):
    """A node that produces samples for injecting noise.

    The node uses a :class:`qiskit.quantum_info.PauliLindbladMap` as the distribution
    to draw samples from. This map can be modified at sample time:

    * ``noise_scales``, a dictionary from :class:`~.StrRef` to floats. If ``modifier_ref``
      is in the dictionary, the distribution is replaced with the one generated by
      by :meth:`qiskit.quantum_info.PauliLindbladMap.scale_rates`.
    * ``local_scales``, a dictionary from :class:`~.StrRef` to lists of floats. If ``modifier_ref``
      is in the dictionary, the rates of
      :class:`qiskit.quantum_info.PauliLindbladMap` are scaled individually.

    Args:
        register_name: The name of the register to store the samples.
        sign_register_name: The name of the register to store the signs.
        noise_ref: Unique identifier of the noise map to draw samples from.
        num_subsystems: The number of subsystems this node generates gates for.
        modifier_ref: Unique identifier for modifiers applied to the noise map before sampling.
    """

    def __init__(
        self,
        register_name: RegisterName,
        sign_register_name: RegisterName,
        noise_ref: StrRef,
        num_subsystems: int,
        modifier_ref: StrRef = "",
    ):
        self._register_name = register_name
        self._sign_register_name = sign_register_name
        self._noise_ref = noise_ref
        self._modifier_ref = modifier_ref
        self._num_subsystems = num_subsystems

    def _to_json_dict(self) -> dict[str, str]:
        return {
            "node_type": "5",
            "register_name": self._register_name,
            "sign_register_name": self._sign_register_name,
            "noise_ref": self._noise_ref,
            "modifier_ref": self._modifier_ref,
            "num_subsystems": str(self._num_subsystems),
        }

    @classmethod
    def _from_json_dict(cls, data: dict[str, str]) -> "InjectNoiseNode":
        return cls(
            data["register_name"],
            data["sign_register_name"],
            data["noise_ref"],
            int(data["num_subsystems"]),
            data["modifier_ref"],
        )

    @property
    def outgoing_register_type(self) -> VirtualType:
        return VirtualType.PAULI

    def instantiates(self) -> dict[RegisterName, tuple[NumSubsystems, VirtualType]]:
        return {
            self._register_name: (self._num_subsystems, VirtualType.PAULI),
            self._sign_register_name: (1, VirtualType.Z2),
        }

    def sample(self, registers, rng, inputs, num_randomizations):
        rates = inputs.get(f"noise_maps.rates.{self._noise_ref}")
        paulis = inputs.get(f"noise_maps.paulis.{self._noise_ref}")
        if self._modifier_ref:
            scale = inputs.get("noise_scales." + self._modifier_ref, 1.0)
            local_scale = inputs.get(
                "local_scales." + self._modifier_ref, np.ones(paulis.num_terms)
            )
            rates = rates * scale * local_scale
        noise_map = PauliLindbladMap.from_sparse_list(
            [(pauli, idxs, rate) for (pauli, idxs), rate in zip(paulis.to_sparse_list(), rates)],
            num_qubits=paulis.num_qubits,
        )
        signs, samples = noise_map.signed_sample(num_randomizations, rng.bit_generator.random_raw())
        registers[self._register_name] = PauliRegister(samples.to_dense_array().transpose())
        registers[self._sign_register_name] = Z2Register(signs.reshape(1, -1))

    def get_style(self):
        return (
            super()
            .get_style()
            .append_data("Register Name", repr(self._register_name))
            .append_data("Subsystems", repr(self._num_subsystems))
            .append_data("Noise Reference", repr(self._noise_ref))
            .append_data("Modifier Reference", repr(self._modifier_ref))
        )
