"""
Unit test file.
"""

import os
import unittest

from dotenv import load_dotenv

from rclone_api import Config, Rclone
from rclone_api.diff import DiffItem, DiffType

load_dotenv()

BUCKET_NAME = os.getenv("BUCKET_NAME")  # Default if not in .env


def _generate_rclone_config() -> Config:

    # BUCKET_NAME = os.getenv("BUCKET_NAME", "TorrentBooks")  # Default if not in .env

    # Load additional environment variables
    BUCKET_KEY_SECRET = os.getenv("BUCKET_KEY_SECRET")
    BUCKET_KEY_PUBLIC = os.getenv("BUCKET_KEY_PUBLIC")
    # BUCKET_URL = os.getenv("BUCKET_URL")
    BUCKET_URL = "sfo3.digitaloceanspaces.com"

    config_text = f"""
[dst]
type = s3
provider = DigitalOcean
access_key_id = {BUCKET_KEY_PUBLIC}
secret_access_key = {BUCKET_KEY_SECRET}
endpoint = {BUCKET_URL}
"""

    out = Config(config_text)
    return out


class RcloneDiffTests(unittest.TestCase):
    """Test rclone functionality."""

    def setUp(self) -> None:
        """Check if all required environment variables are set before running tests."""
        required_vars = [
            "BUCKET_NAME",
            "BUCKET_KEY_SECRET",
            "BUCKET_KEY_PUBLIC",
            "BUCKET_URL",
        ]
        missing = [var for var in required_vars if not os.getenv(var)]
        if missing:
            self.skipTest(
                f"Missing required environment variables: {', '.join(missing)}"
            )
        os.environ["RCLONE_API_VERBOSE"] = "1"

    def test_diff(self) -> None:
        """Test copying a single file to remote storage."""
        rclone = Rclone(_generate_rclone_config())
        item: DiffItem
        all: list[DiffItem] = []
        for item in rclone.diff("dst:rclone-api-unit-test", "dst:rclone-api-unit-test"):
            self.assertEqual(
                item.type, DiffType.EQUAL
            )  # should be equal because same repo
            all.append(item)
        self.assertGreater(len(all), 10)
        msg = "\n".join([str(item) for item in all])
        print(msg)

        all.clear()
        for item in rclone.diff(
            "dst:rclone-api-unit-test/test", "dst:rclone-api-unit-test/test"
        ):
            self.assertEqual(item.type, DiffType.EQUAL)
            print(item)
            all.append(item)

        msg = "\n".join([str(item) for item in all])
        print(msg)
        print("done")


if __name__ == "__main__":
    unittest.main()
