from collections.abc import Iterable
from copy import deepcopy

from django.db.models import QuerySet


def deep_map(data: dict | list, func_cond, func_map, in_place=True):
    if not in_place:
        data = deepcopy(data)

    if isinstance(data, dict):
        for key, value in data.items():
            if isinstance(value, (list, dict, QuerySet)):
                deep_map(value, func_cond, func_map, True)
            elif func_cond(value):
                data[key] = func_map(value)
    elif isinstance(data, (list, QuerySet)):
        for index, value in enumerate(data):
            if isinstance(value, (list, dict, QuerySet)):
                deep_map(value, func_cond, func_map, True)
            elif func_cond(value):
                data[index] = func_map(value)

    return data


def deep_round(data: dict | list, ndigits: int, in_place=True):
    return deep_map(data, lambda value: isinstance(value, float), lambda value: round(value, ndigits), in_place)


def safe_sum(*args, allow_null=True):
    if isinstance(args[0], Iterable):
        args = args[0]

    if all(arg is None for arg in args):
        return None
    if not allow_null and any(arg is None for arg in args):
        return None

    _sum = 0
    for arg in args:
        _sum += arg or 0
    return _sum


def safe_subtract(*args, allow_null=False):
    if isinstance(args[0], Iterable):
        args = args[0]

    if all(arg is None for arg in args):
        return None
    if not allow_null and any(arg is None for arg in args):
        return None

    _sum = args[0] or 0
    for arg in args[1:]:
        _sum -= arg or 0
    return _sum
