import pandas as pd
from typing import List, Optional
import scipy.stats as ss
import statsmodels.api as sm
import factor_analyzer
from .base_stats import _prepare_dataset
from ..dataframe.utils import send_thinking_step


def one_sample_t_test(df: pd.DataFrame, column_name: str, mu: str) -> tuple:
    """ One sample t-test is a statistical procedure used to determine whether a sample of observations could have been generated by a process with a specific mean.
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        column_name (str): "Column name"
        mu (str): "float Population mean value"
    
    Returns:
        tuple: (t_statistic, p_value) - The t-statistic and p-value of the test
    """
    column = df[column_name]
    mu = float(mu)
    t_statistic, p_value = ss.ttest_1samp(column, mu)
    result = f"t-statistic: {t_statistic}, p-value: {p_value}"
    send_thinking_step(func="one_sample_t_test", content=result)
    return t_statistic, p_value


def two_sample_t_test(df: pd.DataFrame, column_one: str, column_two:str) -> tuple:
    """ Two sample t-test is a statistical procedure used to determine whether the means of two samples are equal to each other.
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        column_one (str): "Column name with one sample "
        column_two (str): "Column name for second sample"
    
    Returns:
        tuple: (t_statistic, p_value) - The t-statistic and p-value of the test
    """
    _df = df[[column_one, column_two]]
    _df = _df.dropna()
    column_one = _df[column_one]
    column_two = _df[column_two]
    t_statistic, p_value = ss.ttest_ind(column_one, column_two)
    result = f"t-statistic: {t_statistic}, p-value: {p_value}"
    send_thinking_step(func="two_sample_t_test", content=result)
    return t_statistic, p_value


def one_way_ANOVA(df: pd.DataFrame, columns: List[str], formula: str) -> pd.DataFrame:
    """ One way ANOVA compares the variance in the group means of an independent variable on a dependent variable.
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List of column names used within the R formula
        formula (str): "R formula for representing relationship between variables"
        
    Returns:
        pd.DataFrame: ANOVA table results
    """
    _df, _, _ = _prepare_dataset(df, columns)
    model = sm.formula.api.ols(formula, data=_df).fit()
    table = sm.stats.anova_lm(model)
    send_thinking_step(func="one_way_ANOVA", content=f"ANOVA results:\n{table.to_string()}")
    return table


def two_way_ANOVA(df: pd.DataFrame, columns_set_one: List[str], formula_one: str, 
                 columns_set_two: Optional[List[str]]=None, formula_two: Optional[str]=None) -> pd.DataFrame:
    """ Two way ANOVA compares the variance in the group means of two independent variables on a dependent variable and on each other.
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns_set_one (List[str]): List of column names used within R formula one
        formula_one (str): "R formula for ANOVA"
        columns_set_two (Optional[List[str]], optional): List of column names used within R formula two
        formula_two (Optional[str], optional): "R formula for ANOVA"
        
    Returns:
        pd.DataFrame: ANOVA table results
    """
    if not columns_set_two or not formula_two:
        return one_way_ANOVA(df, columns_set_one, formula_one)
    
    _df_one = df[columns_set_one]
    _df_two = df[columns_set_two]
    
    table = sm.stats.anova_lm(
        sm.formula.api.ols(formula_one, data=_df_one).fit(),
        sm.formula.api.ols(formula_two, data=_df_two).fit()
    )
    
    send_thinking_step(func="two_way_ANOVA", content=f"Two-way ANOVA results:\n{table.to_string()}")
    return table


def kruskal_wallis_test(df: pd.DataFrame, columns: List[str]) -> tuple:
    """ Kruskal-Wallis test is a non-parametric statistical test that is used to evaluate whether the medians of two or more groups are different.
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List of numeric column names
        
    Returns:
        tuple: (statistic, p_value) - The test statistic and p-value
    """
    _df, _, _ = _prepare_dataset(df, columns)
    res = ss.kruskal(*[_df[c] for c in _df.columns])
    result = f"Kruskal-Wallis statistic: {res.statistic}, p-value: {res.pvalue}"
    send_thinking_step(func="kruskal_wallis_test", content=result)
    return res.statistic, res.pvalue
    

def chi_square_test(df: pd.DataFrame, column_one: str, column_two:str) -> tuple:
    """ Chi-square test to determine whether observed frequencies are significantly different from expected frequencies.
    Args:
        df (pd.DataFrame): DataFrame to operate on
        column_one (str): "Numeric column name with one sample "
        column_two (str): "Numeric column name for second sample"
        
    Returns:
        tuple: (statistic, p_value) - The chi-square statistic and p-value
    """
    _df = df[[column_one, column_two]]
    _df = _df.dropna()
    res = ss.chi2_contingency(_df)
    result = f"chi-square-statistic: {res.statistic}, p-value: {res.pvalue}"
    send_thinking_step(func="chi_square_test", content=result)
    return res.statistic, res.pvalue


def chi_square_test_on_data(df: pd.DataFrame, column_one_name: str, column_one_data: List[float], column_two_name: str, column_two_data: List[float]) -> tuple:
    """ Chi-square test on data to determine whether observed frequencies are significantly different from expected frequencies.
    Args:
        df (pd.DataFrame): DataFrame to operate on
        column_one_name (str): "Column name for first sample "
        column_one_data (List[float]): "List of values for first column"
        column_two_name (str): "Column name for second sample"
        column_two_data (List[float]): "List of values for second column"
        
    Returns:
        tuple: (statistic, p_value) - The chi-square statistic and p-value
    """
    _df = pd.DataFrame(
        {
            column_one_name: column_one_data,
            column_two_name: column_two_data
        }
    )
    res = ss.chi2_contingency(_df)
    result = f"chi-square-statistic: {res.statistic}, p-value: {res.pvalue}"
    send_thinking_step(func="chi_square_test_on_data", content=result)
    return res.statistic, res.pvalue


def bartlett_test(df: pd.DataFrame, columns: List[str]) -> tuple:
    """ Bartlett's test tests the null hypothesis that all input samples are from populations with equal variances, returns chi-square value and p-value
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List of numeric column names
        
    Returns:
        tuple: (chi_square_value, p_value) - The chi-square statistic and p-value
    """
    _df, _, _ = _prepare_dataset(df, columns)
    chi_square_value, p_value = factor_analyzer.factor_analyzer.calculate_bartlett_sphericity(_df)
    result = f"Chi-square: {chi_square_value}, p value: {p_value}"
    send_thinking_step(func="bartlett_test", content=result)
    return chi_square_value, p_value
    

def KMO_test(df: pd.DataFrame, columns: List[str]) -> tuple:
    """ Kaiser-Meyer-Olkin (KMO) Test measures the suitability of data for factor analysis
    
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List of numeric column names
        
    Returns:
        tuple: (kmo_per_item, kmo_total) - KMO scores per item and total score
    """
    _df, _, _ = _prepare_dataset(df, columns)
    kmo_per_item, kmo_total = factor_analyzer.factor_analyzer.calculate_kmo(_df)
    
    # Format columns in a table-friendly way
    column_headers = " | ".join(columns)
    kmo_values = " | ".join([str(round(x, 4)) for x in kmo_per_item])
    
    message = f"""#### KMO Test results:\n\n"""
    message += f"""##### KMO Total: {kmo_total}\n\n"""
    message += f"""##### KMO per item: \n\n| {column_headers} |\n| {kmo_values} |"""
    
    send_thinking_step(func="KMO_test", content=message)
    return kmo_per_item, kmo_total



def mann_whitney_test(df: pd.DataFrame, columns: List[str], method: str='auto') -> tuple:
    """ Mann-Whitney U test is a nonparametric test that allows two groups or conditions or treatments to be compared without making the assumption that values are normally distributed.
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List containing two numeric column names
        method (str): "Method to use for calculating p-value, available methods are 'auto', 'asymptotic', 'exact', default is 'auto'"
        
    Returns:
        tuple: (U1, p_value) - The U1 statistic and p-value
    """
    if len(columns) != 2:
        return "ERROR: Mann-Whitney U test requires two columns"
    _df = df[columns].dropna()
    U1, p = ss.mannwhitneyu(_df[columns[0]], _df[columns[1]], method=method)
    result = f"U1-statistics: {U1}, p-value: {p}"
    send_thinking_step(func="mann_whitney_test", content=result)
    return U1, p
    

def wilcoxon_test(df: pd.DataFrame, columns: List[str], alternative:str='two-sided', method: str='auto') -> tuple:
    """ Wilcoxon signed-rank test is a nonparametric test that allows two groups or conditions or treatments to be compared without making the assumption that values are normally distributed.
    Args:
        df (pd.DataFrame): DataFrame to operate on
        columns (List[str]): List containing one or two column names
        alternative (str): "Alternative hypothesis, available options are 'two-sided', 'less', 'greater', default is 'two-sided'"
        method (str): "Method to use for calculating p-value, available methods are 'auto', 'asymptotic', 'exact', default is 'auto'"
        
    Returns:
        tuple: (statistic, p_value) - The Wilcoxon statistic and p-value
    """
    _df = df[columns].dropna()
    if len(columns) > 2:
        return "ERROR: Wilcoxon signed-rank test requires one or two columns"
    if alternative not in ['two-sided', 'less', 'greater']:
        alternative = 'two-sided'
    if method not in ['auto', 'asymptotic', 'exact']:
        method = 'auto'
    res = ss.wilcoxon(*[_df[c] for c in columns], alternative=alternative, method=method)
    result = f"Wilcoxon-statistics: {res.statistic}, p-value: {res.pvalue}"
    send_thinking_step(func="wilcoxon_test", content=result)
    return res.statistic, res.pvalue
