# Migrating from `dea_tools`

The `eo-tides` package contains functions that were previously available in the [`Digital Earth Australia Notebooks and Tools` repository](https://github.com/GeoscienceAustralia/dea-notebooks/).
To migrate your code from DEA Tools to `eo-tides`, please be aware of the following breaking changes:

## Breaking changes

### "tide_m" renamed to "tide_height"

The default output tide heights column generated by the `model_tides` function and the `xarray.DataArray` outputs generated by `tag_tides` and `pixel_tides` have been renamed from `tide_m` to `tide_height`.
This more clearly describes the data, particularly when used in combination with `output_units="cm"` or `output_units="mm"` which returns tide heights in non-metre units.

!!! tip "Action required"

    Update any references of `tide_m` to `tide_height`

### Default model changed to "EOT20"

The default tide model has been changed from "FES2014" to the open source "EOT20" [Empirical Ocean Tide tide model (CC BY 4.0)](https://doi.org/10.17882/79489):

> Hart-Davis Michael, Piccioni Gaia, Dettmering Denise, Schwatke Christian, Passaro Marcello, Seitz Florian (2021). EOT20 - A global Empirical Ocean Tide model from multi-mission satellite altimetry. SEANOE. <https://doi.org/10.17882/79489>

Note that this change in default is primarily due to the permissive license of this model; the choice of optimal model will vary by study area and application.

!!! tip "Action required"

    To use FES2014, set `model="FES2014"` instead of leaving the default `model=None`.

### `tidal_tag` renamed to `tag_tides`

Renamed for consistency with `model_tides` and `pixel_tides`.

!!! tip "Action required"

    Update references to `tidal_tag` to `tag_tides`.

### `ds` param renamed to `data`, now accepts `GeoBox`

The `ds` param in all satellite data functions (`tag_tides`, `pixel_tides`, `tide_stats`, `pixel_tides`) has been updated to accept either `xarray.Dataset`, `xarray.DataArray` or a `odc.geo.geobox.GeoBox`. To account for this change, the `ds` param has been renamed to a more generic name `data`.

!!! tip "Action required"

    Update:
    ```
    tag_tides(ds=your_data)
    ```
    To:
    ```
    tag_tides(data=your_data)
    ```

### `times` param renamed to `time`, accepts any format supported by `pandas.to_datetime()`

The `times` parameter has been renamed to `time`, and updated to more flexibly accept any time format that can be converted by `pandas.to_datetime()`; e.g. `np.ndarray[datetime64]`, `pd.DatetimeIndex`, `pd.Timestamp`, `datetime.datetime` and strings (e.g. `"2020-01-01 23:00"`). For example: `time=pd.date_range(start="2000", end="2001", freq="5h")`.

!!! tip "Action required"

    Update:
    ```
    model_tides(..., times=...)
    ```
    To:
    ```
    model_tides(..., time=...)
    ```

### `tag_tides` now returns an array instead of updating data in-place

The `tag_tides` function now returns an `xarray.DataArray` output containing tide heights, rather than appending tide height data to the original input dataset in-place. This change provides better consistency with `pixel_tides`, which also returns an array of tide heights.

!!! tip "Action required"

    Update:
    ```
    data = tag_tides(data, ...)
    ```
    To:
    ```
    data["tide_height"] = tag_tides(data, ...)
    ```

### `pixel_tides` only returns a single array

The `pixel_tides` function has been updated to only ever return a single array as an output: a high-resolution tide height array matching the resolution of the input `data` by default, and a low-resolution tide height array if `resample=False`.

!!! tip "Action required"

    Update code to handle a single tide height array output from `pixel_tides`, instead of a tuple of high-resolution and low-resolution modelled tide height arrays.

### Tide model directory environment variable updated

The `DEA_TOOLS_TIDE_MODELS` environmental variable has been renamed to `EO_TIDES_TIDE_MODELS`.

!!! tip "Action required"

    Set the `EO_TIDES_TIDE_MODELS` environment variable instead of `DEA_TOOLS_TIDE_MODELS`.

### Error raised if both `directory` parameter and environment variable are missing

Previously, tide modelling functions used a fall-back tide modelling directory (`/var/share/tide_models`) if both the `directory` parameter and the tide model directory environment variable were absent. This fall-back has been removed, and an error will now be raised if no tide model directory is specified.

!!! tip "Action required"

    Ensure that either the `directory` parameter or the `EO_TIDES_TIDE_MODELS` environment variable are provided.
