

import six
import importlib
import logging

from django.apps import AppConfig, apps

from abc import ABCMeta, abstractproperty

logger = logging.getLogger(__name__)


class FlavorPackAppMeta(ABCMeta):

    def __new__(self, *args, **kwargs):
        cls = ABCMeta.__new__(self, *args, **kwargs)
        FlavorSaver.register(cls)
        return cls


class FlavorSaver(object):
    '''
    The FlavorSaver class allows flavorpacks to register themselves with
    serialbox.  Once a flavorpack is registered.  Serialbox can use the
    flavorpack base class to figure out how to generate numbers from regions,
    etc.
    '''
    flavorpacks = {}

    @classmethod
    def register(cls, flavor_pack):
        '''
        Adds a flavor pack to the class level list if it is not there
        already.
        '''
        if flavor_pack not in cls.flavorpacks:
            app_name = flavor_pack.__module__.split('.')[0]
            cls.flavorpacks[app_name] = flavor_pack

    @classmethod
    def get_flavorpack_by_region(self, cls, region):
        ret = None
        app_name = region.__module__.split('.')[0]
        try:
            flavor_pack = cls.flavorpacks[app_name]
            all_apps = apps.get_app_configs()
            for app in all_apps:
                if isinstance(app, flavor_pack):
                    ret = app
        except KeyError:
            raise cls.FlavorPackNotFoundError(
                'The flavor_pack with app_name '
                '%s could not be found.' %
                app_name)
        if not ret:
            raise cls.FlavorPackNotFoundError(
                'Could not find the flavor_pack '
                'with an app name of %s' % app_name)
        return ret

    @classmethod
    def get_flavorpacks(cls):
        ret = []
        for app in apps.get_app_configs():
            if isinstance(app, FlavorPackApp):
                ret.append(app)
        return ret

    @classmethod
    def get_generator_by_region(cls, region):
        # get the region's app if it is not a native serialbox region
        flavor_pack = cls.get_flavorpack_by_region(cls, region)
        # get the generator class name
        try:
            qualname = region.__module__ + "." + region.__class__.__name__
            generator = flavor_pack.generators[qualname]
        except KeyError:
            raise cls.GeneratorNotFoundError('The generator with name % '
                                             'could not be found.' % qualname)
        # instantiate the generator and return
        logger.debug('Attempting to instantiate an instance of %s', generator)
        gen = generator.rsplit('.', 1)
        mod = importlib.import_module(gen[0])
        gencls = getattr(mod, gen[1])
        return gencls()

    @classmethod
    def get_api_urls(cls):
        '''
        Returns any API urls that any installed flavorpacks have registered with
        the serialbox default */api/pools/* call.
        '''
        ret = []
        for fp in cls.get_flavorpacks():
            try:
                urls = fp.api_urls
                if urls.__len__() > 0:
                    ret += urls
            except TypeError:
                raise
        return ret

    class FlavorPackNotFoundError(Exception):
        pass

    class GeneratorNotFoundError(Exception):
        pass


@six.add_metaclass(FlavorPackAppMeta)
class FlavorPackApp(AppConfig):
    '''
    Enforces that all flavor packs implement certain discoverable features
    at runtime.
    '''
    @abstractproperty
    def pool_slug_fields(self):
        '''
        Should return a dictionary that contains the name of the
        field that will be used to add serialized region data
        to the data generated by the PoolSerializer along with the full name
        of the Django Rest Framework
        rest_framework.serializers.SlugRelatedField
        instance that will do the work.  For example, to
        add a field named randomizedregion_set to the PoolSerializer then one
        would return
        ``
        def pool_slug_fields(self):
            return {
            'randomizedregion_set':
            'random_flavorpack.api.serializers.randomizedregion_set'
            }
        ``
        The above sample is used by the SerialBox random_flavorpack to
        add its region info to the PoolSerializer at runtime using a
        Django rest framework SlugRelatedField.
        '''
        pass

    @abstractproperty
    def pool_hyperlink_fields(self):
        '''
        Should return a dictionary that contains the name of the
        field that will be used to add serialized region data
        to the data generated by the PoolSerializer along with the full name
        of the rest_framework.serializers.HyperlinkedRelatedField
        instance that will do the work.  For example, to
        add a field named randomizedregion_set to the PoolSerializer then one
        would return:
        ``
        def pool_hyperlink_fields(self):
            return {
            'randomizedregion_set':
            'random_flavorpack.api.serializers.randomizedregion_hyperlink_set'
            }
        ``
        '''
        pass

    @abstractproperty
    def generators(self):
        '''
        Each flavorpack needs to tell serialbox how to generate numbers
        (if at all).  So each flavorpack, if it's in the business of creating
        numbers, has got to provide an instance of the
        serialbox.pools.generators.Generator to use in order to do this.
        '''
        pass

    @abstractproperty
    def api_urls(self):
        '''
        Any URLs that an implementer wants added to the default *api/pools/*
        api list.  This should return a list **NOT** a dictionary...unlike
        each of the other properties in this class.
        '''
        pass
