# fiddler-client

A Python library for interacting with the Fiddler HTTP API.

## Development

Run unit tests:

```
$ make test
...
156 passed, 1 skipped in 0.59s
```


Run mypy, pylint, ruff, flake8:

```
$ make lint
mypy fiddler
Success: no issues found in 77 source files
ruff check .
All checks passed!
pylint fiddler
...
flake8 fiddler
```


## Logging

### Automatic log handler setup for ad-hoc scripts

Consider this minimal Python program:

```python
import fiddler

# The call to `init()` below detects that there is no root logger configured.
# It automatically configures an outlet for log messages generated by this
# library: they end up being written to stderr with a meaningful default log
# format.
fiddler.init(url="https://f.bar", token="x")
```

This feature can be disabled by passing `auto_attach_log_handler=False` to `init()`.

### Full control in calling code: configure a root logger

Any more complex Python program that uses this client library typically sets up the logging machinery in a custom way. Example:

```python
import logging
import os

import fiddler

# Example for configuring a root logger, with a stream handler attached for
# emission to stderr:
logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s.%(msecs)03d %(levelname)s: %(message)s",
    datefmt="%y%m%d-%H:%M:%S",
)

# Now, the call to init() below does not attach any additional log handlers.
# Log messages emitted by this client library are simply propagated into the
# root logger (and adhere to custom filtering and formatting rules).
fiddler.init(url="https://f.bar", token="x")

# Example for custom verbosity control: do not forward DEBUG+INFO level
# messages from the 'fiddler' logger into the root logger.
logging.getLogger('fiddler').setLevel(logging.WARNING)
```

## Releasing

### Pre Release

As we get ready for distribution of the updated python client we need to ensure that all the associated downstream artifacts are updated.

1. Inform on `#releases` slack channel of the impending python client release with release notes.
1. [`Fiddler Docs`](https://docs.fiddler.ai/docs) needs updating.
2. Quick start and other notebooks linked to in fiddler docs need updating.
3. Ensure that the customer success team is appraised of the impending changes and possible impact.
4. In case of breaking changes, eg: changes to request/response, the following is a must
   - All quick start notebooks in fiddler docs need updation.
   - The code snippets in [`Fiddler Docs`](https://docs.fiddler.ai/docs) needs updating
   - If it's a new feature appropriate care has to be taken to
   - Customer Success needs to sign off on the changes and they need to plan for customer update

### PyPI

The Fiddler Python client is published as
[`fiddler-client`](https://pypi.org/project/fiddler-client/) in the Python
Package Index.

1. Set the new semantic version number in `fiddler/_version.py`, e.g. `1.4.3`;
2. Update `PUBLIC.md` with release notes for the new version;
3. Raise a PR;
4. Once the PR is merged, create a new annotated tag on the `main` branch. For
   example:

   ```bash
   git checkout main
   git pull
   git tag -a 1.4.3 -m 'The one that fixes event publishing'
   git push --tag
   ```

This triggers a pipeline that will automatically build and publish the new
version of the client to PyPI.

Note: dev versions may be published from any branch at any time by using a
`.devN` affix, as described in [PEP-440](https://peps.python.org/pep-0440/). For
example: `1.4.3.dev5`.

### Post Release

1. Inform on `#releases` slack channel of the python client release with release notes.
1. Verify:
   1. [`Fiddler Docs`](https://docs.fiddler.ai/docs) docs reflect the changes
   1. All the quick start guides are working (installing the python client from PyPI)


