import logging
import time
from warnings import warn

_handler_attached = False


def _attach_handler() -> bool:
    """
    Potentially attach a handler to the 'fiddler' logger.

    Not meant to be called by a user of this library directly. Meant to be
    called during client library init, automatically under special conditions.

    Design goal: do this setup at most once per Python interpreter session
    (otherwise more than one StreamHandler gets attached (leading to each msg
    being emitted more than once).

    Note(JP): logging is meant to go to `stderr`. The result of a CLI program
    goes to `stdout`. Noting this here because previously we emitted to stdout.
    Let's not do this.

    The caller (client init) must make sure to set up this handler/formatting
    machinery only optionally. A calling program that has set up logging
    already does define handler and formatters by its own choice, and does not
    want a handler like the below to be active by default.

    Return boolean indicating whether handler was attached or not.
    """

    global _handler_attached
    if _handler_attached:
        return False

    # Show timestamp in UTC, indicate this with the trailing Z for Zulu time.
    # Pad level with whitespace towards fixed-width formatting.
    fmttr = logging.Formatter(
        '%(asctime)s.%(msecs)03dZ %(levelname)8s| %(message)s ',
        datefmt='%y%m%dT%H:%M:%S',
    )

    # Make timestamps reflect time in UTC.
    fmttr.converter = time.gmtime

    h = logging.StreamHandler()
    h.setFormatter(fmttr)
    h.setLevel(logging.INFO)
    flog = logging.getLogger('fiddler')
    flog.addHandler(h)
    flog.setLevel(logging.INFO)

    # And now that this logger has its own handler attached: do not propagate
    # log messages up the hierarchy.
    flog.propagate = False

    _handler_attached = True
    return True


def set_logging(level=logging.INFO) -> bool:  # type: ignore # pylint: disable=unused-argument
    """Set the application logger at the specified logging level.

    ⚠️ Deprecated as of version 3.7.0 and included only for backwards compatibility
    and will be removed in a future release.

    To set the logging level post-version 3.7.0, please configure your application
    or notebook explicitly with its own root logger as this example demonstrates:

    # Create and configure the root logger prior to calling init()
    logging.basicConfig(
        level=logging.WARN,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[logging.StreamHandler()],
    )

    Args:
        level: Logging level from python logging module (default: logging.INFO)

    Returns:
        bool

    Examples:
        set_logging(logging.INFO)
    """
    warn(
        'The set_logging() function will be removed in the future. '
        'For fine-grained control over how log messages generated by this '
        'library end up being emitted please configure the root logger '
        'explicitly, and use the canonical means of '
        'setting up Python standard library logging.',
        DeprecationWarning,
        stacklevel=2,
    )

    # This was approximately the previous behavior, so do that for now.
    return _attach_handler()
