# NEON AI (TM) SOFTWARE, Software Development Kit & Application Development System
# All trademark and other rights reserved by their respective owners
# Copyright 2008-2025 Neongecko.com Inc.
# BSD-3
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 1. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# 3. Neither the name of the copyright holder nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
# PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
# CONTRIBUTORS  BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
# OR PROFITS;  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE,  EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

from os import makedirs
from os.path import isfile, join, isdir
from time import time
from typing import List, Dict
from fastapi import FastAPI, Response

import uvicorn
import gradio

from threading import Event
from ovos_bus_client import Message
from ovos_config import Configuration
from ovos_utils import LOG
from ovos_utils.json_helper import merge_dict
from neon_utils.file_utils import decode_base64_string_to_file
from ovos_utils.xdg_utils import xdg_data_home
from neon_data_models.models.api.messagebus import NeonTtsResponse

from neon_iris.client import NeonAIClient


class GradIOClient(NeonAIClient):
    def __init__(self, lang: str = None):
        config = Configuration()
        self.config = config.get('iris') or dict()
        NeonAIClient.__init__(self, config.get("MQ"))
        self._await_response = Event()
        self._alerts = dict()
        self._response = None
        self._transcribed = None
        self._current_tts = dict()
        self._profiles: Dict[str, dict] = dict()
        self._audio_path = join(xdg_data_home(), "iris", "stt")
        if not isdir(self._audio_path):
            makedirs(self._audio_path)
        self.default_lang = (lang or 
                             self.config.get('default_lang')).split('-')[0]
        self.chat_ui = gradio.Blocks()

    def get_lang(self, session_id: str) -> str:
        """
        Get the ISO 639-1 language code for the specified session
        @param session_id: Gradio session ID
        @returns: ISO 639-1 language code
        """
        if session_id and session_id in self._profiles:
            return self._profiles[session_id]['speech']['stt_language'].split('-')[0]
        return (self.user_config['speech']['stt_language'] or 
                self.default_lang).split('-')[0]

    @property
    def supported_languages(self) -> List[str]:
        """
        Get a list of supported languages from configuration
        @returns: list of ISO 639-1 language codes
        """
        return [lang.split('-')[0] for lang in 
                self.config.get('languages') or [self.default_lang]]

    def _start_session(self, request: gradio.Request):
        sid = request.session_hash
        self._current_tts[sid] = None
        self._profiles[sid] = self.user_config
        self._profiles[sid]['user']['username'] = sid
        return sid

    def check_alerts(self, session_id: str):
        if not self._alerts.get(session_id):
            gradio.Info("No Alerts")
            return session_id
        while self._alerts.get(session_id):
            gradio.Info(self._alerts[session_id].pop())
        return session_id

    def update_profile(self, stt_lang: str, tts_lang: str, tts_lang_2: str,
                       time: int, date: str, uom: str, city: str, state: str,
                       country: str, first: str, middle: str, last: str,
                       pref_name: str, email: str, session_id: str):
        """
        Callback to handle user settings changes from the web UI
        """
        location_dict = dict()
        if any((city, state, country)):
            from neon_utils.location_utils import get_coordinates, get_timezone
            try:
                location_dict = {"city": city, "state": state,
                                 "country": country}
                lat, lon = get_coordinates(location_dict)
                location_dict["lat"] = lat
                location_dict["lng"] = lon
                location_dict["tz"], location_dict["utc"] = get_timezone(lat,
                                                                         lon)
                LOG.debug(f"Got location update: {location_dict}")
            except Exception as e:
                LOG.exception(e)

        profile_update = {"speech": {"stt_language": stt_lang,
                                     "tts_language": tts_lang,
                                     "secondary_tts_language": tts_lang_2},
                          "units": {"time": time, "date": date, "measure": uom},
                          "location": location_dict,
                          "user": {"first_name": first, "middle_name": middle,
                                   "last_name": last,
                                   "preferred_name": pref_name, "email": email}}
        old_profile = self._profiles.get(session_id) or self.user_config
        self._profiles[session_id] = merge_dict(old_profile, profile_update)
        LOG.info(f"Updated profile for: {session_id}")
        return session_id

    def on_user_input(self, utterance: str,
                      chat_history: List[gradio.ChatMessage],
                      audio_input: str,  # gr.Audio object input
                      client_session: str):# -> tuple[List[Tuple[str, str]], str, Literal[''], None, Any]:
        """
        Callback to handle textual user input
        @param utterance: String utterance submitted by the user
        @returns: Input box contents, Updated chat history, Gradio session ID, audio input, audio output
        """
        if not any((utterance, audio_input)):
            # Empty input
            return chat_history, client_session, "", None, None

        input_time = time()
        LOG.debug("Input received")
        if not self._await_response.wait(30):
            LOG.error("Previous response not completed after 30 seconds")
        in_queue = time() - input_time
        self._await_response.clear()
        self._response = None
        self._transcribed = None
        gradio_id = client_session
        lang = self.get_lang(gradio_id)
        if utterance:
            LOG.info(f"Sending utterance: {utterance} with lang: {lang}")
            self.send_utterance(utterance, lang, username=gradio_id,
                                user_profiles=[self._profiles[gradio_id]],
                                context={"gradio": {"session": gradio_id},
                                         "timing": {"wait_in_queue": in_queue,
                                                    "gradio_sent": time()}})
        else:
            LOG.info(f"Sending audio: {audio_input} with lang: {lang}")
            self.send_audio(audio_input, lang, username=gradio_id,
                            user_profiles=[self._profiles[gradio_id]],
                            context={"gradio": {"session": gradio_id},
                                     "timing": {"wait_in_queue": in_queue,
                                                "gradio_sent": time()}})
            # Add the audio input to the chat history
            chat_message = gradio.ChatMessage(role="user", content=audio_input)
            chat_history.append(chat_message)
        if not self._await_response.wait(30):
            LOG.error("No response received after 30s")
            self._await_response.set()
        self._response = self._response or "ERROR"
        LOG.info(f"Got response={self._response}")
        if utterance:
            chat_history.append(gradio.ChatMessage(role="user", content=utterance))
        elif isinstance(self._transcribed, str):
            LOG.info(f"Got transcript: {self._transcribed}")
            chat_history.append(gradio.ChatMessage(role="user", content=self._transcribed))
        chat_history.append(gradio.ChatMessage(role="assistant", content=self._response))
        # Add audio response to chat
        chat_message = gradio.ChatMessage(role="assistant", content={"path": self._current_tts[gradio_id]})
        chat_history.append(chat_message)
        LOG.info(f"Chat history now has {len(chat_history)} messages")
        return chat_history, gradio_id, "", None, self._current_tts[gradio_id]

    def play_tts_audio(self, session_id: str):
        """
        Helper method to return TTS audio for playback
        """
        audio_file = self._current_tts.get(session_id)
        LOG.info(f"Playing TTS audio: {audio_file}")
        return audio_file

    # def play_tts(self, session_id: str):
    #     LOG.info(f"Playing most recent TTS file {self._current_tts}")
    #     return self._current_tts.get(session_id), session_id

    def run(self):
        """
        Blocking method to start the web server
        """
        self._await_response.set()
        title = self.config.get("webui_title", "Neon AI")
        description = self.config.get("webui_description", "Chat With Neon")
        chatbot_label = self.config.get("webui_chatbot_label") or description
        speech = self.config.get("webui_mic_label") or description
        text_label = self.config.get("webui_text_label") or description
        placeholder = self.config.get("webui_input_placeholder",
                                      "Ask me something")
        address = self.config.get("server_address") or "0.0.0.0"
        port = self.config.get("server_port") or 7860

        with self.chat_ui as blocks:
            # Add custom CSS to hide audio waveforms
            blocks.css = """
            /* Ensure elements in a row share the same height */
            #custom-submit {
                height: 100%; /* Match the height of other elements */
                display: flex;
                align-items: center;
            }

            /* Hide the waveform container and visualization */
            .waveform-container, #waveform, .svelte-19usgod .waveform-container {
                display: none !important;
            }
            
            /* Hide timestamps */
            .timestamps {
                display: none !important;
            }
            
            /* Keep only essential playback controls and make them compact */
            [data-testid="unlabelled-audio"] {
                max-height: 60px !important;
                overflow: hidden !important;
            }
            
            /* Make the controls more compact */
            .controls.svelte-ije4bl {
                padding: 5px !important;
                margin: 0 !important;
            }
            
            /* Hide volume and playback speed controls to make it even more minimal */
            .control-wrapper .volume, .control-wrapper .playback {
                display: none !important;
            }
            
            /* Hide rewind/skip buttons for minimal interface */
            .rewind, .skip {
                display: none !important;
            }
            """
            # Do session-specific initialization
            client_session = gradio.State()
            blocks.load(self._start_session, None, outputs=[client_session])

            # Define primary UI
            blocks.title = title
            chatbot = gradio.Chatbot(type='messages', label=chatbot_label)
            with gradio.Row(height='120px'):
                textbox = gradio.Textbox(label=text_label,
                                         placeholder=placeholder, lines=2,
                                         scale=8)
                audio_input = gradio.Audio(sources=["microphone"],
                                           type="filepath",
                                           label=speech,
                                           scale=2)
                submit = gradio.Button(value="Submit", elem_id="custom-submit",
                                       variant="primary")
                tts_audio = gradio.Audio(autoplay=True, visible=True, 
                                         label="Response Audio",
                                         interactive=False,
                                         type="filepath", scale=1)
            submit.click(self.on_user_input,
                         inputs=[textbox, chatbot, audio_input,
                                 client_session],
                         outputs=[chatbot, client_session, textbox,
                                  audio_input, tts_audio])
            textbox.submit(self.on_user_input,
                           inputs=[textbox, chatbot, audio_input,
                                   client_session],
                           outputs=[chatbot, client_session, textbox,
                                    audio_input, tts_audio])
            # with gradio.Row():
            #     tts_button = gradio.Button("Play TTS")
            #     tts_button.click(self.play_tts,
            #                      inputs=[client_session],
            #                      outputs=[tts_audio, client_session])
            # Define settings UI
            with gradio.Row():
                submit = gradio.Button("Update User Settings")
                check_alerts = gradio.Button("Check for Alerts")
            with gradio.Row():
                with gradio.Column():
                    lang = self.get_lang(client_session.value)
                    stt_lang = gradio.Radio(label="Input Language",
                                            choices=self._languages.get("stt")
                                            or self.supported_languages,
                                            value=lang)
                    tts_lang = gradio.Radio(label="Response Language",
                                            choices=self._languages.get("tts")
                                            or self.supported_languages,
                                            value=lang)
                    tts_lang_2 = gradio.Radio(label="Second Response Language",
                                              choices=[None] +
                                              (self._languages.get("tts") or
                                               self.supported_languages),
                                              value=None)
                with gradio.Column():
                    time_format = gradio.Radio(label="Time Format",
                                               choices=[12, 24],
                                               value=12)
                    date_format = gradio.Radio(label="Date Format",
                                               choices=["MDY", "YMD", "DMY",
                                                        "YDM"],
                                               value="MDY")
                    unit_of_measure = gradio.Radio(label="Units of Measure",
                                                   choices=["imperial",
                                                            "metric"],
                                                   value="imperial")
                with gradio.Column():
                    city = gradio.Textbox(label="City")
                    state = gradio.Textbox(label="State")
                    country = gradio.Textbox(label="Country")
                with gradio.Column():
                    first_name = gradio.Textbox(label="First Name")
                    middle_name = gradio.Textbox(label="Middle Name")
                    last_name = gradio.Textbox(label="Last Name")
                    pref_name = gradio.Textbox(label="Preferred Name")
                    email_addr = gradio.Textbox(label="Email Address")
                    # TODO: DoB, pic, about, phone?
            submit.click(self.update_profile,
                         inputs=[stt_lang, tts_lang, tts_lang_2, time_format,
                                 date_format, unit_of_measure, city, state,
                                 country, first_name, middle_name, last_name,
                                 pref_name, email_addr, client_session],
                         outputs=[client_session])
            check_alerts.click(self.check_alerts, inputs=[client_session],
                               outputs=[client_session])
            app = FastAPI()

            def _health_check(*_):
                if self.connection.check_health():
                    return Response(status_code=200, content="Ready")
                elif not self.connection.ready:
                    return Response(status_code=200, content="Starting")
                else:
                    return Response(status_code=500, content="Error")

            app.add_route("/status", _health_check, methods=["GET"])
            gr_app = gradio.mount_gradio_app(app, blocks, '')
            uvicorn.run(gr_app, host=address, port=port)

    def handle_klat_response(self, message: Message):
        """
        Handle a valid response from Neon. This includes text and base64-encoded
        audio in all requested languages.
        @param message: Neon response message
        """
        response = NeonTtsResponse(msg_type=message.msg_type,
                                   data=message.data,
                                   context=message.context)
        LOG.debug(f"gradio context={response.context.gradio}")
        resp_data = response.data.responses
        files = []
        sentences = []
        session = response.context.gradio.session
        LOG.info(f"Got response for session {session}")
        for lang, response in resp_data.items():
            LOG.debug(f"Response for {lang}: {response}")
            sentences.append(response.sentence)
            if response.audio:
                for gender, data in response.audio.items():
                    audio_path = getattr(response, gender)
                    LOG.debug(f"Got audio file: {audio_path}")
                    filepath = "/".join([self.audio_cache_dir] +
                                        audio_path.split('/')[-4:])
                    # TODO: This only plays the most recent, so it doesn't
                    #  support multiple languages or multi-utterance responses
                    self._current_tts[session] = filepath
                    files.append(filepath)
                    if not isfile(filepath):
                        decode_base64_string_to_file(data, filepath)
        self._response = "\n".join(sentences)
        self._await_response.set()

    def handle_complete_intent_failure(self, message: Message):
        """
        Handle an intent failure response from Neon. This should not happen and
        indicates the Neon service is probably not yet ready.
        @param message: Neon intent failure response message
        """
        self._response = "ERROR"
        self._await_response.set()

    def handle_api_response(self, message: Message):
        """
        Catch-all handler for `.response` messages routed to this client that
        are not explicitly handled (i.e. get_stt, get_tts)
        @param message: Response message to something emitted by this client
        """
        LOG.debug(f"Got {message.msg_type}: {message.data}")
        if message.msg_type == "neon.audio_input.response":
            self._transcribed = message.data.get("transcripts", [""])[0]

    def handle_alert(self, message: Message):
        """
        Handle an expired alert that was previously set by this session.
        @param message: neon.alert_expired Message
        """
        user = message.context['username']
        LOG.info(f"Alert expired for user: {user}")
        alert = f"Alert Expired: {message.data.get('alert_name')}"
        self._alerts.setdefault(user, list())
        self._alerts[user].append(alert)

    def _handle_profile_update(self, message: Message):
        updated_profile = message.data["profile"]
        session_id = updated_profile['user']['username']
        if session_id in self._profiles:
            LOG.info(f"Got profile update for {session_id}")
            self._profiles[session_id] = updated_profile
        else:
            LOG.warning(f"Ignoring profile update for {session_id}")

    def handle_error_response(self, message: Message):
        """
        Handle an error response from the MQ service attached to Neon. This
        usually indicates a malformed input.
        @param message: Response message indicating reason for failure
        """
        LOG.error(f"Error response: {message.data}")

    def clear_caches(self, message: Message):
        """
        Handle a request from Neon to clear cached data.
        @param message: Message requesting cache deletion. The context of this
            message will include the requesting user for user-specific caches
        """
        # TODO: remove cached TTS audio responses

    def clear_media(self, message: Message):
        """
        Handle a request from Neon to clear local multimedia. This method does
        not apply to this client as there is no user-generated media to clear.
        @param message: Message requesting media deletion
        """
        pass
