"""
    lager.defaults.commands

    Default settings management commands
"""
import click
from texttable import Texttable
from ..config import read_config_file, write_config_file
from ..dut_storage import load_duts


@click.group(invoke_without_command=True)
@click.pass_context
def defaults(ctx):
    """
        Manage default settings
    """
    if ctx.invoked_subcommand is None:
        # Default behavior: list defaults
        config = read_config_file()

        if not config.has_section('LAGER') or not config.options('LAGER'):
            click.echo("No defaults are currently set. Add one with: lager defaults add")
            return

        table = Texttable()
        table.set_deco(Texttable.HEADER)
        table.set_cols_dtype(['t', 't'])
        table.set_cols_align(["l", "l"])
        table.add_row(['Default', 'Value'])

        for key in config.options('LAGER'):
            value = config.get('LAGER', key)
            # Make the display name more user-friendly
            if key == 'gateway_id':
                display_name = 'dut'
            elif key == 'serial_device':
                display_name = 'serial-port'
            else:
                display_name = key.replace('_', '-')
            table.add_row([display_name, value])

        click.echo(table.draw())


@defaults.command()
@click.option('--dut', required=False, help='Set default DUT')
@click.option('--serial-port', required=False, help='Set default serial port path')
def add(dut, serial_port):
    """
        Set default values
    """
    # Check if at least one option is provided
    if dut is None and serial_port is None:
        click.echo(click.style("Error: You must specify at least one setting (--dut or --serial-port)", fg='red'), err=True)
        raise click.Abort()

    config = read_config_file()
    changes = []

    # Handle DUT default
    if dut is not None:
        # Validate that the DUT exists in saved DUTs
        duts = load_duts()
        if dut not in duts:
            click.secho(f"Error: DUT '{dut}' does not exist in saved DUTs.", fg='red', err=True)
            click.echo("Available DUTs:", err=True)
            if duts:
                for name in duts.keys():
                    click.echo(f"  - {name}", err=True)
            else:
                click.echo("  (No DUTs saved)", err=True)
            click.echo(f"\nUse 'lager duts add --name <name> --ip <ip>' to add a new DUT first.", err=True)
            raise click.Abort()

        config.set('LAGER', 'gateway_id', dut)
        changes.append(f"dut: {dut}")

    # Handle serial port default
    if serial_port is not None:
        config.set('LAGER', 'serial_device', serial_port)
        changes.append(f"serial-port: {serial_port}")

    write_config_file(config)

    # Display success message
    if len(changes) == 1:
        click.secho(f"Set default {changes[0]}", fg='green')
    else:
        click.secho(f"Set defaults:", fg='green')
        for change in changes:
            click.echo(f"  - {change}")


@defaults.command('list')
@click.pass_context
def list_defaults(ctx):
    """
        List default settings
    """
    # Reuse the default behavior
    ctx.invoke(defaults)


@defaults.group()
def delete():
    """
        Delete default settings
    """
    pass


@delete.command()
@click.option('--yes', is_flag=True, help='Skip confirmation prompt')
def dut(yes):
    """
        Delete default DUT
    """
    config = read_config_file()

    # Check if gateway_id is set
    if not config.has_option('LAGER', 'gateway_id'):
        click.echo("No default DUT is currently set.")
        return

    # Get value to display
    value = config.get('LAGER', 'gateway_id')

    # Display what will be deleted and ask for confirmation unless --yes flag is provided
    if not yes:
        click.echo(f"\nYou are about to delete the following default:")
        click.echo(f"  dut: {value}")
        click.echo()

        if not click.confirm("Delete this default?", default=False):
            click.echo("Cancelled. Default not deleted.")
            return

    # Remove the gateway_id option
    config.remove_option('LAGER', 'gateway_id')

    write_config_file(config)
    click.secho(f"Deleted default DUT '{value}'", fg='green')


@delete.command()
@click.option('--yes', is_flag=True, help='Skip confirmation prompt')
def serial_port(yes):
    """
        Delete default serial port
    """
    config = read_config_file()

    # Check if serial_device is set
    if not config.has_option('LAGER', 'serial_device'):
        click.echo("No default serial port is currently set.")
        return

    # Get value to display
    value = config.get('LAGER', 'serial_device')

    # Display what will be deleted and ask for confirmation unless --yes flag is provided
    if not yes:
        click.echo(f"\nYou are about to delete the following default:")
        click.echo(f"  serial-port: {value}")
        click.echo()

        if not click.confirm("Delete this default?", default=False):
            click.echo("Cancelled. Default not deleted.")
            return

    # Remove the serial_device option
    config.remove_option('LAGER', 'serial_device')

    write_config_file(config)
    click.secho(f"Deleted default serial port '{value}'", fg='green')


@defaults.command('delete-all')
@click.option('--yes', is_flag=True, help='Skip confirmation prompt')
def delete_all(yes):
    """
        Delete all defaults
    """
    config = read_config_file()

    # Check if there are any defaults set
    if not config.has_section('LAGER') or not config.options('LAGER'):
        click.echo("No defaults are currently set.")
        return

    # Count the defaults
    defaults_list = []
    for key in config.options('LAGER'):
        value = config.get('LAGER', key)
        if key == 'gateway_id':
            display_name = 'dut'
        elif key == 'serial_device':
            display_name = 'serial-port'
        else:
            display_name = key.replace('_', '-')
        defaults_list.append((display_name, value))

    defaults_count = len(defaults_list)

    # Display warning and defaults list
    click.echo(click.style(f"\n⚠ WARNING: You are about to delete ALL {defaults_count} default(s):", fg='yellow', bold=True))
    click.echo()
    for display_name, value in defaults_list:
        click.echo(f"  - {display_name}: {value}")
    click.echo()

    # Ask for confirmation unless --yes flag is provided (default is No)
    if not yes and not click.confirm("Are you sure you want to delete ALL defaults?", default=False):
        click.echo("Cancelled. No defaults were deleted.")
        return

    # Remove all options from LAGER section
    config.remove_section('LAGER')
    config.add_section('LAGER')

    write_config_file(config)
    click.secho(f"✓ Deleted all {defaults_count} default(s)", fg='green')
