import os
from pathlib import Path
import json
import click
from ..context import get_default_gateway
from ..python.commands import run_python_internal

def _impl_gpio_path() -> str:
    return str(Path(__file__).resolve().parent.parent / "impl" / "gpio.py")

@click.command(name="gpo", help="Set GPIO output level (0/1, low/high, off/on, toggle)")
@click.pass_context
@click.option("--gateway", required=False, hidden=True, help="DUT or Gateway IP")
@click.option("--dut", required=False, help="DUT or Gateway IP")
@click.argument("netname", required=True)
@click.argument("level", required=True,
                type=click.Choice(["low", "high", "on", "off", "0", "1", "toggle"], case_sensitive=False))
def gpo(ctx, gateway, dut, netname, level):
    from ..dut_storage import get_dut_ip

    # Check if gateway is a local DUT name first
    if gateway:
        local_ip = get_dut_ip(gateway)
        if local_ip:
            gateway = local_ip
    elif dut:
        local_ip = get_dut_ip(dut)
        if local_ip:
            gateway = local_ip
        else:
            gateway = dut
    else:
        gateway = get_default_gateway(ctx)
    gateway_image = os.environ.get("LAGER_GATEWAY_IMAGE", "python")

    payload = json.dumps({"netname": netname, "action": "output", "level": level})

    run_python_internal(
        ctx=ctx,
        runnable=_impl_gpio_path(),
        gateway=gateway,
        image=gateway_image,
        env=(),
        passenv=(),
        kill=False,
        download=(),
        allow_overwrite=False,
        signum="SIGTERM",
        timeout=None,
        detach=False,
        port=(),
        org=None,
        args=[payload],
    )