# This code is a Qiskit project.
#
# (C) Copyright IBM 2025.
#
# This code is licensed under the Apache License, Version 2.0. You may
# obtain a copy of this license in the LICENSE.txt file in the root directory
# of this source tree or at http://www.apache.org/licenses/LICENSE-2.0.
#
# Any modifications or derivative works of this code must retain this
# copyright notice, and modified files need to carry a notice indicating
# that they have been altered from the originals.

"""AddInjectNoise"""

import itertools
from collections import defaultdict

from qiskit.circuit import CircuitInstruction
from qiskit.dagcircuit import DAGCircuit
from qiskit.transpiler.basepasses import TransformationPass

from ...annotations import InjectNoise, Twirl
from ...utils import BoxKey, get_annotation, undress_box
from ..noise_injection_strategies import NoiseInjectionStrategy, NoiseInjectionStrategyLiteral


class AddInjectNoise(TransformationPass):
    """Inserts :class:`~.InjectNoise` annotations to all the unique boxes with twirling annotation.

    This pass finds all the twirled-annotated boxes in the given circuit and adds inject noise
    annotations to all the boxes that either contain entanglers or own classical registers.

    Args:
        strategy: The noise injection strategy.
        overwrite: Whether to overwrite the ``ref`` of inject noise annotations that are already
            present in the circuit.
        prefix_ref: A prefix to all the ``ref`` generated by this class.
        prefix_modifier_ref: A prefix to all the ``modifier_refs`` generated by this class.

    .. note::
        Every ``ref`` is generated by appending a counter to ``prefix_ref``. The counter is shared
        across all the instances of this class. In order to avoid collisions, it is incremented
        every time that a new ``ref`` is created. The same mechanism (with a different counter)
        is used to generate the ``modifier_refs``.
    """

    _REF_COUNTER = itertools.count()
    _MODIFIER_REF_COUNTER = itertools.count()

    def __init__(
        self,
        strategy: NoiseInjectionStrategyLiteral = "none",
        overwrite: bool = False,
        prefix_ref: str = "r",
        prefix_modifier_ref: str = "m",
    ):
        TransformationPass.__init__(self)
        self.strategy = NoiseInjectionStrategy(strategy)
        self.overwrite = overwrite
        self.prefix_ref = prefix_ref
        self.prefix_modifier_ref = prefix_modifier_ref

    def run(self, dag: DAGCircuit) -> DAGCircuit:
        if self.strategy == NoiseInjectionStrategy.NONE:
            return dag

        # A dictionary to map unique boxes to their ``ref``
        box_to_ref: dict[BoxKey, int] = defaultdict(
            lambda: f"{self.prefix_ref}{next(self._REF_COUNTER)}"
        )

        for node in dag.op_nodes():
            if node.op.name == "box" and (twirl := get_annotation(node.op, Twirl)):
                undressed_box = undress_box(node.op)
                undressed_box.annotations = [twirl]
                if undressed_box.body.num_nonlocal_gates() == 0 and not undressed_box.body.clbits:
                    # Skip boxes that do not contain entanglers or measurements
                    continue

                instr_qubits = undressed_box.body.qubits
                instr_clbits = undressed_box.body.clbits
                instr = CircuitInstruction(undressed_box, instr_qubits, instr_clbits)
                box_key = BoxKey(instr)

                if inject_noise_annotation := get_annotation(node.op, InjectNoise):
                    if box_key not in box_to_ref:
                        # The box already has a noise injection annotation, but it is the first box
                        # to be found for this hash. Its `ref` will be reused for all future boxes
                        # that produce the same hash.
                        box_to_ref[box_key] = inject_noise_annotation.ref
                    else:
                        # The box already has a noise injection annotation, and another box with the
                        # same hash has previously been found.
                        if self.overwrite is True:
                            annotation_idx = node.op.annotations.index(inject_noise_annotation)
                            node.op.annotations[annotation_idx] = InjectNoise(
                                box_to_ref[box_key],
                                inject_noise_annotation.modifier_ref,
                            )
                else:
                    # The box does not have a noise injection annotation.
                    ref = box_to_ref[box_key]

                    if self.strategy == NoiseInjectionStrategy.NO_MODIFICATION:
                        modifier_ref = ""
                    elif self.strategy == NoiseInjectionStrategy.UNIFORM_MODIFICATION:
                        modifier_ref = ref
                    else:
                        # individual modification
                        modifier_ref = (
                            f"{self.prefix_modifier_ref}{next(self._MODIFIER_REF_COUNTER)}"
                        )

                    node.op.annotations += [InjectNoise(ref, modifier_ref)]
        return dag
