##############################################################################
#
# Copyright (c) 2010 Vifib SARL and Contributors. All Rights Reserved.
#
# WARNING: This program as such is intended to be used by professional
# programmers who take the whole responsibility of assessing all potential
# consequences resulting from its eventual inadequacies and bugs
# End users who are looking for a ready-to-use solution with commercial
# guarantees and support are strongly adviced to contract a Free Software
# Service Company
#
# This program is Free Software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
##############################################################################

import os, shlex
from slapos.recipe.librecipe import GenericBaseRecipe, generateHashFromFiles
from slapos.recipe.librecipe.execute import parse_signal
from zc.buildout import UserError

class Recipe(GenericBaseRecipe):
    """Recipe to create a script from given command and options.

    :param str command-line: shell command which launches the intended process
    :param str wrapper-path: absolute path to file's destination
    :param lines wait-for-files: list of files to wait for
    :param lines hash-files: list of buildout-generated files to be checked by hash
    :param lines hash-existing-files: list of existing files to be checked by hash
    :param str pidfile: path to pidfile ensure exclusivity for the process
    :param lines private-tmpfs: list of "<size> <path>" private tmpfs, using user namespaces
    :param bool reserve-cpu: command will ask for an exclusive CPU core
    :param str sig-ign: see slapos.recipe.librecipe.execute.generic_exec
    :param str redirect-{signal,stdout,stderr}: set 'redirect' param
        to slapos.recipe.librecipe.execute.generic_exec
    """

    _existing = ()

    def __init__(self, buildout, name, options):
        self.buildout = buildout
        self.options = options
        hash_files = options.get('hash-files')
        if hash_files:
          self.hash_files = hash_files.split()
          prefix = os.path.realpath(buildout['buildout']['directory']) + os.sep
          self._existing = [x for x in self.hash_files
              if os.path.exists(x) and not os.path.realpath(x).startswith(prefix)]
        else:
          self.hash_files = []
        hash_files = options.get('hash-existing-files')
        if hash_files:
          hash_files = hash_files.split()
          options['__hash_files__'] = generateHashFromFiles(hash_files)
          self.hash_files += hash_files
        sig = options.get('sig-ign')
        if sig:
          parse_signal(sig)
          self.sig_ign = sig
        redirect = tuple(options.get('redirect-' + x)
                         for x in ('signal', 'stdout', 'stderr',))
        if any(redirect):
          sig = redirect[0]
          if sig:
            parse_signal(sig)
            if not any(redirect[1:]):
              raise UserError(
                "redirect-signal without redirecting any output makes no sense")
          self.redirect = redirect

    def getWrapperPath(self):
        wrapper_path = self.options['wrapper-path']
        if self.hash_files:
          wrapper_path += '-' + generateHashFromFiles(self.hash_files)
        return wrapper_path

    def install(self):
        if self._existing:
          raise UserError(
            "hash-files must only list files that are generated by buildout:"
            "\n  " + "\n  ".join(self._existing))
        options = self.options
        args = shlex.split(options['command-line'])
        wait_files = options.get('wait-for-files')
        pidfile = options.get('pidfile')
        private_tmpfs = self.parsePrivateTmpfs()

        environment = {}
        for line in (options.get('environment') or '').splitlines():
          line = line.strip()
          if line:
            k, v = line.split('=', 1)
            environment[k.rstrip()] = v.lstrip()

        kw = {}
        if wait_files:
          kw['wait_list'] = wait_files.split()
        if pidfile:
          kw['pidfile'] = pidfile
        if private_tmpfs:
          kw['private_tmpfs'] = private_tmpfs
        if self.isTrueValue(options.get('reserve-cpu')):
          kw['reserve_cpu'] = True
        for x in 'redirect', 'sig_ign':
          try:
            kw[x] = getattr(self, x)
          except AttributeError:
            pass
        return self.createWrapper(self.getWrapperPath(),
                                  args, environment, **kw)

    def update(self):
        wrapper_path = self.getWrapperPath()
        if not os.path.isfile(wrapper_path):
          raise UserError("unstable wrapper path (%r)" % wrapper_path)
