
# Implementation Plan: Write-Only Freewriting Interface

**Branch**: `003-write-only-freewriting` | **Date**: 2025-09-24 | **Spec**: [spec.md](./spec.md)
**Input**: Feature specification from `/specs/003-write-only-freewriting/spec.md`

## Execution Flow (/plan command scope)
```
1. Load feature spec from Input path
   → If not found: ERROR "No feature spec at {path}"
2. Fill Technical Context (scan for NEEDS CLARIFICATION)
   → Detect Project Type from context (web=frontend+backend, mobile=app+api)
   → Set Structure Decision based on project type
3. Fill the Constitution Check section based on the content of the constitution document.
4. Evaluate Constitution Check section below
   → If violations exist: Document in Complexity Tracking
   → If no justification possible: ERROR "Simplify approach first"
   → Update Progress Tracking: Initial Constitution Check
5. Execute Phase 0 → research.md
   → If NEEDS CLARIFICATION remain: ERROR "Resolve unknowns"
6. Execute Phase 1 → contracts, data-model.md, quickstart.md, agent-specific template file (e.g., `CLAUDE.md` for Claude Code, `.github/copilot-instructions.md` for GitHub Copilot, `GEMINI.md` for Gemini CLI, `QWEN.md` for Qwen Code or `AGENTS.md` for opencode).
7. Re-evaluate Constitution Check section
   → If new violations: Refactor design, return to Phase 1
   → Update Progress Tracking: Post-Design Constitution Check
8. Plan Phase 2 → Describe task generation approach (DO NOT create tasks.md)
9. STOP - Ready for /tasks command
```

**IMPORTANT**: The /plan command STOPS at step 7. Phases 2-4 are executed by other commands:
- Phase 2: /tasks command creates tasks.md
- Phase 3-4: Implementation execution (manual or via tools)

## Summary
A write-only freewriting interface that provides a distraction-free TUI environment where users can capture thoughts without editing capability. Content is saved either to specified UUID nodes or timestamped daily files in the current directory, implementing a textual UI with Textual framework.

## Technical Context
**Language/Version**: Python 3.13 (from existing project context)
**Primary Dependencies**: Textual (TUI framework), Typer (existing CLI), prosemark domain (existing)
**Storage**: Plain text files (Markdown with YAML frontmatter per constitution)
**Testing**: pytest (existing test framework)
**Target Platform**: Linux/macOS/Windows terminal environments
**Project Type**: single (CLI-based tool with TUI component)
**Performance Goals**: Real-time text display updates, instant save on ENTER
**Constraints**: Terminal-based UI, readline-style editing in input box only
**Scale/Scope**: Single user, local file operations, unlimited session duration

## Constitution Check
*GATE: Must pass before Phase 0 research. Re-check after Phase 1 design.*

- [x] **Hexagonal Architecture**: TUI adapter separated from domain logic via ports
- [x] **Test-First Development**: All code requires tests written first (TDD mandatory)
- [x] **Plain Text Storage**: Markdown files with YAML frontmatter, Obsidian-compatible
- [x] **Code Quality Standards**: 100% mypy, ruff, test coverage required (enforced via agents)
- [x] **CLI-First Interface**: `pmk write` command with text in/out protocol

## Project Structure

### Documentation (this feature)
```
specs/[###-feature]/
├── plan.md              # This file (/plan command output)
├── research.md          # Phase 0 output (/plan command)
├── data-model.md        # Phase 1 output (/plan command)
├── quickstart.md        # Phase 1 output (/plan command)
├── contracts/           # Phase 1 output (/plan command)
└── tasks.md             # Phase 2 output (/tasks command - NOT created by /plan)
```

### Source Code (repository root)
```
# Option 1: Single project (DEFAULT)
src/
├── models/
├── services/
├── cli/
└── lib/

tests/
├── contract/
├── integration/
└── unit/

# Option 2: Web application (when "frontend" + "backend" detected)
backend/
├── src/
│   ├── models/
│   ├── services/
│   └── api/
└── tests/

frontend/
├── src/
│   ├── components/
│   ├── pages/
│   └── services/
└── tests/

# Option 3: Mobile + API (when "iOS/Android" detected)
api/
└── [same as backend above]

ios/ or android/
└── [platform-specific structure]
```

**Structure Decision**: Option 1 (Single project - CLI tool with TUI component)

## Phase 0: Outline & Research
1. **Extract unknowns from Technical Context** above:
   - For each NEEDS CLARIFICATION → research task
   - For each dependency → best practices task
   - For each integration → patterns task

2. **Generate and dispatch research agents**:
   ```
   For each unknown in Technical Context:
     Task: "Research {unknown} for {feature context}"
   For each technology choice:
     Task: "Find best practices for {tech} in {domain}"
   ```

3. **Consolidate findings** in `research.md` using format:
   - Decision: [what was chosen]
   - Rationale: [why chosen]
   - Alternatives considered: [what else evaluated]

**Output**: research.md with all NEEDS CLARIFICATION resolved

## Phase 1: Design & Contracts
*Prerequisites: research.md complete*

1. **Extract entities from feature spec** → `data-model.md`:
   - Entity name, fields, relationships
   - Validation rules from requirements
   - State transitions if applicable

2. **Generate API contracts** from functional requirements:
   - For each user action → endpoint
   - Use standard REST/GraphQL patterns
   - Output OpenAPI/GraphQL schema to `/contracts/`

3. **Generate contract tests** from contracts:
   - One test file per endpoint
   - Assert request/response schemas
   - Tests must fail (no implementation yet)

4. **Extract test scenarios** from user stories:
   - Each story → integration test scenario
   - Quickstart test = story validation steps

5. **Update agent file incrementally** (O(1) operation):
   - Run `.specify/scripts/bash/update-agent-context.sh claude` for your AI assistant
   - If exists: Add only NEW tech from current plan
   - Preserve manual additions between markers
   - Update recent changes (keep last 3)
   - Keep under 150 lines for token efficiency
   - Output to repository root

**Output**: data-model.md, /contracts/*, failing tests, quickstart.md, agent-specific file

## Phase 2: Task Planning Approach
*This section describes what the /tasks command will do - DO NOT execute during /plan*

**Task Generation Strategy**:
- Load `.specify/templates/tasks-template.md` as base
- Generate tasks from Phase 1 design docs (contracts, data model, quickstart)
- **Contract Tests**: Each port interface → failing contract test [P]
  - `FreewriteServicePort` → contract test for domain operations
  - `TUIAdapterPort` → contract test for UI interactions
  - `CLIAdapterPort` → contract test for command handling
  - `NodeServicePort` → contract test for node management
  - `FileSystemPort` → contract test for file operations
- **Domain Models**: Each entity → model class with validation [P]
  - `FreewriteSession` dataclass with validation
  - `SessionConfig` dataclass with validation
  - Domain exceptions hierarchy
- **Integration Tests**: Each quickstart scenario → integration test
  - Daily file creation scenario
  - Node targeting scenario
  - Error handling scenarios
  - Multiple session scenarios
- **Adapter Implementations**: Implementation tasks to make tests pass
  - Textual TUI adapter implementation
  - Typer CLI adapter implementation
  - File system adapter implementation
  - Node service adapter implementation

**Ordering Strategy**:
- **Phase A** (Parallel): Contract tests for all ports [P]
- **Phase B** (Parallel): Domain models and exceptions [P]
- **Phase C** (Sequential): Integration test scenarios (depend on models)
- **Phase D** (Parallel): Adapter implementations [P] (one per port)
- **Phase E** (Sequential): CLI command integration (depends on all adapters)
- **Phase F** (Sequential): End-to-end validation and quickstart verification

**Parallelization Markers**:
- [P] = Can run in parallel with other [P] tasks in same phase
- Sequential phases build on previous phase completion
- Within phases, [P] tasks are independent

**Constitutional Compliance Integration**:
- Every task includes quality gate validation (mypy, ruff, tests)
- TDD enforced: test task always precedes implementation task
- Code quality agents automatically run after each implementation
- No task marked complete until 100% quality compliance

**Estimated Output**:
- 5 contract test tasks [P]
- 3 domain model tasks [P]
- 6 integration test tasks
- 4 adapter implementation tasks [P]
- 2 CLI integration tasks
- 3 validation/cleanup tasks
- **Total**: ~23 numbered, ordered tasks in tasks.md

**Dependencies Mapped**:
- Contract tests → Domain models → Integration tests → Adapters → CLI → Validation
- Clear dependency chain enables proper TDD workflow
- Parallel opportunities maximize implementation efficiency

**IMPORTANT**: This phase is executed by the /tasks command, NOT by /plan

## Phase 3+: Future Implementation
*These phases are beyond the scope of the /plan command*

**Phase 3**: Task execution (/tasks command creates tasks.md)
**Phase 4**: Implementation (execute tasks.md following constitutional principles)
**Phase 5**: Validation (run tests, execute quickstart.md, performance validation)

## Complexity Tracking
*Fill ONLY if Constitution Check has violations that must be justified*

| Violation | Why Needed | Simpler Alternative Rejected Because |
|-----------|------------|-------------------------------------|
| [e.g., 4th project] | [current need] | [why 3 projects insufficient] |
| [e.g., Repository pattern] | [specific problem] | [why direct DB access insufficient] |


## Progress Tracking
*This checklist is updated during execution flow*

**Phase Status**:
- [x] Phase 0: Research complete (/plan command)
- [x] Phase 1: Design complete (/plan command)
- [x] Phase 2: Task planning complete (/plan command - describe approach only)
- [ ] Phase 3: Tasks generated (/tasks command)
- [ ] Phase 4: Implementation complete
- [ ] Phase 5: Validation passed

**Gate Status**:
- [x] Initial Constitution Check: PASS
- [x] Post-Design Constitution Check: PASS
- [x] All NEEDS CLARIFICATION resolved
- [x] Complexity deviations documented

---
*Based on Constitution v1.1.0 - See `.specify/memory/constitution.md`*
