# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['libvirt_instance']

package_data = \
{'': ['*'],
 'libvirt_instance': ['domain-presets/headless-server-aarch64.xml',
                      'domain-presets/headless-server-aarch64.xml',
                      'domain-presets/headless-server-x86_64.xml',
                      'domain-presets/headless-server-x86_64.xml']}

install_requires = \
['PyYAML>=5.4.1,<6.0.0',
 'libvirt-python>=8.0.0,<9.0.0',
 'pycdlib>=1.13.0,<2.0.0']

entry_points = \
{'console_scripts': ['libvirt-instance = libvirt_instance.cli:main']}

setup_kwargs = {
    'name': 'libvirt-instance',
    'version': '0.1.1',
    'description': 'VM provisioning and management tool',
    'long_description': '# libvirt-instance\n\n`libvirt-instance` is a CLI tool for creating virtual machines. It uses the\nLibvirt API, and is compatible with other Libvirt applications.\n\nThink of it as a more opinionated alternative to `virt-install`.\n\n\n## Project goals\n\n- Make creating Libvirt virtual machines (VMs) via the command-line interface\n  simpler for human operators, by providing a way to group some commonly-used\n  configurations as presets.\n- Provide a convenient way to deploy cloud instances on Libvirt without the\n  need for a metadata service (the\n  [NoCloud](https://cloudinit.readthedocs.io/en/latest/topics/datasources/nocloud.html)\n  data source is used to provide metadata to instances).\n\nThe tool does not support deleting Libvirt resources, and there are no plans to\nimplement that so far. Any delete operations can be performed using `virsh`.\n\n\n## Installation\n\nThis package depends on `libvirt-python`. Installing into a virtualenv might\nrequire GCC, Python3 and Libvirt development OS packages to be installed.\nRun `dnf --enablerepo=devel install gcc python3-devel libvirt-devel` to install\nthose packages if you\'re using RockyLinux 9.\n\nOtherwise (when not using virtualenv), having the `python3-libvirt` OS package\ninstalled should be sufficient.\n\n`libvirt-instance` can be installed by running `pip3 install libvirt-instance`.\nIt requires Python 3.9+ to run.\n\n\n## Showcase\n\nWhile both local and remote Libvirt daemons are supported, the following example\nis using a local Libvirt daemon for sake of simplicity. The commands should be\nexecuted by a user who has sufficient Libvirt access privileges.\n\n```shell\nURI=qemu:///system\n\n# All operations on disks are done using the Libvirt pool APIs.\n# Libvirt doesn\'t usually come with any storage pools defined, so let\'s define\n# one named "images".\nvirsh -c "$URI" pool-define-as images dir --target /var/lib/libvirt/images\nvirsh -c "$URI" pool-autostart images\nvirsh -c "$URI" pool-start images\n\n# Create a config file with a preset for disks from the above pool,\n# and a preset for network interfaces in the default Libvirt NAT network (this\n# network exists by default).\ncat <<"EOF" >./libvirt-instance-config.yaml\npreset:\n  disk:\n    local:\n      type: volume\n      pool: images\n      bus: virtio\n      cache: none\n  interface:\n    nat:\n      type: network\n      model-type: virtio\n      network: default\nEOF\n\n# Fetch a cloud image from the Internet and upload it to Libvirt as a volume,\n# so we can use it as the base image for VM disks.\ncurl -LfsS \\\n  https://download.fedoraproject.org/pub/fedora/linux/releases/37/Cloud/x86_64/images/Fedora-Cloud-Base-37-1.7.x86_64.raw.xz \\\n  | xzcat >./f37-cloud-amd64.raw\nimage_size=$(stat --printf="%s" ./f37-cloud-amd64.raw)\nvirsh -c "$URI" vol-create-as images f37-cloud-amd64.raw "${image_size}b" --format raw\nvirsh -c "$URI" vol-upload f37-cloud-amd64.raw ./f37-cloud-amd64.raw --pool images\n\n# Generate a passphraseless SSH key for this demo.\nssh-keygen -f mykey -N \'\'\n\n# Create user-data.\ncat <<EOF >./user-data\n#cloud-config\nssh_authorized_keys:\n  - $(cat ./mykey.pub)\npackages:\n  - nginx\nruncmd:\n  - systemctl start nginx\nEOF\n\n# Create network-config.\ncat <<"EOF" >./network-config\nversion: 2\nethernets:\n    eth0:\n        dhcp4: false\n        dhcp6: false\n        addresses:\n          - 192.168.122.10/24\n        gateway4: 192.168.122.1\n        nameservers:\n          addresses:\n            - 1.1.1.1\n            - 8.8.8.8\nEOF\n\n# Create the VM.\n# headless-server-x86_64 is a built-in domain preset.\ninstance_id=$(\n  libvirt-instance -c "$URI" --config-file ./libvirt-instance-config.yaml create \\\n                   --domain-preset headless-server-x86_64 \\\n                   --memory 2GiB \\\n                   --vcpu 2 \\\n                   --disk local,5GiB,source=f37-cloud-amd64.raw \\\n                   --nic nat \\\n                   --cloud-seed-disk=local \\\n                   --cloud-user-data-file ./user-data \\\n                   --cloud-network-config-file ./network-config \\\n                   myvm \\\n    | jq -er \'."instance-id"\')\n\n# Start the VM.\nvirsh -c "$URI" start "$instance_id"\n\n# Wait until cloud-init has finished executing.\nuntil\n  ssh -i mykey \\\n      -o IdentitiesOnly=true \\\n      -o StrictHostKeyChecking=no \\\n      -o UserKnownHostsFile=/dev/null \\\n      fedora@192.168.122.10 \\\n      cloud-init status --wait\ndo\n  sleep 1\ndone\n\n# Get a page from Nginx on the VM.\ncurl http://192.168.122.10/\n\n# Cleanup.\nvirsh -c "$URI" destroy "$instance_id"\nvirsh -c "$URI" undefine "$instance_id" --nvram --remove-all-storage\n```\n\n\n## More examples\n\nCreating non-image-based VMs is also an option.\n\nThis example creates a VM with two disks and the PXE boot as the first option:\n\n```shell\nlibvirt-instance -c "$URI" --config-file ./libvirt-instance-config.yaml create \\\n                 --domain-preset headless-server-x86_64 \\\n                 --memory 2GiB \\\n                 --vcpu 2 \\\n                 --disk local,5GiB,boot-order=2 \\\n                 --disk local,10GiB \\\n                 --nic nat,boot-order=1 \\\n                 myvm\n```\n\nAlternative/non-native architectures are also supported. `libvirt-instance` comes\nwith two built-in domain presets - `headless-server-x86_64` and\n`headless-server-aarch64`. More presets can be defined in a config file.\n\nRun `libvirt-instance get-config` to see the currently-defined presets.\n\nThe following example deploys an ARM cloud image-based Fedora VM on a non-ARM KVM host:\n\n```shell\ncurl -LfsS \\\n  https://download.fedoraproject.org/pub/fedora/linux/releases/37/Cloud/aarch64/images/Fedora-Cloud-Base-37-1.7.aarch64.raw.xz \\\n  | xzcat >./f37-cloud-arm64.raw\nimage_size=$(stat --printf="%s" ./f37-cloud-arm64.raw)\nvirsh -c "$URI" vol-create-as images f37-cloud-arm64.raw "${image_size}b" --format raw\nvirsh -c "$URI" vol-upload f37-cloud-arm64.raw ./f37-cloud-arm64.raw --pool images\n\nlibvirt-instance -c "$URI" --config-file ./libvirt-instance-config.yaml create \\\n                 --domain-preset headless-server-aarch64 \\\n                 --cpu-model cortex-a57 \\\n                 --domain-type qemu \\\n                 --memory 2GiB \\\n                 --vcpu 2 \\\n                 --disk local,5GiB,source=f37-cloud-arm64.raw \\\n                 --nic nat \\\n                 --cloud-seed-disk=local \\\n                 --cloud-user-data-file ./user-data \\\n                 --cloud-network-config-file ./network-config \\\n                 myvm\n```\n\n\n## Configuration file\n\nSome defaults and presets are already built in. Configuration file is a way to\nadd more presets, or override existing presets and settings.\n\nThree types of presets are currently supported: `domain`, `disk`, and `interface`.\n\nThe default location of the config file is `/etc/libvirt-instance-config.yaml`.\nThe `--config-file` CLI argument provides a way to override that.\n\nRun `libvirt-instance get-config` to see the current config.\n\n\n### Domain presets\n\nExample config snippet:\n\n```yaml\npreset:\n  domain:\n    windows-server:\n      arch-name: x86_64\n      machine-type: pc\n      xml-file: /path/to/windows-server-base.xml\n```\n\nThe above preset can be selected on the CLI using `--domain-preset windows-server`\nwhen creating a new VM.\n\n`arch-name` can be any arhitecture (e.g. `x86_64`, `aarch64`) supported by the\ntarget host.\n\n`machine-type` can be any machine type (e.g. `pc`, `q35`, `virt`) supported by\nthe chosen architecture.\n\n`xml-file` specifies a file containing some\n[domain XML](https://libvirt.org/formatdomain.html) used as the foundation for\nthe VM being created. The tool will fill in the architecture, domain and\nmachine type, CPU (count and model), memory size, network interface,\ndisk (including any necessary SCSI controllers) entries into the base XML\nautomatically using the information from presets and CLI arguments.\n\nDomain XML may alternatively be provided inline via the `xml` key.\n\n\n### Disk presets\n\nAll operations on disks are done using Libvirt pool APIs, so disk presets may\nonly reference Libvirt pools.\n\nCurrently `dir`, `logical`, and `rbd` pools are supported.\n\nExample config snippet:\n\n```yaml\npreset:\n  disk:\n    ceph-ssd:\n      type: volume\n      pool: ceph-rbd-ssd\n      bus: scsi\n      cache: writeback\n    ceph-hdd:\n      type: volume\n      pool: ceph-rbd\n      bus: virtio\n      cache: writeback\n    x86worker:\n      type: volume\n      pool: local-lvm1\n      bus: virtio\n      cache: none\n      source: fedora37-cloud-amd64.raw\n      source-pool: ceph-rbd\n\n```\nCLI examples for the above: `--disk ceph-ssd,16GiB --disk ceph-hdd,1TiB`,\n`--disk x86worker,32GiB`.\n\nThe only supported value for `type` is `volume` at the moment.\n\n`pool` specifies a target Libvirt pool for the volumes. This pool will also be\nused to pull any information about volumes when adding disk devices to the\ndomain XML.\n\n`bus` is either `virtio` for virtio-blk disks, or `scsi` for virtio-scsi disks.\n\n`cache` specifies any disk cache mode supported by Libvirt.\n\n`source` specifies a Libvirt volume containing the base image for disks.\n\n`source-pool` specifies the Libvirt pool for the `source` image. Defaults to the\nsame value as `pool` when not specified.\n\n`boot-order` sets the device position in the boot order.\n\nIn most cases, the resulting disk device description in the domain XML will be a\nvolume reference (`<disk type="volume">`). Some pool types (`rbd` for instance)\ndo not support backing volume disks yet\n(see [domain_conf.c#L29929-L29939](https://github.com/libvirt/libvirt/blob/v8.10.0/src/conf/domain_conf.c#L29929-L29939).\nWhen adding disks from such pools, `libvirt-instance` will transparently inline\nthe disk definition into the domain XML using the information (MONs, auth)\nfrom the pool.\n\n\n### Interface presets\n\nCurrently `bridge` and `network` type interfaces are supported.\n\nExample config snippet:\n\n```yaml\npreset:\n  interface:\n    nat:\n      type: network\n      model-type: virtio\n      network: default\n    dmz:\n      type: bridge\n      model-type: virtio\n      bridge: br0\n    storage:\n      type: bridge\n      model-type: virtio\n      bridge: br1\n      mtu: 9000\n```\n\nCLI examples for the above: `--nic nat`, `--nic dmz --nic storage`.\n\n`model-type` specifies any model type supported by Libvirt (e.g. `virtio`,\n`e1000`, `rtl8139`, etc).\n\n`network` specifies the Libvirt network name for `type: network` interfaces.\n\n`bridge` specifies the name of an existing network bridge for `type: bridge`\ninterfaces.\n\n`mtu` sets the MTU on the host-side TAP interface. Note, the MTU also needs\nto be configured inside the guest.\n\n`boot-order` sets the device position in the boot order.\n\nTechnically, `mac-address` can also be specified in an interface preset, but\nit makes more sense to specify any MAC addresses on the command line\n(e.g. `--nic nat,mac-address=00:11:22:33:44:55:66`).\n\n\n## CLI options\n\nSee `libvirt-instance --help` for the full list of CLI options.\nThere are also some examples in the "Showcase" and "More examples" sections\nillustrating how different options work together.\n\n`--disk <SPEC>` may be specified more than once to attach multiple disks to a\nVM. Disks are created using the `<VM-NAME>-disk<N>` naming scheme. If a volume\nwith the same name already exists in the target pool - `libvirt-instance` will\nexit with an error.\nDisks are attached to the instance in the same order as specified on the command\nline.\n\nThe `pool`, `bus`, `cache`, `source`, `source-pool`, and `boot-order` options\nmay be included in the disk spec, taking precedence over any corresponding\noptions from the chosen preset.\nFor example:\n`--disk local,10GiB,bus=scsi,cache=writeback,source=jammy-server-cloudimg-amd64.img`.\n\n`--nic <SPEC>` may also be specified more than once. The `mac-address`,\n`model-type`, `network`, `bridge`, `mtu`, and `boot-order` options are supported\nby `--nic`, similar to `--disk` options.\n\n`--cloud-seed-disk` enables the cloud-init support and is required when\neither `--cloud-user-data-file` or `--cloud-network-config-file` have been\nspecified.\nIt works like `--disk` without the size part. `--cloud-seed-disk` specifies\nwhich disk preset to use when creating the\n[NoCloud](https://cloudinit.readthedocs.io/en/latest/topics/datasources/nocloud.html)\nseed disk.\nThe naming pattern for the seed disk is `<VM-NAME>-seed`.\nExample: `--cloud-seed-disk local,bus=scsi`.\n\n\n## Developing\n\nSee [DEVELOPING.md](DEVELOPING.md)\n',
    'author': 'Sergej Alikov',
    'author_email': 'sergej@alikov.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/node13h/libvirt-instance',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
