"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Container = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const awslogs = require("aws-cdk-lib/aws-logs");
const cxapi = require("aws-cdk-lib/cx-api");
const extension_interfaces_1 = require("./extension-interfaces");
/**
 * The main container of a service. This is generally the container
 * which runs your application business logic. Other extensions will attach
 * sidecars alongside this main container.
 */
class Container extends extension_interfaces_1.ServiceExtension {
    constructor(props) {
        super('service-container');
        this.props = props;
        this.trafficPort = props.trafficPort;
        this.logGroup = props.logGroup;
    }
    prehook(service, scope) {
        this.parentService = service;
        this.scope = scope;
    }
    // This hook sets the overall task resource requirements to the
    // resource requirements of the application itself.
    modifyTaskDefinitionProps(props) {
        return {
            ...props,
            cpu: this.props.cpu.toString(),
            memoryMiB: this.props.memoryMiB.toString(),
        };
    }
    // This hook adds the application container to the task definition.
    useTaskDefinition(taskDefinition) {
        let containerProps = {
            image: this.props.image,
            cpu: Number(this.props.cpu),
            memoryLimitMiB: Number(this.props.memoryMiB),
            environment: this.props.environment,
            environmentFiles: this.props.environmentFiles,
            secrets: this.props.secrets,
        };
        // Let other extensions mutate the container definition. This is
        // used by extensions which want to add environment variables, modify
        // logging parameters, etc.
        this.containerMutatingHooks.forEach((hookProvider) => {
            containerProps = hookProvider.mutateContainerDefinition(containerProps);
        });
        // If no observability extensions have been added to the service description then we can configure the `awslogs` log driver
        if (!containerProps.logging) {
            // Create a log group for the service if one is not provided by the user (only if feature flag is set)
            if (!this.logGroup && this.parentService.node.tryGetContext(cxapi.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER)) {
                this.logGroup = new awslogs.LogGroup(this.scope, `${this.parentService.id}-logs`, {
                    logGroupName: `${this.parentService.id}-logs`,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    retention: awslogs.RetentionDays.ONE_MONTH,
                });
            }
            if (this.logGroup) {
                containerProps = {
                    ...containerProps,
                    logging: new ecs.AwsLogDriver({
                        streamPrefix: this.parentService.id,
                        logGroup: this.logGroup,
                    }),
                };
            }
        }
        else {
            if (this.logGroup) {
                throw Error(`Log configuration already specified. You cannot provide a log group for the application container of service '${this.parentService.id}' while also adding log configuration separately using service extensions.`);
            }
        }
        this.container = taskDefinition.addContainer('app', containerProps);
        // Create a port mapping for the container
        this.container.addPortMappings({
            containerPort: this.trafficPort,
        });
        // Raise the ulimits for this main application container
        // so that it can handle more concurrent requests
        this.container.addUlimits({
            softLimit: 1024000,
            hardLimit: 1024000,
            name: ecs.UlimitName.NOFILE,
        });
    }
    resolveContainerDependencies() {
        if (!this.container) {
            throw new Error('The container dependency hook was called before the container was created');
        }
        const firelens = this.parentService.serviceDescription.get('firelens');
        if (firelens && firelens.container) {
            this.container.addContainerDependencies({
                container: firelens.container,
                condition: ecs.ContainerDependencyCondition.START,
            });
        }
        const appmeshextension = this.parentService.serviceDescription.get('appmesh');
        if (appmeshextension && appmeshextension.container) {
            this.container.addContainerDependencies({
                container: appmeshextension.container,
                condition: ecs.ContainerDependencyCondition.HEALTHY,
            });
        }
        const cloudwatchextension = this.parentService.serviceDescription.get('cloudwatchAgent');
        if (cloudwatchextension && cloudwatchextension.container) {
            this.container.addContainerDependencies({
                container: cloudwatchextension.container,
                condition: ecs.ContainerDependencyCondition.START,
            });
        }
        const xrayextension = this.parentService.serviceDescription.get('xray');
        if (xrayextension && xrayextension.container) {
            this.container.addContainerDependencies({
                container: xrayextension.container,
                condition: ecs.ContainerDependencyCondition.HEALTHY,
            });
        }
    }
}
exports.Container = Container;
_a = JSII_RTTI_SYMBOL_1;
Container[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.Container", version: "2.0.0-alpha.160" };
//# sourceMappingURL=data:application/json;base64,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