# Installation Migration Guide

This guide helps you migrate from the Python/pip installation method to the new Homebrew tap installation for Riveter.

## Why Migrate to Homebrew?

The Homebrew installation offers several advantages over the Python/pip method:

- ✅ **No Python environment management** - Works without Python installed
- ✅ **Faster startup times** - Standalone binary vs Python interpreter
- ✅ **Automatic updates** - Use `brew upgrade riveter` to get latest version
- ✅ **Simpler CI/CD integration** - No virtual environment activation needed
- ✅ **Cross-platform consistency** - Same installation method on macOS and Linux
- ✅ **Reduced dependencies** - No need to manage Python versions or virtual environments

## Installation Method Comparison

| Feature | Homebrew | Python/pip |
|---------|----------|------------|
| **Installation Complexity** | Simple one-liner | Multi-step with venv setup |
| **Python Required** | No | Yes (3.12+) |
| **Virtual Environment** | Not needed | Required |
| **Startup Time** | Fast (binary) | Slower (Python interpreter) |
| **Updates** | `brew upgrade riveter` | Manual git pull + pip install |
| **CI/CD Setup** | Single brew command | Clone + venv + pip install |
| **Platform Support** | macOS, Linux | macOS, Linux, Windows |
| **Disk Space** | ~50MB (binary) | ~200MB (Python + deps) |
| **Offline Usage** | Yes (after install) | Yes (after install) |

## Step-by-Step Migration

### Step 1: Install Homebrew (if not already installed)

**macOS:**
```bash
/bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
```

**Linux:**
```bash
/bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
echo 'eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"' >> ~/.bashrc
eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"
```

### Step 2: Install Riveter via Homebrew

Choose one of these installation methods:

**Option A: One-step installation**
```bash
brew install scottryanhoward/homebrew-riveter/riveter
```

**Option B: Two-step installation**
```bash
brew tap scottryanhoward/homebrew-riveter
brew install riveter
```

### Step 3: Verify Installation

Test that the Homebrew version works correctly:

```bash
# Check version
riveter --version

# List rule packs
riveter list-rule-packs

# Test a quick scan (if you have a Terraform file)
riveter scan -p aws-security -t main.tf
```

### Step 4: Update Your Workflows

#### Shell Scripts and Aliases

**Before (Python/pip):**
```bash
#!/bin/bash
cd /path/to/riveter
source venv/bin/activate
riveter scan -p aws-security -t main.tf
```

**After (Homebrew):**
```bash
#!/bin/bash
riveter scan -p aws-security -t main.tf
```

#### CI/CD Pipelines

**Before (GitHub Actions):**
```yaml
- name: Install Riveter
  run: |
    git clone https://github.com/riveter/riveter.git
    cd riveter && pip install -e .

- name: Run Validation
  run: |
    cd riveter && source venv/bin/activate
    riveter scan -p aws-security -t ../main.tf
```

**After (GitHub Actions):**
```yaml
- name: Install Riveter
  run: brew install scottryanhoward/homebrew-riveter/riveter

- name: Run Validation
  run: riveter scan -p aws-security -t main.tf
```

**Before (GitLab CI):**
```yaml
script:
  - git clone https://github.com/riveter/riveter.git
  - cd riveter && pip install -e .
  - source venv/bin/activate
  - riveter scan -p aws-security -t ../main.tf
```

**After (GitLab CI):**
```yaml
before_script:
  - apt-get update && apt-get install -y curl git
  - /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"
  - eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"
script:
  - brew install scottryanhoward/homebrew-riveter/riveter
  - riveter scan -p aws-security -t main.tf
```

### Step 5: Clean Up Old Installation (Optional)

Once you've verified the Homebrew installation works, you can clean up the old Python installation:

```bash
# Remove the cloned repository (backup any custom rules first!)
rm -rf /path/to/riveter

# Remove any shell aliases or PATH modifications for the old installation
# Edit your ~/.bashrc, ~/.zshrc, or ~/.profile to remove Riveter-related entries
```

## Troubleshooting Migration Issues

### Issue: Command Not Found After Installation

**Problem:** `riveter: command not found` after Homebrew installation

**Solution:**
```bash
# Ensure Homebrew is in your PATH
echo 'eval "$(/opt/homebrew/bin/brew shellenv)"' >> ~/.zshrc  # macOS
echo 'eval "$(/home/linuxbrew/.linuxbrew/bin/brew shellenv)"' >> ~/.bashrc  # Linux

# Reload your shell
source ~/.zshrc  # or ~/.bashrc
```

### Issue: Different Behavior Between Versions

**Problem:** Results differ between Python and Homebrew versions

**Solution:**
1. Check versions are the same:
   ```bash
   riveter --version
   ```

2. Ensure you're using the same rule packs:
   ```bash
   riveter list-rule-packs
   ```

3. Compare with identical commands and files

### Issue: Missing Custom Rules

**Problem:** Custom rule files not found after migration

**Solution:**
- Custom rule files are not included in the binary
- Copy your custom `.yml` rule files to your new working directory
- Update paths in your commands to reference the correct location

### Issue: CI/CD Pipeline Failures

**Problem:** CI/CD pipelines fail after switching to Homebrew

**Solution:**
1. **GitHub Actions:** Ensure you're using a runner that supports Homebrew (ubuntu-latest, macos-latest)
2. **GitLab CI:** Use `ubuntu:latest` image and install Homebrew in `before_script`
3. **Jenkins:** Install Homebrew on your Jenkins agents or use Docker containers with Homebrew

## Performance Comparison

### Startup Time Comparison

| Installation Method | Cold Start | Warm Start |
|-------------------|------------|------------|
| Homebrew (binary) | ~0.1s | ~0.05s |
| Python/pip | ~0.8s | ~0.3s |

### Memory Usage Comparison

| Installation Method | Memory Usage |
|-------------------|--------------|
| Homebrew (binary) | ~15MB |
| Python/pip | ~45MB |

## Keeping Both Installations

If you need to keep both installations temporarily:

1. **Use full paths** to distinguish between versions:
   ```bash
   # Homebrew version
   /opt/homebrew/bin/riveter --version

   # Python version (with venv activated)
   /path/to/riveter/venv/bin/riveter --version
   ```

2. **Use aliases** for easy switching:
   ```bash
   alias riveter-brew='/opt/homebrew/bin/riveter'
   alias riveter-python='/path/to/riveter/venv/bin/riveter'
   ```

## Getting Help

If you encounter issues during migration:

1. **Check the troubleshooting section** in the main README
2. **Compare versions** to ensure you're using the same Riveter version
3. **Test with simple commands** first (like `riveter --version`)
4. **Open an issue** on GitHub if you find bugs or inconsistencies

## Rollback Plan

If you need to rollback to the Python installation:

1. **Uninstall Homebrew version:**
   ```bash
   brew uninstall riveter
   brew untap scottryanhoward/homebrew-riveter
   ```

2. **Reinstall Python version:**
   ```bash
   git clone https://github.com/riveter/riveter.git
   cd riveter
   python3 -m venv venv
   source venv/bin/activate
   pip install -e .
   ```

3. **Update your workflows** back to the Python method

## Future Considerations

- **Windows Support:** Currently, Homebrew installation is available for macOS and Linux. Windows users should continue using the Python/pip method until Windows binaries are available.

- **Custom Extensions:** If you've modified the Riveter source code, you'll need to stick with the Python installation or contribute your changes back to the main repository.

- **Enterprise Environments:** Some enterprise environments may have restrictions on Homebrew. In such cases, the Python/pip method may be more appropriate.

## Feedback

We'd love to hear about your migration experience! Please:

- Report any issues on our [GitHub Issues](https://github.com/riveter/riveter/issues)
- Share feedback in [GitHub Discussions](https://github.com/riveter/riveter/discussions)
- Contribute improvements to this migration guide
