# Token Management Guide

This guide explains how to securely manage GitHub tokens used in Riveter's automated workflows, including setup, rotation, and monitoring procedures.

## Overview

Riveter uses GitHub tokens for automated operations such as:

- Updating Homebrew formulae in the tap repository
- Uploading release assets
- Creating and managing releases
- Cross-repository automation

## Token Types and Permissions

### HOMEBREW_TAP_TOKEN

**Purpose**: Allows the main Riveter repository to update the Homebrew tap repository.

**Required Permissions**:
- `contents: write` - Update formula files
- `metadata: read` - Read repository information
- `pull_requests: write` - Create PRs for formula updates (if needed)

**Repository Access**: `ScottRyanHoward/homebrew-riveter`

### GITHUB_TOKEN

**Purpose**: Standard GitHub Actions token for repository operations.

**Required Permissions**:
- `contents: write` - Create releases and tags
- `actions: read` - Read workflow information
- `metadata: read` - Read repository metadata

**Repository Access**: `riveter/riveter` (automatically scoped)

## Token Setup

### Creating a Fine-Grained Personal Access Token

1. **Navigate to GitHub Settings**:
   - Go to GitHub.com → Settings → Developer settings → Personal access tokens → Fine-grained tokens

2. **Create New Token**:
   - Click "Generate new token"
   - Set expiration (recommended: 90 days maximum)
   - Add description: "Riveter Homebrew Tap Automation"

3. **Configure Repository Access**:
   - Select "Selected repositories"
   - Add `ScottRyanHoward/homebrew-riveter`

4. **Set Permissions**:
   ```
   Repository permissions:
   - Contents: Write
   - Metadata: Read
   - Pull requests: Write
   ```

5. **Generate and Copy Token**:
   - Click "Generate token"
   - Copy the token immediately (it won't be shown again)

### Adding Token to Repository Secrets

1. **Navigate to Repository Settings**:
   - Go to the main Riveter repository
   - Settings → Secrets and variables → Actions

2. **Add Repository Secret**:
   - Click "New repository secret"
   - Name: `HOMEBREW_TAP_TOKEN`
   - Value: Paste the token from above
   - Click "Add secret"

## Token Validation

### Using the Token Management Script

```bash
# Validate token format and permissions
python scripts/manage_tokens.py validate --token-name HOMEBREW_TAP_TOKEN

# Check specific repository permissions
python scripts/manage_tokens.py check-permissions --token-name HOMEBREW_TAP_TOKEN --repository ScottRyanHoward/homebrew-riveter

# Test token in debug mode
python scripts/manage_tokens.py validate --token-name HOMEBREW_TAP_TOKEN --debug
```

### Manual Validation

```bash
# Test token access to homebrew tap repository
curl -H "Authorization: token $HOMEBREW_TAP_TOKEN" \
     -H "Accept: application/vnd.github.v3+json" \
     https://api.github.com/repos/ScottRyanHoward/homebrew-riveter

# Check token permissions
curl -H "Authorization: token $HOMEBREW_TAP_TOKEN" \
     -H "Accept: application/vnd.github.v3+json" \
     https://api.github.com/repos/ScottRyanHoward/homebrew-riveter | jq '.permissions'
```

## Token Rotation

### Rotation Schedule

- **Fine-grained tokens**: Every 90 days (maximum GitHub allows)
- **Classic tokens**: Every 30-60 days
- **Emergency rotation**: Immediately if compromise is suspected

### Rotation Procedure

1. **Generate Rotation Reminder**:
   ```bash
   python scripts/manage_tokens.py rotation-reminder --token-name HOMEBREW_TAP_TOKEN
   ```

2. **Create New Token**:
   - Follow the setup procedure above
   - Use the same permissions as the current token
   - Set a clear description with creation date

3. **Test New Token**:
   ```bash
   # Set new token temporarily
   export HOMEBREW_TAP_TOKEN="new_token_here"

   # Validate new token
   python scripts/manage_tokens.py validate --token-name HOMEBREW_TAP_TOKEN
   ```

4. **Update Repository Secret**:
   - Go to repository Settings → Secrets and variables → Actions
   - Click on `HOMEBREW_TAP_TOKEN`
   - Click "Update" and paste the new token
   - Click "Update secret"

5. **Test in Production**:
   - Trigger a manual workflow run
   - Monitor for successful completion
   - Check logs for any authentication errors

6. **Revoke Old Token**:
   - Only after confirming the new token works
   - Go to GitHub Settings → Developer settings → Personal access tokens
   - Find the old token and click "Delete"

### Automated Rotation Reminders

Set up calendar reminders or use the token management script:

```bash
# Add to crontab for monthly reminders
0 9 1 * * /usr/bin/python3 /path/to/riveter/scripts/manage_tokens.py rotation-reminder --token-name HOMEBREW_TAP_TOKEN | mail -s "Token Rotation Reminder" admin@example.com
```

## Access Logging and Monitoring

### Logging Token Usage

The token management script automatically logs token access:

```bash
# Log successful formula update
python scripts/manage_tokens.py log-access \
  --action "formula_update" \
  --repository "homebrew-riveter" \
  --token-name HOMEBREW_TAP_TOKEN \
  --success \
  --details "Updated formula to version 1.2.3"

# Log failed access attempt
python scripts/manage_tokens.py log-access \
  --action "release_upload" \
  --repository "riveter" \
  --token-name GITHUB_TOKEN \
  --details "Permission denied error"
```

### Viewing Access Logs

```bash
# View last 30 days of token access
python scripts/manage_tokens.py view-logs

# View last 7 days
python scripts/manage_tokens.py view-logs --days 7
```

### Log File Location

Access logs are stored in `logs/token_access.log` in JSON format:

```json
{
  "timestamp": "2024-01-15T10:30:00+00:00",
  "action": "formula_update",
  "repository": "homebrew-riveter",
  "token_name": "HOMEBREW_TAP_TOKEN",
  "success": true,
  "details": "Updated formula to version 1.2.3"
}
```

## Security Best Practices

### Token Security

1. **Use Fine-Grained Tokens**: More secure than classic tokens
2. **Minimal Permissions**: Only grant necessary permissions
3. **Short Expiration**: Use the shortest practical expiration time
4. **Regular Rotation**: Rotate tokens on a schedule
5. **Secure Storage**: Never commit tokens to code or logs

### Access Control

1. **Repository Secrets**: Store tokens in GitHub repository secrets
2. **Environment Isolation**: Use different tokens for different environments
3. **Access Monitoring**: Log and monitor all token usage
4. **Principle of Least Privilege**: Grant minimum required access

### Incident Response

1. **Immediate Revocation**: Revoke compromised tokens immediately
2. **Access Review**: Review all recent token access logs
3. **New Token Generation**: Create new tokens with fresh permissions
4. **Security Audit**: Conduct security review of affected systems

## Troubleshooting

### Common Issues

#### "Token authentication failed"

1. **Check token format**: Ensure token starts with correct prefix
2. **Verify token validity**: Token may have expired
3. **Check permissions**: Token may lack required permissions
4. **Repository access**: Token may not have access to target repository

#### "Insufficient permissions"

1. **Review required permissions**: Check documentation for required permissions
2. **Update token permissions**: Regenerate token with correct permissions
3. **Repository settings**: Verify repository allows the required actions

#### "Token not found in environment"

1. **Check secret name**: Ensure secret name matches exactly
2. **Repository context**: Verify secret is set in correct repository
3. **Workflow permissions**: Check workflow has access to secrets

### Debugging Commands

```bash
# Debug token validation
python scripts/manage_tokens.py validate --token-name HOMEBREW_TAP_TOKEN --debug

# Check specific permissions
python scripts/manage_tokens.py check-permissions \
  --token-name HOMEBREW_TAP_TOKEN \
  --repository ScottRyanHoward/homebrew-riveter \
  --debug

# Test API access manually
curl -v -H "Authorization: token $HOMEBREW_TAP_TOKEN" \
     https://api.github.com/repos/ScottRyanHoward/homebrew-riveter
```

## Workflow Integration

### GitHub Actions Integration

```yaml
# Example workflow step with token validation
- name: Validate tokens
  run: |
    python scripts/manage_tokens.py validate --token-name HOMEBREW_TAP_TOKEN
    python scripts/manage_tokens.py log-access \
      --action "workflow_start" \
      --repository "${{ github.repository }}" \
      --token-name HOMEBREW_TAP_TOKEN \
      --success
  env:
    HOMEBREW_TAP_TOKEN: ${{ secrets.HOMEBREW_TAP_TOKEN }}

# Example with error handling
- name: Update homebrew formula
  run: |
    # Your formula update logic here
    if formula_update_success; then
      python scripts/manage_tokens.py log-access \
        --action "formula_update" \
        --repository "homebrew-riveter" \
        --token-name HOMEBREW_TAP_TOKEN \
        --success \
        --details "Updated to version ${{ github.ref_name }}"
    else
      python scripts/manage_tokens.py log-access \
        --action "formula_update" \
        --repository "homebrew-riveter" \
        --token-name HOMEBREW_TAP_TOKEN \
        --details "Update failed: $error_message"
      exit 1
    fi
  env:
    HOMEBREW_TAP_TOKEN: ${{ secrets.HOMEBREW_TAP_TOKEN }}
```

## Compliance and Auditing

### Audit Requirements

1. **Access Logging**: All token usage must be logged
2. **Regular Review**: Monthly review of token access logs
3. **Rotation Tracking**: Document all token rotations
4. **Permission Changes**: Log any permission modifications

### Compliance Checklist

- [ ] Tokens use minimal required permissions
- [ ] Tokens have appropriate expiration dates
- [ ] All token access is logged
- [ ] Regular rotation schedule is followed
- [ ] Access logs are reviewed monthly
- [ ] Incident response procedures are documented
- [ ] Token storage follows security best practices

## Related Documentation

- [Security Setup Guide](SECURITY_SETUP.md)
- [Release Workflow](RELEASE_WORKFLOW.md)
- [Homebrew Tap Setup](HOMEBREW_TAP_SETUP.md)
- [Workflow Dependencies](WORKFLOW_DEPENDENCIES.md)
