# Dictionary Workflow 🔄

Complete dictionary processing workflow including cleaning, validation, and supplementation. Streamlines the entire dictionary preparation process from raw data to validated, enhanced dictionaries ready for use in data validation workflows.

---

📅 **Build Date:** [INSERT_DATE_HERE]

This package was last updated on the date above.  
For reproducibility and support, always refer to this date when sharing logs or output.

---

## 📦 Project Structure

```
dictionary_workflow/
├── __init__.py         # Package interface and version info
├── main.py            # CLI entry point
├── utils.py           # Helper functions
├── env.py             # Environment detection
└── README.md         # This documentation
```

---

## 🚀 Usage (Development)

### Command Line
```bash
python -m scriptcraft.tools.dictionary_workflow --input-paths supplements.csv --dictionary-paths dict.csv --output-dir output
```

### Python API
```python
from scriptcraft.tools.dictionary_workflow import DictionaryWorkflow

workflow = DictionaryWorkflow()
workflow.run(
    input_paths=["supplements.csv"],
    dictionary_paths=["dict.csv"],
    output_dir="output"
)
```

Arguments:
- `--input-paths`: List of supplement file paths
- `--dictionary-paths`: List of dictionary file paths
- `--output-dir`: Output directory for processed files
- `--workflow-steps`: Steps to run (prepare, split, enhance)
- `--merge-strategy`: Merge strategy (outer, inner, left, right)
- `--enhancement-strategy`: Enhancement strategy (append, merge, replace)
- `--domain-column`: Domain column name
- `--clean-data`: Enable data cleaning

---

## ⚙️ Features

- 🔄 Complete dictionary processing workflow
- 🧹 Dictionary cleaning and standardization
- ✅ Dictionary validation and quality assessment
- 📝 Dictionary supplementation and enhancement
- 📊 Comprehensive reporting and metrics
- 🛡️ Error handling and recovery
- 📈 Performance optimization
- 🎯 Domain-specific processing

---

## 🔧 Dev Tips

- Use domain-specific settings for healthcare data dictionaries
- Test workflow steps individually before running the complete workflow
- Check supplement and dictionary file formats for compatibility
- Review workflow output for each step before proceeding
- Use batch processing for multiple supplement/dictionary files
- Customize merge and enhancement strategies based on data requirements

---

## 🧪 Testing

### Unit Tests
```bash
python -m pytest tests/tools/test_dictionary_workflow.py
```

### Integration Tests
```bash
python -m pytest tests/integration/tools/test_dictionary_workflow_integration.py
```

### Test Data
Example files needed:
- Sample supplement files with various formats
- Sample dictionary files
- Expected workflow output files
- Test cases for different workflow steps
- Domain-specific test data

---

## 🔄 Dependencies

Required packages:
- pandas >= 1.3.0
- openpyxl >= 3.0.0
- Python >= 3.8

System requirements:
- Memory: 200MB base + 100MB per file
- Storage: 500MB for processing and output
- CPU: Multi-core recommended for batch processing

---

## 🚨 Error Handling

Common errors and solutions:
1. **Supplement Format Error**
   - Cause: Supplement file format not recognized
   - Solution: Check file format and required columns
2. **Dictionary Format Error**
   - Cause: Dictionary file format not recognized
   - Solution: Verify dictionary format and required structure
3. **Workflow Step Error**
   - Cause: Workflow step failed or incompatible
   - Solution: Check step configuration and input data compatibility

---

## 📊 Performance

Expectations:
- Processing speed: 1000-3000 records per second
- Memory usage: 200MB base + 100MB per file
- File size limits: Up to 200MB per input file

Optimization tips:
- Use specific workflow steps instead of complete workflow
- Process large files in chunks
- Enable parallel processing for multiple files
- Optimize merge and enhancement strategies

---

## 📋 Development Checklist

### 1. File Structure ✅
- [x] Standard package layout
  - [x] __init__.py with version info
  - [x] main.py for CLI
  - [x] utils.py for helpers
  - [x] env.py for environment detection
  - [x] README.md
- [x] Clean organization
- [x] No deprecated files

### 2. Documentation ✅
- [x] Version information
- [x] Package-level docstring
- [x] Function docstrings
- [x] Type hints
- [x] README.md
- [x] API documentation
- [x] Error code reference
- [x] Troubleshooting guide

### 3. Code Implementation ✅
- [x] Core functionality
- [x] CLI interface
- [x] Error handling
- [x] Input validation
- [x] Type checking
- [x] Performance optimization
- [x] Security considerations

### 4. Testing ⬜
- [ ] Unit tests
- [ ] Integration tests
- [ ] Performance tests
- [ ] Edge case tests
- [ ] Error condition tests
- [ ] Test data examples

### 5. Error Handling ✅
- [x] Custom exceptions
- [x] Error messages
- [x] Error logging
- [x] Error recovery
- [x] Input validation

### 6. Performance ✅
- [x] Large dataset testing
- [x] Memory optimization
- [x] Progress reporting
- [x] Chunked processing
- [x] Performance metrics

### 7. Configuration ✅
- [x] Command-line arguments
- [x] Configuration validation
- [x] Environment variables
- [x] Default settings
- [x] Documentation

### 8. Packaging ✅
- [x] Dependencies specified
- [x] Version information
- [x] Package structure
- [x] Installation tested
- [x] Distribution tested

---

## 📋 Current Status and Future Improvements

### ✅ Completed Items
1. **Core Implementation**
   - Complete workflow processing
   - Multiple workflow steps
   - Merge and enhancement strategies
   - Domain-specific processing
   - Batch processing support

2. **Documentation**
   - Main README structure
   - Usage examples
   - Error handling guide
   - Performance metrics

3. **Infrastructure**
   - Environment detection
   - CLI integration
   - Error handling
   - Configuration management

### 🔄 Partially Complete
1. **Testing**
   - ✅ Basic structure
   - ❌ Need comprehensive test suite
   - ❌ Need integration tests
   - ❌ Need performance tests

2. **Features**
   - ✅ Basic workflow functionality
   - ❌ Need advanced workflow steps
   - ❌ Need workflow optimization
   - ❌ Need enhanced reporting

### 🎯 Prioritized Improvements

#### High Priority
1. **Testing Enhancement**
   - Add comprehensive test suite
   - Create integration tests
   - Add performance benchmarks
   - Improve error case coverage

2. **Feature Enhancement**
   - Add advanced workflow steps
   - Implement workflow optimization
   - Add enhanced reporting
   - Improve user experience

#### Medium Priority
3. **Documentation**
   - Add detailed API docs
   - Create troubleshooting guide
   - Add performance tuning guide
   - Document common patterns

4. **User Experience**
   - Add progress tracking
   - Improve error messages
   - Add configuration validation
   - Create interactive mode

#### Low Priority
5. **Advanced Features**
   - Add ML-based workflow optimization
   - Support more formats
   - Add workflow learning
   - Create workflow summaries

6. **Development Tools**
   - Add development utilities
   - Create debugging helpers
   - Add profiling support
   - Improve error messages

---

## 🤝 Contributing

1. Branch naming: `feature/dictionary-workflow-[feature]`
2. Required for all changes:
   - Unit tests
   - Documentation updates
   - Checklist review
3. Code review process in CONTRIBUTING.md 