"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolve = resolve;
const util_1 = require("util");
function isResolvable(obj) {
    return obj.toJSON !== undefined;
}
function resolve(value, options = {}) {
    const args = options.args ?? [];
    const omitEmpty = options.omitEmpty ?? false;
    if (value == null) {
        return value;
    }
    if (isResolvable(value)) {
        const resolved = value.toJSON();
        return resolve(resolved, options);
    }
    // Special resolution for few JavaScript built-in types
    // that by default would be stringified as empty objects ('{}')
    // as they are missing a `toJSON` implementation.
    switch (true) {
        case util_1.types.isRegExp(value):
            if (value.flags) {
                throw new Error("RegExp with flags should be explicitly converted to a string");
            }
            return value.source;
        case util_1.types.isSet(value):
            return resolve(Array.from(value), options);
        case util_1.types.isMap(value):
            return resolve(Object.fromEntries(value), options);
        case util_1.types.isBoxedPrimitive(value):
            return resolve(value.valueOf(), options);
    }
    // if value is a function, call it and resolve the result.
    if (typeof value === "function") {
        const resolved = value.apply(undefined, args);
        return resolve(resolved, options);
    }
    if (typeof value !== "object") {
        return value;
    }
    if (Array.isArray(value)) {
        if (omitEmpty && value.length === 0) {
            return undefined;
        }
        return value.map((x) => resolve(x, options)).filter((x) => x != null); // filter undefined/null/omitted
    }
    // only allow data types (i.e. objects without constructors)
    if (value.constructor && value.constructor?.name !== "Object") {
        throw new Error(`only data types can be resolved. trying to resolve object of type ${value.constructor?.name}`);
    }
    const result = {};
    for (const [k, v] of Object.entries(value)) {
        const resolved = resolve(v, options);
        // skip undefined values
        if (resolved === undefined) {
            continue;
        }
        result[k] = resolved;
    }
    if (omitEmpty && Object.keys(result).length === 0) {
        return undefined;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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