import { Task } from "..";
import { Component } from "../component";
import { GitIdentity } from "../github";
import { Job, JobPermissions, JobStep, Tools, Triggers } from "../github/workflows-model";
import { Project } from "../project";
import { GroupRunnerOptions } from "../runner-options";
export interface BuildWorkflowCommonOptions {
    /**
     * Name of the buildfile (e.g. "build" becomes "build.yml").
     *
     * @default "build"
     */
    readonly name?: string;
    /**
     * Steps to execute before the build.
     * @default []
     */
    readonly preBuildSteps?: JobStep[];
    /**
     * Build workflow triggers
     * @default "{ pullRequest: {}, workflowDispatch: {} }"
     */
    readonly workflowTriggers?: Triggers;
    /**
     * Permissions granted to the build job
     * To limit job permissions for `contents`, the desired permissions have to be explicitly set, e.g.: `{ contents: JobPermission.NONE }`
     * @default `{ contents: JobPermission.WRITE }`
     */
    readonly permissions?: JobPermissions;
    /**
     * Build environment variables.
     * @default {}
     */
    readonly env?: {
        [key: string]: string;
    };
}
export interface BuildWorkflowOptions extends BuildWorkflowCommonOptions {
    /**
     * The task to execute in order to build the project.
     */
    readonly buildTask: Task;
    /**
     * A name of a directory that includes build artifacts.
     * @default "dist"
     */
    readonly artifactsDirectory?: string;
    /**
     * The container image to use for builds.
     * @default - the default workflow container
     */
    readonly containerImage?: string;
    /**
     * Automatically update files modified during builds to pull-request branches.
     * This means that any files synthesized by projen or e.g. test snapshots will
     * always be up-to-date before a PR is merged.
     *
     * Implies that PR builds do not have anti-tamper checks.
     *
     * This is enabled by default only if `githubTokenSecret` is set. Otherwise it
     * is disabled, which implies that file changes that happen during build will
     * not be pushed back to the branch.
     *
     * @default true
     */
    readonly mutableBuild?: boolean;
    /**
     * Steps to execute after build.
     * @default []
     */
    readonly postBuildSteps?: JobStep[];
    /**
     * Git identity to use for the workflow.
     * @default - default GitHub Actions user
     */
    readonly gitIdentity?: GitIdentity;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     * @description Defines a target Runner by labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOn?: string[];
    /**
     * Github Runner Group selection options
     * @description Defines a target Runner Group by name and/or labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOnGroup?: GroupRunnerOptions;
}
export declare class BuildWorkflow extends Component {
    /**
     * Name of generated github workflow
     */
    readonly name: string;
    private readonly postBuildSteps;
    private readonly preBuildSteps;
    private readonly gitIdentity;
    private readonly buildTask;
    private readonly github;
    private readonly workflow;
    private readonly artifactsDirectory;
    private readonly _postBuildJobs;
    constructor(project: Project, options: BuildWorkflowOptions);
    private addBuildJob;
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds(): string[];
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps: JobStep[]): void;
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id: string, job: Job): void;
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task: Task, options?: AddPostBuildJobTaskOptions): void;
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id: string, commands: string[], options?: AddPostBuildJobCommandsOptions): void;
    private addSelfMutationJob;
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    private renderBuildSteps;
}
/**
 * Options for `BuildWorkflow.addPostBuildJobTask`
 */
export interface AddPostBuildJobTaskOptions {
    /**
     * Tools that should be installed before the task is run.
     */
    readonly tools?: Tools;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     * @description Defines a target Runner by labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOn?: string[];
    /**
     * Github Runner Group selection options
     * @description Defines a target Runner Group by name and/or labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOnGroup?: GroupRunnerOptions;
}
/**
 * Options for `BuildWorkflow.addPostBuildJobCommands`
 */
export interface AddPostBuildJobCommandsOptions {
    /**
     * Tools that should be installed before the commands are run.
     */
    readonly tools?: Tools;
    /**
     * Check out the repository at the pull request branch before commands are
     * run.
     *
     * @default false
     */
    readonly checkoutRepo?: boolean;
    /**
     * Install project dependencies before running commands. `checkoutRepo` must
     * also be set to true.
     *
     * Currently only supported for `NodeProject`.
     *
     * @default false
     */
    readonly installDeps?: boolean;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     * @description Defines a target Runner by labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOn?: string[];
    /**
     * Github Runner Group selection options
     * @description Defines a target Runner Group by name and/or labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOnGroup?: GroupRunnerOptions;
}
