"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitAttributesFile = exports.EndOfLine = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const file_1 = require("./file");
/**
 * The end of line characters supported by git.
 */
var EndOfLine;
(function (EndOfLine) {
    /**
     * Maintain existing (mixed values within one file are normalised by looking
     * at what's used after the first line)
     */
    EndOfLine["AUTO"] = "auto";
    /**
     * Carriage Return + Line Feed characters (\r\n), common on Windows
     */
    EndOfLine["CRLF"] = "crlf";
    /**
     * Line Feed only (\n), common on Linux and macOS as well as inside git repos
     */
    EndOfLine["LF"] = "lf";
    /**
     * Disable and do not configure the end of line character
     */
    EndOfLine["NONE"] = "none";
})(EndOfLine || (exports.EndOfLine = EndOfLine = {}));
/**
 * Assign attributes to file names in a git repository.
 *
 * @see https://git-scm.com/docs/gitattributes
 */
class GitAttributesFile extends file_1.FileBase {
    constructor(scope, options) {
        super(scope, ".gitattributes", {
            editGitignore: false,
        });
        this.attributes = new Map();
        this.endOfLine = options?.endOfLine ?? EndOfLine.LF;
        if (this.endOfLine != EndOfLine.NONE) {
            let endOfLineAttributes = [`text=auto`];
            if (this.endOfLine != EndOfLine.AUTO) {
                endOfLineAttributes.push(`eol=${this.endOfLine}`);
            }
            // Setting a default end of line for all text files in the repository
            // This line should be the first one in order to use it as a default for text files and allow for overriding in later lines
            this.addAttributes("*", ...endOfLineAttributes);
        }
    }
    /**
     * Maps a set of attributes to a set of files.
     * @param glob Glob pattern to match files in the repo.
     * @param attributes Attributes to assign to these files.
     */
    addAttributes(glob, ...attributes) {
        if (!this.attributes.has(glob)) {
            this.attributes.set(glob, new Set());
        }
        const set = this.attributes.get(glob);
        for (const attribute of attributes) {
            set.add(attribute);
        }
    }
    /**
     * Removes attributes from a set of files.
     *
     * If no attributes are provided, the glob pattern will be removed completely.
     *
     * @param glob Glob pattern to modify.
     * @param attributes Attributes to remove from matched files.
     */
    removeAttributes(glob, ...attributes) {
        if (attributes.length === 0) {
            this.attributes.delete(glob);
            return;
        }
        const mapping = this.attributes.get(glob);
        if (!mapping) {
            return;
        }
        for (const attribute of attributes) {
            mapping.delete(attribute);
        }
        if (mapping.size === 0) {
            this.attributes.delete(glob);
        }
    }
    /**
     * Add attributes necessary to mark these files as stored in LFS
     */
    addLfsPattern(glob) {
        this.addAttributes(glob, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
    }
    /**
     * Whether the current gitattributes file has any LFS patterns
     */
    get hasLfsPatterns() {
        return Array.from(this.attributes.values()).some((attrs) => attrs.has("filter=lfs"));
    }
    preSynthesize() {
        this.project.addPackageIgnore("/.gitattributes");
        super.preSynthesize();
    }
    synthesizeContent(_) {
        // We can assume the file map is never empty.
        const entries = Array.from(this.attributes.entries()).sort(([l], [r]) => l.localeCompare(r));
        if (entries.length === 0) {
            return undefined;
        }
        return [
            ...(this.marker ? [`# ${this.marker}`] : []),
            "",
            ...entries.map(([name, attributes]) => `${name} ${Array.from(attributes).join(" ")}`),
        ].join("\n");
    }
}
exports.GitAttributesFile = GitAttributesFile;
_a = JSII_RTTI_SYMBOL_1;
GitAttributesFile[_a] = { fqn: "projen.GitAttributesFile", version: "0.98.20" };
//# sourceMappingURL=data:application/json;base64,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