"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dependabot = exports.VersioningStrategy = exports.DependabotScheduleInterval = exports.DependabotGroupUpdateType = exports.DependabotGroupDependencyType = exports.DependabotGroupAppliesTo = exports.DependabotRegistryType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Each configuration type requires you to provide particular settings.
 * Some types allow more than one way to connect
 * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#configuration-options-for-private-registries
 */
var DependabotRegistryType;
(function (DependabotRegistryType) {
    /**
     * The composer-repository type supports username and password.
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#composer-repository
     */
    DependabotRegistryType["COMPOSER_REGISTRY"] = "composer-registry";
    /**
     * The docker-registry type supports username and password.
     * The docker-registry type can also be used to pull from Amazon ECR using static AWS credentials
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#docker-registry
     */
    DependabotRegistryType["DOCKER_REGISTRY"] = "docker-registry";
    /**
     * The git type supports username and password
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#git
     */
    DependabotRegistryType["GIT"] = "git";
    /**
     * The hex-organization type supports organization and key
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#hex-organization
     */
    DependabotRegistryType["HEX_ORGANIZATION"] = "hex-organization";
    /**
     * The maven-repository type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#maven-repository
     */
    DependabotRegistryType["MAVEN_REPOSITORY"] = "maven-repository";
    /**
     * The npm-registry type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#npm-registry
     */
    DependabotRegistryType["NPM_REGISTRY"] = "npm-registry";
    /**
     * The nuget-feed type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#nuget-feed
     */
    DependabotRegistryType["NUGET_FEED"] = "nuget-feed";
    /**
     * The python-index type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#python-index
     */
    DependabotRegistryType["PYTHON_INDEX"] = "python-index";
    /**
     * The rubygems-server type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#rubygems-server
     */
    DependabotRegistryType["RUBYGEMS_SERVER"] = "rubygems-server";
    /**
     * The terraform-registry type supports a token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#terraform-registry
     */
    DependabotRegistryType["TERRAFORM_REGISTRY"] = "terraform-registry";
})(DependabotRegistryType || (exports.DependabotRegistryType = DependabotRegistryType = {}));
/**
 * The type of update a group applies to.
 */
var DependabotGroupAppliesTo;
(function (DependabotGroupAppliesTo) {
    /**
     * Apply only to version updates.
     */
    DependabotGroupAppliesTo["VERSION_UPDATES"] = "version-updates";
    /**
     * Apply only to security updates.
     */
    DependabotGroupAppliesTo["SECURITY_UPDATES"] = "security-updates";
})(DependabotGroupAppliesTo || (exports.DependabotGroupAppliesTo = DependabotGroupAppliesTo = {}));
/**
 * The type of dependency a group may be limited to.
 */
var DependabotGroupDependencyType;
(function (DependabotGroupDependencyType) {
    /**
     * Include only dependencies in the "Development dependency group".
     */
    DependabotGroupDependencyType["DEVELOPMENT"] = "development";
    /**
     * Include only dependencies in the "Production dependency group".
     */
    DependabotGroupDependencyType["PRODUCTION"] = "production";
})(DependabotGroupDependencyType || (exports.DependabotGroupDependencyType = DependabotGroupDependencyType = {}));
/**
 * The semantic versioning levels a group may be limited to.
 */
var DependabotGroupUpdateType;
(function (DependabotGroupUpdateType) {
    /**
     * Include major releases.
     */
    DependabotGroupUpdateType["MAJOR"] = "major";
    /**
     * Include minor releases.
     */
    DependabotGroupUpdateType["MINOR"] = "minor";
    /**
     * Include patch releases.
     */
    DependabotGroupUpdateType["PATCH"] = "patch";
})(DependabotGroupUpdateType || (exports.DependabotGroupUpdateType = DependabotGroupUpdateType = {}));
/**
 * How often to check for new versions and raise pull requests for version
 * updates.
 */
var DependabotScheduleInterval;
(function (DependabotScheduleInterval) {
    /**
     * Runs on every weekday, Monday to Friday.
     */
    DependabotScheduleInterval["DAILY"] = "daily";
    /**
     * Runs once each week. By default, this is on Monday.
     */
    DependabotScheduleInterval["WEEKLY"] = "weekly";
    /**
     * Runs once each month. This is on the first day of the month.
     */
    DependabotScheduleInterval["MONTHLY"] = "monthly";
})(DependabotScheduleInterval || (exports.DependabotScheduleInterval = DependabotScheduleInterval = {}));
/**
 * The strategy to use when edits manifest and lock files.
 */
var VersioningStrategy;
(function (VersioningStrategy) {
    /**
     * Only create pull requests to update lockfiles updates. Ignore any new
     * versions that would require package manifest changes.
     */
    VersioningStrategy["LOCKFILE_ONLY"] = "lockfile-only";
    /**
     * - For apps, the version requirements are increased.
     * - For libraries, the range of versions is widened.
     */
    VersioningStrategy["AUTO"] = "auto";
    /**
     * Relax the version requirement to include both the new and old version, when
     * possible.
     */
    VersioningStrategy["WIDEN"] = "widen";
    /**
     * Always increase the version requirement to match the new version.
     */
    VersioningStrategy["INCREASE"] = "increase";
    /**
     * Increase the version requirement only when required by the new version.
     */
    VersioningStrategy["INCREASE_IF_NECESSARY"] = "increase-if-necessary";
})(VersioningStrategy || (exports.VersioningStrategy = VersioningStrategy = {}));
/**
 * Defines dependabot configuration for node projects.
 *
 * Since module versions are managed in projen, the versioning strategy will be
 * configured to "lockfile-only" which means that only updates that can be done
 * on the lockfile itself will be proposed.
 */
class Dependabot extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        const project = github.project;
        this.allow = [];
        this.ignore = [];
        this.ignoresProjen = options.ignoreProjen ?? true;
        const registries = options.registries
            ? (0, util_1.kebabCaseKeys)(options.registries)
            : undefined;
        for (const [name, group] of Object.entries(options.groups ?? {})) {
            validateGroup(name, group);
        }
        const groups = options.groups ? (0, util_1.kebabCaseKeys)(options.groups) : undefined;
        this.config = {
            version: 2,
            registries,
            updates: [
                {
                    "package-ecosystem": "npm",
                    "versioning-strategy": "lockfile-only",
                    directory: "/",
                    schedule: {
                        interval: options.scheduleInterval ?? DependabotScheduleInterval.DAILY,
                    },
                    allow: () => (this.allow.length > 0 ? this.allow : undefined),
                    ignore: () => (this.ignore.length > 0 ? this.ignore : undefined),
                    labels: options.labels ? options.labels : undefined,
                    registries: registries ? Object.keys(registries) : undefined,
                    groups: groups ? groups : undefined,
                    assignees: options.assignees && options.assignees.length > 0
                        ? options.assignees
                        : undefined,
                    reviewers: options.reviewers && options.reviewers.length > 0
                        ? options.reviewers
                        : undefined,
                    "open-pull-requests-limit": options.openPullRequestsLimit !== undefined
                        ? options.openPullRequestsLimit
                        : undefined,
                    "target-branch": options.targetBranch,
                },
            ],
        };
        new yaml_1.YamlFile(project, ".github/dependabot.yml", {
            obj: this.config,
            committed: true,
        });
        for (const i of options.allow ?? []) {
            this.addAllow(i.dependencyName);
        }
        for (const i of options.ignore ?? []) {
            this.addIgnore(i.dependencyName, ...(i.versions ?? []));
        }
        if (this.ignoresProjen) {
            this.addIgnore("projen");
        }
    }
    /**
     * Allows a dependency from automatic updates.
     *
     * @param dependencyName Use to allow updates for dependencies with matching
     * names, optionally using `*` to match zero or more characters.
     */
    addAllow(dependencyName) {
        this.allow.push({
            "dependency-name": dependencyName,
        });
    }
    /**
     * Ignores a dependency from automatic updates.
     *
     * @param dependencyName Use to ignore updates for dependencies with matching
     * names, optionally using `*` to match zero or more characters.
     * @param versions Use to ignore specific versions or ranges of versions. If
     * you want to define a range, use the standard pattern for the package
     * manager (for example: `^1.0.0` for npm, or `~> 2.0` for Bundler).
     */
    addIgnore(dependencyName, ...versions) {
        this.ignore.push({
            "dependency-name": dependencyName,
            versions: () => (versions.length > 0 ? versions : undefined),
        });
    }
}
exports.Dependabot = Dependabot;
_a = JSII_RTTI_SYMBOL_1;
Dependabot[_a] = { fqn: "projen.github.Dependabot", version: "0.98.20" };
function validateGroup(name, group) {
    // update types
    if (group.updateTypes !== undefined) {
        if (group.updateTypes.length < 1) {
            throw new Error(`Dependabot group "${name}" must not have an empty array of update types.`);
        }
        if (new Set(group.updateTypes).size !== group.updateTypes.length) {
            throw new Error(`Dependabot group "${name}" must not have duplicate update types.`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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