import { JobStep, AppPermissions } from "./workflows-model";
/**
 * Options for `GithubCredentials.fromPersonalAccessToken`
 */
export interface GithubCredentialsPersonalAccessTokenOptions {
    readonly secret?: string;
}
/**
 * Options for `GithubCredentials.fromApp`
 */
export interface GithubCredentialsAppOptions {
    /**
     * The secret containing the GitHub App ID
     *
     * @default "PROJEN_APP_ID"
     */
    readonly appIdSecret?: string;
    /**
     * The secret containing the GitHub App private key
     *
     * Escaped newlines (\\n) will be automatically replaced with actual newlines.
     *
     * @default "PROJEN_APP_PRIVATE_KEY"
     */
    readonly privateKeySecret?: string;
    /**
     * The owner of the GitHub App installation
     *
     * @default - if empty, defaults to the current repository owner
     */
    readonly owner?: string;
    /**
     * List of repositories to grant access to
     *
     * @default - if owner is set and repositories is empty, access will be scoped to all repositories in the provided repository owner's installation.
     * If owner and repositories are empty, access will be scoped to only the current repository.
     */
    readonly repositories?: string[];
    /**
     * The permissions granted to the token.
     *
     * @default - all permissions granted to the app
     */
    readonly permissions?: AppPermissions;
}
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
export declare class GithubCredentials {
    private readonly options;
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options?: GithubCredentialsPersonalAccessTokenOptions): GithubCredentials;
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @see https://projen.io/docs/integrations/github/#github-app
     * @default - app id stored in "PROJEN_APP_ID" and private key stored in "PROJEN_APP_PRIVATE_KEY" with all permissions attached to the app
     */
    static fromApp(options?: GithubCredentialsAppOptions): GithubCredentials;
    private constructor();
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps(): JobStep[];
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef(): string;
}
