import { GitHub } from ".";
import { Component } from "../component";
import { GroupRunnerOptions } from "../runner-options";
/**
 * Options for PullRequestLint
 */
export interface PullRequestLintOptions {
    /**
     * Validate that pull request titles follow Conventional Commits.
     *
     * @default true
     * @see https://www.conventionalcommits.org/
     */
    readonly semanticTitle?: boolean;
    /**
     * Options for validating the conventional commit title linter.
     * @default - title must start with "feat", "fix", or "chore"
     */
    readonly semanticTitleOptions?: SemanticTitleOptions;
    /**
     * Github Runner selection labels
     * @default ["ubuntu-latest"]
     * @description Defines a target Runner by labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOn?: string[];
    /**
     * Github Runner Group selection options
     * @description Defines a target Runner Group by name and/or labels
     * @throws {Error} if both `runsOn` and `runsOnGroup` are specified
     */
    readonly runsOnGroup?: GroupRunnerOptions;
    /**
     * Require a contributor statement to be included in the PR description.
     * For example confirming that the contribution has been made by the contributor and complies with the project's license.
     *
     * Appends the statement to the end of the Pull Request template.
     *
     * @default - no contributor statement is required
     */
    readonly contributorStatement?: string;
    /**
     * Options for requiring a contributor statement on Pull Requests
     * @default - none
     */
    readonly contributorStatementOptions?: ContributorStatementOptions;
}
/**
 * Options for linting that PR titles follow Conventional Commits.
 * @see https://www.conventionalcommits.org/
 */
export interface SemanticTitleOptions {
    /**
     * Configure a list of commit types that are allowed.
     * @default ["feat", "fix", "chore"]
     */
    readonly types?: string[];
    /**
     * Configure that a scope must always be provided.
     * e.g. feat(ui), fix(core)
     * @default false
     */
    readonly requireScope?: boolean;
    /**
     * Configure which scopes are allowed (newline-delimited).
     * These are regex patterns auto-wrapped in `^ $`.
     *
     * @default - all scopes allowed
     */
    readonly scopes?: string[];
}
/**
 * Options for requiring a contributor statement on Pull Requests
 */
export interface ContributorStatementOptions {
    /**
     * Pull requests from these GitHub users are exempted from a contributor statement.
     * @default - no users are exempted
     */
    readonly exemptUsers?: string[];
    /**
     * Pull requests with one of these labels are exempted from a contributor statement.
     * @default - no labels are excluded
     */
    readonly exemptLabels?: string[];
}
/**
 * Configure validations to run on GitHub pull requests.
 * Only generates a file if at least one linter is configured.
 */
export declare class PullRequestLint extends Component {
    private readonly github;
    private readonly options;
    constructor(github: GitHub, options?: PullRequestLintOptions);
    preSynthesize(): void;
}
