import { GitHub } from "./github";
import { GithubCredentials } from "./github-credentials";
import * as workflows from "./workflows-model";
import { Component } from "../component";
import { YamlFile } from "../yaml";
/**
 * Options for `concurrency`.
 */
export interface ConcurrencyOptions {
    /**
     * Concurrency group controls which workflow runs will share the same concurrency limit.
     * For example, if you specify `${{ github.workflow }}-${{ github.ref }}`, workflow runs triggered
     * on the same branch cannot run concurrenty, but workflows runs triggered on different branches can.
     *
     * @default - ${{ github.workflow }}
     *
     * @see https://docs.github.com/en/actions/writing-workflows/choosing-what-your-workflow-does/using-concurrency#example-concurrency-groups
     */
    readonly group?: string;
    /**
     * When a workflow is triggered while another one (in the same group) is running, should GitHub cancel
     * the running workflow?
     *
     * @default false
     */
    readonly cancelInProgress?: boolean;
}
/**
 * Options for `GithubWorkflow`.
 */
export interface GithubWorkflowOptions {
    /**
     * Force the creation of the workflow even if `workflows` is disabled in `GitHub`.
     *
     * @default false
     */
    readonly force?: boolean;
    /**
     * Enable concurrency limitations. Use `concurrencyOptions` to configure specific non default values.
     *
     * @default false
     */
    readonly limitConcurrency?: boolean;
    /**
     * Concurrency ensures that only a single job or workflow using the same concurrency group will run at a time. Currently in beta.
     *
     * @default - { group: ${{ github.workflow }}, cancelInProgress: false }
     *
     * @see https://docs.github.com/en/actions/learn-github-actions/workflow-syntax-for-github-actions#concurrency
     */
    readonly concurrencyOptions?: ConcurrencyOptions;
    /**
     * Additional environment variables to set for the workflow.
     *
     * @default - no additional environment variables
     */
    readonly env?: Record<string, string>;
    /**
     * Set a custom file name for the workflow definition file. Must include either a .yml or .yaml file extension.
     *
     * Use this option to set a file name for the workflow file, that is different than the display name.
     *
     * @example "build-new.yml"
     * @example "my-workflow.yaml"
     *
     * @default - a path-safe version of the workflow name plus the .yml file ending, e.g. build.yml
     */
    readonly fileName?: string;
}
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
export declare class GithubWorkflow extends Component {
    /**
     * The name of the workflow. GitHub displays the names of your workflows under your repository's
     * "Actions" tab.
     * @see https://docs.github.com/en/actions/writing-workflows/workflow-syntax-for-github-actions#name
     */
    readonly name: string;
    /**
     * All current jobs of the workflow.
     *
     * This is a read-only copy, use the respective helper methods to add, update or remove jobs.
     */
    get jobs(): Record<string, workflows.Job | workflows.JobCallingReusableWorkflow>;
    /**
     * The concurrency configuration of the workflow. undefined means no concurrency limitations.
     */
    readonly concurrency?: ConcurrencyOptions;
    /**
     * Additional environment variables to set for the workflow.
     */
    readonly env?: Record<string, string>;
    /**
     * The workflow YAML file. May not exist if `workflowsEnabled` is false on `GitHub`.
     */
    readonly file: YamlFile | undefined;
    /**
     * GitHub API authentication method used by projen workflows.
     */
    readonly projenCredentials: GithubCredentials;
    /**
     * The name for workflow runs generated from the workflow. GitHub displays the
     * workflow run name in the list of workflow runs on your repository's
     * "Actions" tab. If `run-name` is omitted or is only whitespace, then the run
     * name is set to event-specific information for the workflow run. For
     * example, for a workflow triggered by a `push` or `pull_request` event, it
     * is set as the commit message.
     *
     * This value can include expressions and can reference `github` and `inputs`
     * contexts.
     */
    runName?: string;
    private readonly _jobs;
    private actions;
    private events;
    /**
     * @param github The GitHub component of the project this workflow belongs to.
     * @param name The name of the workflow, displayed under the repository's "Actions" tab.
     * @param options Additional options to configure the workflow.
     */
    constructor(github: GitHub, name: string, options?: GithubWorkflowOptions);
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events: workflows.Triggers): void;
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id: string, job: workflows.Job | workflows.JobCallingReusableWorkflow): void;
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs: Record<string, workflows.Job | workflows.JobCallingReusableWorkflow>): void;
    /**
     * Get a single job from the workflow.
     * @param id The job name (unique within the workflow)
     */
    getJob(id: string): workflows.Job | workflows.JobCallingReusableWorkflow;
    /**
     * Updates a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    updateJob(id: string, job: workflows.Job | workflows.JobCallingReusableWorkflow): void;
    /**
     * Updates jobs for this workflow
     * Does a complete replace, it does not try to merge the jobs
     *
     * @param jobs Jobs to update.
     */
    updateJobs(jobs: Record<string, workflows.Job | workflows.JobCallingReusableWorkflow>): void;
    /**
     * Removes a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    removeJob(id: string): void;
    private renderWorkflow;
}
