"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TypescriptConfig = exports.TypescriptConfigExtends = exports.TypeScriptJsxMode = exports.TypeScriptImportsNotUsedAsValues = exports.TypeScriptModuleDetection = exports.TypeScriptModuleResolution = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const semver = require("semver");
const _1 = require(".");
const component_1 = require("../component");
const json_1 = require("../json");
const util_1 = require("../util");
/**
 * Determines how modules get resolved.
 *
 * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html
 */
var TypeScriptModuleResolution;
(function (TypeScriptModuleResolution) {
    /**
     * TypeScript's former default resolution strategy.
     *
     * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html#classic
     */
    TypeScriptModuleResolution["CLASSIC"] = "classic";
    /**
     * Resolution strategy which attempts to mimic the Node.js module resolution strategy at runtime.
     *
     * @see https://www.typescriptlang.org/docs/handbook/module-resolution.html#node
     */
    TypeScriptModuleResolution["NODE"] = "node";
    /**
     * `--moduleResolution node` was renamed to `node10`
     * (keeping `node` as an alias for backward compatibility) in TypeScript 5.0.
     * It reflects the CommonJS module resolution algorithm as it existed in Node.js versions
     * earlier than v12. It should no longer be used.
     *
     * @see https://www.typescriptlang.org/docs/handbook/modules/reference.html#node10-formerly-known-as-node
     */
    TypeScriptModuleResolution["NODE10"] = "node10";
    /**
     * Node.js’ ECMAScript Module Support from TypeScript 4.7 onwards
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["NODE16"] = "node16";
    /**
     * Node.js’ ECMAScript Module Support from TypeScript 4.7 onwards
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["NODE_NEXT"] = "nodenext";
    /**
     * Resolution strategy which attempts to mimic resolution patterns of modern bundlers; from TypeScript 5.0 onwards.
     *
     * @see https://www.typescriptlang.org/tsconfig#moduleResolution
     */
    TypeScriptModuleResolution["BUNDLER"] = "bundler";
})(TypeScriptModuleResolution || (exports.TypeScriptModuleResolution = TypeScriptModuleResolution = {}));
/**
 * This setting controls how TypeScript determines whether a file is a script or a module.
 *
 * @see https://www.typescriptlang.org/docs/handbook/modules/theory.html#scripts-and-modules-in-javascript
 */
var TypeScriptModuleDetection;
(function (TypeScriptModuleDetection) {
    /**
     * TypeScript will not only look for import and export statements, but it will also check whether the "type" field in a package.json is set to "module" when running with module: nodenext or node16, and check whether the current file is a JSX file when running under jsx: react-jsx.
     *
     * @see https://www.typescriptlang.org/tsconfig/#moduleDetection
     */
    TypeScriptModuleDetection["AUTO"] = "auto";
    /**
     * The same behavior as 4.6 and prior, usings import and export statements to determine whether a file is a module.
     *
     * @see https://www.typescriptlang.org/tsconfig/#moduleDetection
     */
    TypeScriptModuleDetection["LEGACY"] = "legacy";
    /**
     * Ensures that every non-declaration file is treated as a module.
     *
     * @see https://www.typescriptlang.org/tsconfig/#moduleDetection
     */
    TypeScriptModuleDetection["FORCE"] = "force";
})(TypeScriptModuleDetection || (exports.TypeScriptModuleDetection = TypeScriptModuleDetection = {}));
/**
 * This flag controls how `import` works, there are 3 different options.
 *
 * @see https://www.typescriptlang.org/tsconfig#importsNotUsedAsValues
 */
var TypeScriptImportsNotUsedAsValues;
(function (TypeScriptImportsNotUsedAsValues) {
    /**
     * The default behavior of dropping `import` statements which only reference types.
     */
    TypeScriptImportsNotUsedAsValues["REMOVE"] = "remove";
    /**
     * Preserves all `import` statements whose values or types are never used. This can cause imports/side-effects to be preserved.
     */
    TypeScriptImportsNotUsedAsValues["PRESERVE"] = "preserve";
    /**
     * This preserves all imports (the same as the preserve option), but will error when a value import is only used as a type.
     * This might be useful if you want to ensure no values are being accidentally imported, but still make side-effect imports explicit.
     */
    TypeScriptImportsNotUsedAsValues["ERROR"] = "error";
})(TypeScriptImportsNotUsedAsValues || (exports.TypeScriptImportsNotUsedAsValues = TypeScriptImportsNotUsedAsValues = {}));
/**
 * Determines how JSX should get transformed into valid JavaScript.
 *
 * @see https://www.typescriptlang.org/docs/handbook/jsx.html
 */
var TypeScriptJsxMode;
(function (TypeScriptJsxMode) {
    /**
     * Keeps the JSX as part of the output to be further consumed by another transform step (e.g. Babel).
     */
    TypeScriptJsxMode["PRESERVE"] = "preserve";
    /**
     * Converts JSX syntax into React.createElement, does not need to go through a JSX transformation before use, and the output will have a .js file extension.
     */
    TypeScriptJsxMode["REACT"] = "react";
    /**
     * Keeps all JSX like 'preserve' mode, but output will have a .js extension.
     */
    TypeScriptJsxMode["REACT_NATIVE"] = "react-native";
    /**
     * Passes `key` separately from props and always passes `children` as props (since React 17).
     *
     * @see https://www.typescriptlang.org/docs/handbook/release-notes/typescript-4-1.html#react-17-jsx-factories
     */
    TypeScriptJsxMode["REACT_JSX"] = "react-jsx";
    /**
     * Same as `REACT_JSX` with additional debug data.
     */
    TypeScriptJsxMode["REACT_JSXDEV"] = "react-jsxdev";
})(TypeScriptJsxMode || (exports.TypeScriptJsxMode = TypeScriptJsxMode = {}));
/**
 * Container for `TypescriptConfig` `tsconfig.json` base configuration(s).
 * Extending from more than one base config file requires TypeScript 5.0+.
 */
class TypescriptConfigExtends {
    /**
     * Factory for creation from array of file paths.
     *
     * @remarks
     * TypeScript 5.0+ is required to specify more than one value in `paths`.
     *
     * @param paths Absolute or relative paths to base `tsconfig.json` files.
     */
    static fromPaths(paths) {
        return new TypescriptConfigExtends(paths);
    }
    /**
     * Factory for creation from array of other `TypescriptConfig` instances.
     *
     * @remarks
     * TypeScript 5.0+ is required to specify more than on value in `configs`.
     *
     * @param configs Base `TypescriptConfig` instances.
     */
    static fromTypescriptConfigs(configs) {
        const paths = configs.map((config) => config.file.absolutePath);
        return TypescriptConfigExtends.fromPaths(paths);
    }
    constructor(bases) {
        this.bases = bases;
    }
    toJSON() {
        return this.bases;
    }
}
exports.TypescriptConfigExtends = TypescriptConfigExtends;
_a = JSII_RTTI_SYMBOL_1;
TypescriptConfigExtends[_a] = { fqn: "projen.javascript.TypescriptConfigExtends", version: "0.98.20" };
class TypescriptConfig extends component_1.Component {
    constructor(project, options) {
        super(project);
        const fileName = options.fileName ?? "tsconfig.json";
        if (!options.extends && !options.compilerOptions) {
            throw new Error("TypescriptConfig: Must provide either `extends` or `compilerOptions` (or both).");
        }
        this._extends = options.extends ?? TypescriptConfigExtends.fromPaths([]);
        this.includeSet = options.include
            ? new Set(options.include)
            : new Set(["**/*.ts"]);
        this.excludeSet = options.exclude
            ? new Set(options.exclude)
            : new Set(["node_modules"]);
        this.fileName = fileName;
        this.compilerOptions = options.compilerOptions;
        this.file = new json_1.JsonFile(project, fileName, {
            allowComments: true,
            obj: {
                extends: () => this.renderExtends(),
                compilerOptions: this.compilerOptions,
                include: () => this.include,
                exclude: () => this.exclude,
            },
        });
        if (project instanceof _1.NodeProject) {
            project.npmignore?.exclude(`/${fileName}`);
        }
    }
    get include() {
        return [...this.includeSet];
    }
    get exclude() {
        return [...this.excludeSet];
    }
    /**
     * Render appropriate value for `extends` field.
     * @private
     */
    renderExtends() {
        if (this.extends.length <= 1) {
            // render string value when only one extension (TS<5.0);
            // omit if no extensions.
            return this.extends[0];
        }
        // render many extensions as array (TS>=5.0)
        return this.extends;
    }
    /**
     * Resolve valid TypeScript extends paths relative to this config.
     *
     * @remarks
     * This will only resolve the relative path from this config to another given
     * an absolute path as input. Any non-absolute path or other string will be returned as is.
     * This is to preserve manually specified relative paths as well as npm import paths.
     *
     * @param configPath Path to resolve against.
     */
    resolveExtendsPath(configPath) {
        // if not absolute assume user-defined path (or npm package).
        if (!path.isAbsolute(configPath))
            return configPath;
        const relativeConfig = path.relative(path.dirname(this.file.absolutePath), configPath);
        // ensure immediate sibling files are prefixed with './'
        // typescript figures this out, but some tools seemingly break without it (i.e, eslint).
        const { dir, ...pathParts } = path.parse(relativeConfig);
        const configDir = dir
            ? path.format({ dir: dir.startsWith("..") ? "" : ".", base: dir })
            : ".";
        const extendsPath = path.format({ ...pathParts, dir: configDir });
        return (0, util_1.normalizePersistedPath)(extendsPath);
    }
    /**
     * Validate usage of `extends` against current TypeScript version.
     * @private
     */
    validateExtends() {
        const project = this.project;
        const hasOneOrNoneExtends = this.extends.length <= 1;
        const isNodeProject = project instanceof _1.NodeProject;
        if (hasOneOrNoneExtends || !isNodeProject) {
            // always accept no extends or singular extends.
            return;
        }
        const tsVersion = semver.coerce(project.package.tryResolveDependencyVersion("typescript"), { loose: true });
        if (tsVersion && tsVersion.major < 5) {
            this.project.logger.warn("TypeScript < 5.0.0 can only extend from a single base config.", `TypeScript Version: ${tsVersion.format()}`, `File: ${this.file.absolutePath}`, `Extends: ${this.extends}`);
        }
    }
    /**
     * Array of base `tsconfig.json` paths.
     * Any absolute paths are resolved relative to this instance,
     * while any relative paths are used as is.
     */
    get extends() {
        return this._extends
            .toJSON()
            .map((value) => this.resolveExtendsPath(value));
    }
    /**
     * Extend from base `TypescriptConfig` instance.
     *
     * @remarks
     * TypeScript 5.0+ is required to extend from more than one base `TypescriptConfig`.
     *
     * @param value Base `TypescriptConfig` instance.
     */
    addExtends(value) {
        this._extends = TypescriptConfigExtends.fromPaths([
            ...this._extends.toJSON(),
            value.file.absolutePath,
        ]);
    }
    /**
     * Add an include pattern to the `include` array of the TSConfig.
     *
     * @see https://www.typescriptlang.org/tsconfig#include
     *
     * @param pattern The pattern to add.
     */
    addInclude(pattern) {
        this.include.push(pattern);
        this.includeSet.add(pattern);
    }
    /**
     * Add an exclude pattern to the `exclude` array of the TSConfig.
     *
     * @see https://www.typescriptlang.org/tsconfig#exclude
     *
     * @param pattern The pattern to add.
     */
    addExclude(pattern) {
        this.exclude.push(pattern);
        this.excludeSet.add(pattern);
    }
    /**
     * Remove an include pattern from the `include` array of the TSConfig.
     *
     * @see https://www.typescriptlang.org/tsconfig#include
     *
     * @param pattern The pattern to remove.
     */
    removeInclude(pattern) {
        this.includeSet.delete(pattern);
    }
    /**
     * Remove an exclude pattern from the `exclude` array of the TSConfig.
     *
     * @see https://www.typescriptlang.org/tsconfig#exclude
     *
     * @param pattern The pattern to remove.
     */
    removeExclude(pattern) {
        this.excludeSet.delete(pattern);
    }
    preSynthesize() {
        super.preSynthesize();
        this.validateExtends();
    }
}
exports.TypescriptConfig = TypescriptConfig;
_b = JSII_RTTI_SYMBOL_1;
TypescriptConfig[_b] = { fqn: "projen.javascript.TypescriptConfig", version: "0.98.20" };
//# sourceMappingURL=data:application/json;base64,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