"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const constructs_1 = require("constructs");
const glob = require("fast-glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
const constructs_2 = require("./util/constructs");
/**
 * The default output directory for a project if none is specified.
 */
const DEFAULT_OUTDIR = ".";
/**
 * Base project
 */
class Project extends constructs_1.Construct {
    /**
     * Test whether the given construct is a project.
     */
    static isProject(x) {
        return (0, constructs_2.isProject)(x);
    }
    /**
     * Find the closest ancestor project for given construct.
     * When given a project, this it the project itself.
     *
     * @throws when no project is found in the path to the root
     */
    static of(construct) {
        return (0, constructs_2.findClosestProject)(construct, this.name);
    }
    /**
     * The command to use in order to run the projen CLI.
     */
    get projenCommand() {
        return this._projenCommand ?? "npx projen";
    }
    constructor(options) {
        const outdir = determineOutdir(options.parent, options.outdir);
        const autoId = `${new.target.name}#${options.name}@${path.normalize(options.outdir ?? "<root>")}`;
        if (options.parent?.subprojects.find((p) => p.outdir === outdir)) {
            throw new Error(`There is already a subproject with "outdir": ${outdir}`);
        }
        super(options.parent, autoId);
        this.tips = new Array();
        (0, constructs_2.tagAsProject)(this);
        this.node.addMetadata("type", "project");
        this.node.addMetadata("construct", new.target.name);
        this.initProject = (0, render_options_1.resolveInitProject)(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = (0, util_1.isTruthy)(process.env.PROJEN_EJECTING);
        this._projenCommand = options.projenCommand;
        if (this.ejected) {
            this._projenCommand = "scripts/run-task.cjs";
        }
        this.outdir = outdir;
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this, {
            endOfLine: options.gitOptions?.endOfLine,
        });
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore", options.gitIgnoreOptions);
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injection but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            // Subtasks should call the root task for synth
            if (this.parent) {
                const cwd = path.relative(this.outdir, this.root.outdir);
                const normalizedCwd = (0, util_1.normalizePersistedPath)(cwd);
                this.defaultTask.exec(`${this.projenCommand} ${Project.DEFAULT_TASK}`, {
                    cwd: normalizedCwd,
                });
            }
            if (!this.parent) {
                this.ejectTask = this.tasks.addTask("eject", {
                    description: "Remove projen from the project",
                    env: {
                        PROJEN_EJECTING: "true",
                    },
                });
                this.ejectTask.spawn(this.defaultTask);
            }
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (!this.parent && projenrcJson) {
            new projenrc_json_1.ProjenrcJson(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => (0, util_1.normalizePersistedPath)(f.path)),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * The root project.
     */
    get root() {
        return (0, constructs_2.isProject)(this.node.root) ? this.node.root : this;
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return this.node
            .findAll()
            .filter((c) => (0, constructs_2.isComponent)(c) && c.project.node.path === this.node.path);
    }
    /**
     * Returns all the subprojects within this project.
     */
    get subprojects() {
        return this.node.children.filter(constructs_2.isProject);
    }
    /**
     * All files in this project.
     */
    get files() {
        return this.components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const candidate = this.node
            .findAll()
            .find((c) => (0, constructs_2.isComponent)(c) && isFile(c) && c.absolutePath === absolute);
        return candidate;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const candidate = this.tryFindFile(filePath);
        if (candidate) {
            candidate.node.scope?.node.tryRemoveChild(candidate.node.id);
            return candidate;
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        const pj = this._projenCommand ?? `npx projen@${common_1.PROJEN_VERSION}`;
        return `${pj} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all subprojects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this.components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        (0, cleanup_1.cleanup)(outdir, this.files.map((f) => (0, util_1.normalizePersistedPath)(f.path)), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this.components) {
            comp.synthesize();
        }
        if (!(0, util_1.isTruthy)(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this.components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                onlyFiles: true,
                followSymbolicLinks: false,
                absolute: true,
            });
            for (const file of files) {
                (0, fs_1.renameSync)(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.98.20" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType || (exports.ProjectType = ProjectType = {}));
/**
 * Resolves the project's output directory.
 */
function determineOutdir(parent, outdirOption) {
    if (parent && outdirOption && path.isAbsolute(outdirOption)) {
        throw new Error('"outdir" must be a relative path');
    }
    // if this is a subproject, it is relative to the parent
    if (parent) {
        if (!outdirOption) {
            throw new Error('"outdir" must be specified for subprojects');
        }
        return path.resolve(parent.outdir, outdirOption);
    }
    // if this is running inside a test and outdir is not explicitly set
    // use a temp directory (unless cwd is already under tmp)
    if (common_1.IS_TEST_RUN && !outdirOption) {
        const realCwd = (0, fs_1.realpathSync)(process.cwd());
        const realTmp = (0, fs_1.realpathSync)((0, os_1.tmpdir)());
        if (realCwd.startsWith(realTmp)) {
            return path.resolve(realCwd, outdirOption ?? DEFAULT_OUTDIR);
        }
        return (0, fs_1.mkdtempSync)(path.join((0, os_1.tmpdir)(), "projen."));
    }
    return path.resolve(outdirOption ?? DEFAULT_OUTDIR);
}
function isFile(c) {
    return c instanceof file_1.FileBase;
}
//# sourceMappingURL=data:application/json;base64,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