"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Projenrc = void 0;
exports.resolvePythonImportName = resolvePythonImportName;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const case_1 = require("case");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const inventory_1 = require("../inventory");
const projenrc_1 = require("../projenrc");
/**
 * Allows writing projenrc files in python.
 *
 * This will install `projen` as a Python dependency and will add a
 * `synth` task which will run `.projenrc.py`.
 */
class Projenrc extends projenrc_1.ProjenrcFile {
    constructor(project, options = {}) {
        super(project);
        const projenVersion = options.projenVersion ?? common_1.PROJEN_VERSION;
        this.filePath = options.filename ?? ".projenrc.py";
        this.pythonExec = options.pythonExec ?? "python";
        project.deps.addDependency(`projen@${projenVersion}`, dependencies_1.DependencyType.DEVENV);
        // set up the "default" task which is the task executed when `projen` is executed for this project.
        project.defaultTask?.exec(`${this.pythonExec} .projenrc.py`);
        // if this is a new project, generate a skeleton for projenrc.py
        this.generateProjenrc();
    }
    generateProjenrc() {
        const bootstrap = this.project.initProject;
        if (!bootstrap) {
            return;
        }
        const jsiiFqn = bootstrap.fqn;
        const jsiiManifest = (0, inventory_1.readJsiiManifest)(jsiiFqn);
        const jsiiType = jsiiManifest.types[jsiiFqn];
        const optionsTypeFqn = jsiiType.initializer?.parameters?.[0].type?.fqn;
        if (!optionsTypeFqn) {
            this.project.logger.warn("cannot determine jsii type for project options");
            return;
        }
        const pythonFile = (0, path_1.join)(this.project.outdir, this.filePath);
        // skip if file exists
        if ((0, fs_1.existsSync)(pythonFile)) {
            return;
        }
        const lines = new Array();
        let indent = 0;
        const emit = (line = "") => lines.push(" ".repeat(indent * 4) + line);
        // const openBlock = (line: string = '') => { emit(line + ' {'); indent++; };
        // const closeBlock = () => { indent--; emit('}'); };
        const optionFqns = {};
        for (const option of bootstrap.type.options) {
            if (option.fqn) {
                optionFqns[option.name] = option.fqn;
            }
        }
        const { renderedOptions, imports } = renderPythonOptions(indent, optionFqns, bootstrap.args);
        const importName = resolvePythonImportName(jsiiFqn, jsiiManifest);
        emit(toPythonImport(importName));
        for (const fqn of imports) {
            emit(toPythonImport(fqn));
        }
        emit();
        emit(`project = ${jsiiType.name}(${renderedOptions})`);
        emit();
        emit("project.synth()");
        (0, fs_1.mkdirSync)((0, path_1.dirname)(pythonFile), { recursive: true });
        (0, fs_1.writeFileSync)(pythonFile, lines.join("\n"));
        this.project.logger.info(`Project definition file was created at ${pythonFile}`);
    }
}
exports.Projenrc = Projenrc;
_a = JSII_RTTI_SYMBOL_1;
Projenrc[_a] = { fqn: "projen.python.Projenrc", version: "0.98.20" };
function resolvePythonImportName(jsiiFqn, jsiiManifest) {
    const moduleName = jsiiManifest?.targets?.python?.module;
    // Module name prefix should take precedence in the event moduleName !== fqn prefix
    return moduleName
        ? [moduleName, ...jsiiFqn.split(".").slice(1)].join(".")
        : jsiiFqn;
}
function renderPythonOptions(indent, optionFqns, initOptions) {
    const imports = new Set();
    if (!initOptions || Object.keys(initOptions).length === 0) {
        // no options
        return { renderedOptions: "", imports };
    }
    const lines = [""];
    for (const [name, value] of Object.entries(initOptions)) {
        const { pythonValue, importName } = toPythonValue(value, name, optionFqns);
        if (importName)
            imports.add(importName);
        lines.push(`${toPythonProperty(name)}=${pythonValue},`);
    }
    const renderedOptions = lines
        .join(`\n${" ".repeat((indent + 1) * 4)}`)
        .concat("\n");
    return { renderedOptions, imports };
}
function toPythonProperty(prop) {
    return (0, case_1.snake)(prop);
}
function toPythonValue(value, name, optionFqns) {
    if (typeof value === "boolean") {
        return { pythonValue: value ? "True" : "False" };
    }
    else if (typeof value === "number") {
        return { pythonValue: JSON.stringify(value) };
    }
    else if (typeof value === "string") {
        if (optionFqns[name] !== undefined) {
            const parts = optionFqns[name].split(".");
            const base = parts[parts.length - 1];
            const choice = String(value).toUpperCase().replace(/-/g, "_");
            return { pythonValue: `${base}.${choice}`, importName: optionFqns[name] };
        }
        else {
            return { pythonValue: JSON.stringify(value) };
        }
    }
    else if (value === undefined || value === null) {
        return { pythonValue: "None" };
    }
    else {
        return { pythonValue: JSON.stringify(value) };
    }
}
function toPythonImport(fqn) {
    const parts = fqn.split(".");
    if (parts.length === 1) {
        return `import ${parts[0]}`;
    }
    else {
        return `from ${parts.slice(0, -1).join(".")} import ${parts[parts.length - 1]}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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