"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Resolves the latest version from git tags and uses `commit-and-tag-version` to bump
 * to the next version based on commits.
 *
 * This expects `commit-and-tag-version` to be installed in the path.
 *
 * Environment variables:
 *
 * - OUTFILE: (required) the name of the JSON output file (the "version" field
 *   will be updated with the latest version)
 * - PRERELEASE: (optional) a prerelease tag to use (e.g. "beta")
 * - MAJOR: major version number NN to filter (tags are filtered by "vNN."
 *   prefix). if not specified, the last major version is selected
 * - MIN_MAJOR: minimum major version number to use
 * - CHANGELOG: name of changelog file to create
 * - RELEASE_TAG_PREFIX: (optional) a prefix to apply to the release tag
 * - BUMP_PACKAGE: (optional) the bump package to use
 * - NEXT_VERSION_COMMAND: (optional) a command to decide the version to release
 */
const bump_version_1 = require("./bump-version");
const logging = require("../logging");
const versionFile = process.env.OUTFILE;
const prerelease = process.env.PRERELEASE;
const major = process.env.MAJOR;
const minMajor = process.env.MIN_MAJOR;
const changelog = process.env.CHANGELOG;
const bumpFile = process.env.BUMPFILE;
const releaseTagFile = process.env.RELEASETAG;
const prefix = process.env.RELEASE_TAG_PREFIX;
const versionrcOptions = process.env.VERSIONRCOPTIONS;
const releasableCommits = process.env.RELEASABLE_COMMITS;
const bumpPackage = process.env.BUMP_PACKAGE;
const nextVersionCommand = process.env.NEXT_VERSION_COMMAND;
if (!versionFile) {
    throw new Error("OUTFILE is required");
}
if (!changelog) {
    throw new Error("CHANGELOG is required");
}
if (!bumpFile) {
    throw new Error("BUMPFILE is required");
}
if (!releaseTagFile) {
    throw new Error("RELEASETAG is required");
}
const majorVersion = major == null || major === "" ? undefined : parseInt(major);
if (Number.isNaN(majorVersion)) {
    throw new Error(`MAJOR must be a number: ${majorVersion}`);
}
const minMajorVersion = minMajor == null || minMajor === "" ? undefined : parseInt(minMajor);
if (Number.isNaN(minMajorVersion)) {
    throw new Error(`minMajor must be a number: ${minMajorVersion}`);
}
const opts = {
    versionFile: versionFile,
    changelog: changelog,
    majorVersion: majorVersion,
    minMajorVersion: minMajorVersion,
    prerelease: prerelease,
    bumpFile: bumpFile,
    releaseTagFile: releaseTagFile,
    tagPrefix: prefix,
    // doesn't work with long customization
    versionrcOptions: JSON.parse(versionrcOptions ?? "{}"),
    releasableCommits,
    bumpPackage,
    nextVersionCommand: nextVersionCommand ? nextVersionCommand : undefined,
};
logging.debug(opts);
(0, bump_version_1.bump)(process.cwd(), opts).catch((e) => {
    console.log(e.stack);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVtcC12ZXJzaW9uLnRhc2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvcmVsZWFzZS9idW1wLXZlcnNpb24udGFzay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQkc7QUFDSCxpREFBbUQ7QUFDbkQsc0NBQXNDO0FBRXRDLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDO0FBQ3hDLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDO0FBQzFDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDO0FBQ2hDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDO0FBQ3ZDLE1BQU0sU0FBUyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDO0FBQ3hDLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDO0FBQ3RDLE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDO0FBQzlDLE1BQU0sTUFBTSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUM7QUFDOUMsTUFBTSxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDO0FBQ3RELE1BQU0saUJBQWlCLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQztBQUN6RCxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQztBQUM3QyxNQUFNLGtCQUFrQixHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsb0JBQW9CLENBQUM7QUFFNUQsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLENBQUMsQ0FBQztBQUN6QyxDQUFDO0FBRUQsSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO0lBQ2YsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0FBQzNDLENBQUM7QUFFRCxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUVELElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQztJQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixDQUFDLENBQUM7QUFDNUMsQ0FBQztBQUVELE1BQU0sWUFBWSxHQUNoQixLQUFLLElBQUksSUFBSSxJQUFJLEtBQUssS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzlELElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDO0lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLFlBQVksRUFBRSxDQUFDLENBQUM7QUFDN0QsQ0FBQztBQUVELE1BQU0sZUFBZSxHQUNuQixRQUFRLElBQUksSUFBSSxJQUFJLFFBQVEsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ3ZFLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDO0lBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsOEJBQThCLGVBQWUsRUFBRSxDQUFDLENBQUM7QUFDbkUsQ0FBQztBQUVELE1BQU0sSUFBSSxHQUFnQjtJQUN4QixXQUFXLEVBQUUsV0FBVztJQUN4QixTQUFTLEVBQUUsU0FBUztJQUNwQixZQUFZLEVBQUUsWUFBWTtJQUMxQixlQUFlLEVBQUUsZUFBZTtJQUNoQyxVQUFVLEVBQUUsVUFBVTtJQUN0QixRQUFRLEVBQUUsUUFBUTtJQUNsQixjQUFjLEVBQUUsY0FBYztJQUM5QixTQUFTLEVBQUUsTUFBTTtJQUNqQix1Q0FBdUM7SUFDdkMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUM7SUFDdEQsaUJBQWlCO0lBQ2pCLFdBQVc7SUFDWCxrQkFBa0IsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLFNBQVM7Q0FDeEUsQ0FBQztBQUNGLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7QUFFcEIsSUFBQSxtQkFBSSxFQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsRUFBRSxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFRLEVBQUUsRUFBRTtJQUMzQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNyQixPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ2xCLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBSZXNvbHZlcyB0aGUgbGF0ZXN0IHZlcnNpb24gZnJvbSBnaXQgdGFncyBhbmQgdXNlcyBgY29tbWl0LWFuZC10YWctdmVyc2lvbmAgdG8gYnVtcFxuICogdG8gdGhlIG5leHQgdmVyc2lvbiBiYXNlZCBvbiBjb21taXRzLlxuICpcbiAqIFRoaXMgZXhwZWN0cyBgY29tbWl0LWFuZC10YWctdmVyc2lvbmAgdG8gYmUgaW5zdGFsbGVkIGluIHRoZSBwYXRoLlxuICpcbiAqIEVudmlyb25tZW50IHZhcmlhYmxlczpcbiAqXG4gKiAtIE9VVEZJTEU6IChyZXF1aXJlZCkgdGhlIG5hbWUgb2YgdGhlIEpTT04gb3V0cHV0IGZpbGUgKHRoZSBcInZlcnNpb25cIiBmaWVsZFxuICogICB3aWxsIGJlIHVwZGF0ZWQgd2l0aCB0aGUgbGF0ZXN0IHZlcnNpb24pXG4gKiAtIFBSRVJFTEVBU0U6IChvcHRpb25hbCkgYSBwcmVyZWxlYXNlIHRhZyB0byB1c2UgKGUuZy4gXCJiZXRhXCIpXG4gKiAtIE1BSk9SOiBtYWpvciB2ZXJzaW9uIG51bWJlciBOTiB0byBmaWx0ZXIgKHRhZ3MgYXJlIGZpbHRlcmVkIGJ5IFwidk5OLlwiXG4gKiAgIHByZWZpeCkuIGlmIG5vdCBzcGVjaWZpZWQsIHRoZSBsYXN0IG1ham9yIHZlcnNpb24gaXMgc2VsZWN0ZWRcbiAqIC0gTUlOX01BSk9SOiBtaW5pbXVtIG1ham9yIHZlcnNpb24gbnVtYmVyIHRvIHVzZVxuICogLSBDSEFOR0VMT0c6IG5hbWUgb2YgY2hhbmdlbG9nIGZpbGUgdG8gY3JlYXRlXG4gKiAtIFJFTEVBU0VfVEFHX1BSRUZJWDogKG9wdGlvbmFsKSBhIHByZWZpeCB0byBhcHBseSB0byB0aGUgcmVsZWFzZSB0YWdcbiAqIC0gQlVNUF9QQUNLQUdFOiAob3B0aW9uYWwpIHRoZSBidW1wIHBhY2thZ2UgdG8gdXNlXG4gKiAtIE5FWFRfVkVSU0lPTl9DT01NQU5EOiAob3B0aW9uYWwpIGEgY29tbWFuZCB0byBkZWNpZGUgdGhlIHZlcnNpb24gdG8gcmVsZWFzZVxuICovXG5pbXBvcnQgeyBidW1wLCBCdW1wT3B0aW9ucyB9IGZyb20gXCIuL2J1bXAtdmVyc2lvblwiO1xuaW1wb3J0ICogYXMgbG9nZ2luZyBmcm9tIFwiLi4vbG9nZ2luZ1wiO1xuXG5jb25zdCB2ZXJzaW9uRmlsZSA9IHByb2Nlc3MuZW52Lk9VVEZJTEU7XG5jb25zdCBwcmVyZWxlYXNlID0gcHJvY2Vzcy5lbnYuUFJFUkVMRUFTRTtcbmNvbnN0IG1ham9yID0gcHJvY2Vzcy5lbnYuTUFKT1I7XG5jb25zdCBtaW5NYWpvciA9IHByb2Nlc3MuZW52Lk1JTl9NQUpPUjtcbmNvbnN0IGNoYW5nZWxvZyA9IHByb2Nlc3MuZW52LkNIQU5HRUxPRztcbmNvbnN0IGJ1bXBGaWxlID0gcHJvY2Vzcy5lbnYuQlVNUEZJTEU7XG5jb25zdCByZWxlYXNlVGFnRmlsZSA9IHByb2Nlc3MuZW52LlJFTEVBU0VUQUc7XG5jb25zdCBwcmVmaXggPSBwcm9jZXNzLmVudi5SRUxFQVNFX1RBR19QUkVGSVg7XG5jb25zdCB2ZXJzaW9ucmNPcHRpb25zID0gcHJvY2Vzcy5lbnYuVkVSU0lPTlJDT1BUSU9OUztcbmNvbnN0IHJlbGVhc2FibGVDb21taXRzID0gcHJvY2Vzcy5lbnYuUkVMRUFTQUJMRV9DT01NSVRTO1xuY29uc3QgYnVtcFBhY2thZ2UgPSBwcm9jZXNzLmVudi5CVU1QX1BBQ0tBR0U7XG5jb25zdCBuZXh0VmVyc2lvbkNvbW1hbmQgPSBwcm9jZXNzLmVudi5ORVhUX1ZFUlNJT05fQ09NTUFORDtcblxuaWYgKCF2ZXJzaW9uRmlsZSkge1xuICB0aHJvdyBuZXcgRXJyb3IoXCJPVVRGSUxFIGlzIHJlcXVpcmVkXCIpO1xufVxuXG5pZiAoIWNoYW5nZWxvZykge1xuICB0aHJvdyBuZXcgRXJyb3IoXCJDSEFOR0VMT0cgaXMgcmVxdWlyZWRcIik7XG59XG5cbmlmICghYnVtcEZpbGUpIHtcbiAgdGhyb3cgbmV3IEVycm9yKFwiQlVNUEZJTEUgaXMgcmVxdWlyZWRcIik7XG59XG5cbmlmICghcmVsZWFzZVRhZ0ZpbGUpIHtcbiAgdGhyb3cgbmV3IEVycm9yKFwiUkVMRUFTRVRBRyBpcyByZXF1aXJlZFwiKTtcbn1cblxuY29uc3QgbWFqb3JWZXJzaW9uID1cbiAgbWFqb3IgPT0gbnVsbCB8fCBtYWpvciA9PT0gXCJcIiA/IHVuZGVmaW5lZCA6IHBhcnNlSW50KG1ham9yKTtcbmlmIChOdW1iZXIuaXNOYU4obWFqb3JWZXJzaW9uKSkge1xuICB0aHJvdyBuZXcgRXJyb3IoYE1BSk9SIG11c3QgYmUgYSBudW1iZXI6ICR7bWFqb3JWZXJzaW9ufWApO1xufVxuXG5jb25zdCBtaW5NYWpvclZlcnNpb24gPVxuICBtaW5NYWpvciA9PSBudWxsIHx8IG1pbk1ham9yID09PSBcIlwiID8gdW5kZWZpbmVkIDogcGFyc2VJbnQobWluTWFqb3IpO1xuaWYgKE51bWJlci5pc05hTihtaW5NYWpvclZlcnNpb24pKSB7XG4gIHRocm93IG5ldyBFcnJvcihgbWluTWFqb3IgbXVzdCBiZSBhIG51bWJlcjogJHttaW5NYWpvclZlcnNpb259YCk7XG59XG5cbmNvbnN0IG9wdHM6IEJ1bXBPcHRpb25zID0ge1xuICB2ZXJzaW9uRmlsZTogdmVyc2lvbkZpbGUsXG4gIGNoYW5nZWxvZzogY2hhbmdlbG9nLFxuICBtYWpvclZlcnNpb246IG1ham9yVmVyc2lvbixcbiAgbWluTWFqb3JWZXJzaW9uOiBtaW5NYWpvclZlcnNpb24sXG4gIHByZXJlbGVhc2U6IHByZXJlbGVhc2UsXG4gIGJ1bXBGaWxlOiBidW1wRmlsZSxcbiAgcmVsZWFzZVRhZ0ZpbGU6IHJlbGVhc2VUYWdGaWxlLFxuICB0YWdQcmVmaXg6IHByZWZpeCxcbiAgLy8gZG9lc24ndCB3b3JrIHdpdGggbG9uZyBjdXN0b21pemF0aW9uXG4gIHZlcnNpb25yY09wdGlvbnM6IEpTT04ucGFyc2UodmVyc2lvbnJjT3B0aW9ucyA/PyBcInt9XCIpLFxuICByZWxlYXNhYmxlQ29tbWl0cyxcbiAgYnVtcFBhY2thZ2UsXG4gIG5leHRWZXJzaW9uQ29tbWFuZDogbmV4dFZlcnNpb25Db21tYW5kID8gbmV4dFZlcnNpb25Db21tYW5kIDogdW5kZWZpbmVkLFxufTtcbmxvZ2dpbmcuZGVidWcob3B0cyk7XG5cbmJ1bXAocHJvY2Vzcy5jd2QoKSwgb3B0cykuY2F0Y2goKGU6IEVycm9yKSA9PiB7XG4gIGNvbnNvbGUubG9nKGUuc3RhY2spO1xuICBwcm9jZXNzLmV4aXQoMSk7XG59KTtcbiJdfQ==