export interface ScheduledReleaseOptions {
    /**
     * Cron schedule for releases.
     *
     * Only defined if this is a scheduled release.
     *
     * @example '0 17 * * *' - every day at 5 pm
     */
    readonly schedule: string;
}
export interface ManualReleaseOptions {
    /**
     * Maintain a project-level changelog.
     *
     * @default true
     */
    readonly changelog?: boolean;
    /**
     * Project-level changelog file path.
     *
     * Ignored if `changelog` is false.
     *
     * @default 'CHANGELOG.md'
     */
    readonly changelogPath?: string;
    /**
     * Override git-push command.
     *
     * Set to an empty string to disable pushing.
     */
    readonly gitPushCommand?: string;
}
export interface ContinuousReleaseOptions {
    /**
     * Paths for which pushes should trigger a release
     */
    readonly paths?: string[];
}
/**
 * Used to manage release strategies. This includes release
 * and release artifact automation
 */
export declare class ReleaseTrigger {
    /**
     * Creates a manual release trigger.
     *
     * Use this option if you want totally manual releases.
     *
     * This will give you a release task that, in addition to the normal
     * release activities will trigger a `publish:git` task. This task will
     * handle project-level changelog management, release tagging, and pushing
     * these artifacts to origin.
     *
     * The command used for pushing can be customised by specifying
     * `gitPushCommand`. Set to an empty string to disable pushing entirely.
     *
     * Simply run `yarn release` to trigger a manual release.
     *
     * @param options release options
     */
    static manual(options?: ManualReleaseOptions): ReleaseTrigger;
    /**
     * Creates a scheduled release trigger.
     *
     * Automated releases will occur based on the provided cron schedule.
     *
     * @param options release options.
     */
    static scheduled(options: ScheduledReleaseOptions): ReleaseTrigger;
    /**
     * The release can only be triggered using the GitHub UI.
     */
    static workflowDispatch(): ReleaseTrigger;
    /**
     * Creates a continuous release trigger.
     *
     * Automated releases will occur on every commit.
     */
    static continuous(options?: ContinuousReleaseOptions): ReleaseTrigger;
    /**
     * Project-level changelog file path.
     */
    readonly changelogPath?: string;
    /**
     * Cron schedule for releases.
     *
     * Only defined if this is a scheduled release.
     *
     * @example '0 17 * * *' - every day at 5 pm
     */
    readonly schedule?: string;
    /**
     * Whether or not this is a continuous release.
     */
    readonly isContinuous: boolean;
    /**
     * Paths for which pushes will trigger a release when `isContinuous` is `true`
     */
    readonly paths?: string[];
    /**
     * Override git-push command used when releasing manually.
     *
     * Set to an empty string to disable pushing.
     */
    readonly gitPushCommand?: string;
    private readonly workflowDispatchOnly?;
    private constructor();
    /**
     * Whether or not this is a release trigger with a manual task run in a working copy.
     *
     * If the `ReleaseTrigger` is a GitHub-only manual task, this will return `false`.
     */
    get isManual(): boolean;
}
