"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exec;
exports.execCapture = execCapture;
exports.execOrUndefined = execOrUndefined;
exports.getFilePermissions = getFilePermissions;
exports.writeFile = writeFile;
exports.decamelizeKeysRecursively = decamelizeKeysRecursively;
exports.isTruthy = isTruthy;
exports.isObject = isObject;
exports.deepMerge = deepMerge;
exports.dedupArray = dedupArray;
exports.sorted = sorted;
exports.formatAsPythonModule = formatAsPythonModule;
exports.getGitVersion = getGitVersion;
exports.kebabCaseKeys = kebabCaseKeys;
exports.snakeCaseKeys = snakeCaseKeys;
exports.tryReadFile = tryReadFile;
exports.tryReadFileSync = tryReadFileSync;
exports.isWritable = isWritable;
exports.assertExecutablePermissions = assertExecutablePermissions;
exports.isExecutable = isExecutable;
exports.getNodeMajorVersion = getNodeMajorVersion;
exports.anySelected = anySelected;
exports.multipleSelected = multipleSelected;
exports.isRoot = isRoot;
exports.findUp = findUp;
exports.normalizePersistedPath = normalizePersistedPath;
const child_process = require("child_process");
const fs_1 = require("fs");
const path = require("path");
const Case = require("case");
const logging = require("./logging");
const MAX_BUFFER = 10 * 1024 * 1024;
/**
 * Executes a command with STDOUT > STDERR.
 */
function exec(command, options) {
    logging.debug(`${command} (cwd: ${options.cwd})`);
    child_process.execSync(command, {
        stdio: options.stdio || ["inherit", 2, "pipe"], // "pipe" for STDERR means it appears in exceptions
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
        env: options.env,
    });
}
/**
 * Executes command and returns STDOUT. If the command fails (non-zero), throws an error.
 */
function execCapture(command, options) {
    logging.debug(`${command} (cwd: ${options.cwd})`);
    return child_process.execSync(command, {
        stdio: ["inherit", "pipe", "pipe"], // "pipe" for STDERR means it appears in exceptions
        maxBuffer: MAX_BUFFER,
        cwd: options.cwd,
        env: {
            ...process.env,
            ...options.modEnv,
        },
    });
}
/**
 * Executes `command` and returns its value or undefined if the command failed.
 */
function execOrUndefined(command, options) {
    logging.debug(`${command} (cwd: ${options.cwd})`);
    try {
        const value = child_process
            .execSync(command, {
            stdio: ["inherit", "pipe", "pipe"], // "pipe" for STDERR means it appears in exceptions
            maxBuffer: MAX_BUFFER,
            cwd: options.cwd,
        })
            .toString("utf-8")
            .trim();
        if (!value) {
            return undefined;
        } // an empty string is the same as undefined
        return value;
    }
    catch {
        return undefined;
    }
}
function getFilePermissions(options) {
    const readonly = options.readonly ?? false;
    const executable = options.executable ?? false;
    if (readonly && executable) {
        return "544";
    }
    else if (readonly) {
        return "444";
    }
    else if (executable) {
        return "755";
    }
    else {
        return "644";
    }
}
function writeFile(filePath, data, options = {}) {
    if ((0, fs_1.existsSync)(filePath)) {
        (0, fs_1.chmodSync)(filePath, "600");
    }
    (0, fs_1.mkdirSync)(path.dirname(filePath), { recursive: true });
    (0, fs_1.writeFileSync)(filePath, data);
    (0, fs_1.chmodSync)(filePath, getFilePermissions(options));
}
function decamelizeKeysRecursively(input, opt) {
    const shouldAlwaysDecamelize = () => true;
    const shouldDecamelize = opt?.shouldDecamelize ?? shouldAlwaysDecamelize;
    const separator = opt?.separator ?? "_";
    const path_ = opt?.path ?? [];
    const maxDepth = opt?.maxDepth ?? 10;
    if (path_.length > maxDepth) {
        throw new Error("Decamelled too deeply - check that the input has no circular references");
    }
    if (Array.isArray(input)) {
        return input.map((k, i) => decamelizeKeysRecursively(k, {
            ...opt,
            path: [...path_, i.toString()],
        }));
    }
    if (typeof input === "object" && input !== null) {
        const mappedObject = {};
        for (const [key, value] of Object.entries(input)) {
            const transformedKey = shouldDecamelize([...path_, key], value)
                ? decamelize(key, separator)
                : key;
            mappedObject[transformedKey] = decamelizeKeysRecursively(value, {
                ...opt,
                path: [...path_, key],
            });
        }
        return mappedObject;
    }
    return input;
}
/**
 * Returns false if value is unset or a falsey value, and true otherwise.
 * @param value an environment variable
 */
function isTruthy(value) {
    return !(value === undefined ||
        ["null", "undefined", "0", "false", ""].includes(value.toLocaleLowerCase()));
}
/**
 * Return whether the given value is a plain struct object
 *
 * Even though arrays and instances of classes technically are objects, we
 * usually want to treat them differently, so we return false in those cases.
 */
function isObject(x) {
    return (x !== null &&
        typeof x === "object" &&
        !Array.isArray(x) &&
        x.constructor.name === "Object");
}
/**
 * Recursively merge objects together
 *
 * The leftmost object is mutated and returned.
 *
 * If an object is merged into something other than an object, the non-object is lost.
 * Arrays are overwritten not merged; set `mergeArrays: true` to merge arrays and deduplicate the result.
 * An `undefined` key in a source object will persist; set `destructive: true` to fully remove the key instead.
 * Empty objects as values are preserved in the output; set `destructive: true` to remove them instead.
 */
function deepMerge(objects, { destructive = false, mergeArrays = false } = {}) {
    function mergeOne(target, source) {
        for (const key of Object.keys(source)) {
            const value = source[key];
            // if the current value is a plain object, we recursively merge it
            if (isObject(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = value;
                }
                // Special handling for __$APPEND, which is used to append values to arrays
                if ("__$APPEND" in value && Array.isArray(value.__$APPEND)) {
                    if (Array.isArray(target[key])) {
                        target[key].push(...value.__$APPEND);
                    }
                    else {
                        target[key] = value.__$APPEND;
                    }
                }
                // recursively merge the object
                mergeOne(target[key], value);
                // if the result of the merge is an empty object, it's because the
                // eventual value we assigned is `undefined`, and there are no
                // sibling concrete values alongside, so we can delete this tree.
                if (destructive &&
                    typeof target[key] === "object" &&
                    Object.keys(target[key]).length === 0) {
                    delete target[key];
                }
                continue;
            }
            // in destructive mode, we delete the existing key if the value is undefined
            if (destructive && value === undefined) {
                delete target[key];
                continue;
            }
            // in array merging mode, we merge and deduplicate arrays
            if (mergeArrays && Array.isArray(target[key]) && Array.isArray(value)) {
                target[key] = [...new Set([...target[key], ...value])];
                continue;
            }
            // all other values are simply overwritten by overriding object
            if (typeof value !== "undefined") {
                target[key] = value;
                continue;
            }
        }
    }
    const others = objects.filter((x) => x != null);
    if (others.length === 0) {
        return {};
    }
    const into = others.splice(0, 1)[0];
    others.forEach((other) => mergeOne(into, other));
    return into;
}
/*
 * Deduplicate values in a list, returning a new array.
 * @param array list of values
 */
function dedupArray(array) {
    return array.filter((val, idx) => array.indexOf(val) === idx);
}
/**
 * Returns a sorted version of `x` or `undefined` if it is an empty array or object.
 */
function sorted(x) {
    if (x == null) {
        return undefined;
    }
    if (Array.isArray(x)) {
        if (x.length === 0) {
            return undefined;
        }
        return x.sort();
    }
    else if (typeof x === "object") {
        if (Object.keys(x).length === 0) {
            return undefined;
        }
        const result = {};
        for (const [key, value] of Object.entries(x).sort(([l], [r]) => l.localeCompare(r))) {
            result[key] = value;
        }
        return result;
    }
    else {
        return x;
    }
}
function formatAsPythonModule(name) {
    return name.replace(/-/g, "_").replace(/\./g, "_");
}
/**
 * Extract git version number from command line
 *
 * @param gitVersionOutput the output from `git version` CLI
 * @returns the version of git
 */
function getGitVersion(gitVersionOutput) {
    const match = gitVersionOutput.match(/\d+.\d+.\d+/);
    if (!match) {
        throw new Error("Unable to retrieve git version");
    }
    return match[0];
}
function kebabCaseKeys(obj, recursive = true) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => kebabCaseKeys(v, recursive));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = kebabCaseKeys(v, recursive);
        }
        result[decamelize(k).replace(/_/gm, "-")] = v;
    }
    return result;
}
function snakeCaseKeys(obj, recursive = true, exclusiveForRecordKeys = []) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        if (recursive) {
            obj = obj.map((v) => snakeCaseKeys(v, recursive, exclusiveForRecordKeys));
        }
        return obj;
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (recursive) {
            v = snakeCaseKeys(v, recursive, exclusiveForRecordKeys);
        }
        const modifiedKey = exclusiveForRecordKeys.length == 0 || exclusiveForRecordKeys.includes(k)
            ? Case.snake(k)
            : k;
        result[modifiedKey] = v;
    }
    return result;
}
async function tryReadFile(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return "";
    }
    return fs_1.promises.readFile(file, "utf-8");
}
function tryReadFileSync(file) {
    if (!(0, fs_1.existsSync)(file)) {
        return undefined;
    }
    return (0, fs_1.readFileSync)(file, "utf-8");
}
function isWritable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.W_OK);
        return true;
    }
    catch {
        return false;
    }
}
/**
 * Asserts that the file should be executable. Always returns true on Windows.
 *
 * In Windows, the executable attribute is stored in the system setting PATHEXT, not in each file. Then, checking for executability is equivalent to checking for existence. To bypass checking for executability, we always return true on Windows.
 *
 * @param filePath The path to the file
 * @param shouldBeExecutable Whether the file should be executable
 * @returns true if `filePath` executable attribute matches `shouldBeExecutable` or if the platform is Windows, false otherwise
 */
function assertExecutablePermissions(filePath, shouldBeExecutable) {
    const isWindows = process.platform === "win32";
    if (isWindows) {
        return true;
    }
    const prevExecutable = isExecutable(filePath);
    return prevExecutable === shouldBeExecutable;
}
function isExecutable(file) {
    try {
        (0, fs_1.accessSync)(file, fs_1.constants.X_OK);
        return true;
    }
    catch (e) {
        return false;
    }
}
function decamelize(s, sep = "_") {
    if (Case.of(s) === "camel") {
        return Case.lower(s, sep);
    }
    else {
        return s;
    }
}
function getNodeMajorVersion() {
    const match = process.version.match(/(\d+)\.(\d+)\.(\d+)/);
    if (match) {
        const [major] = match.slice(1).map((x) => parseInt(x));
        return major;
    }
    return undefined;
}
function anySelected(options) {
    return options.some((opt) => opt);
}
function multipleSelected(options) {
    return options.filter((opt) => opt).length > 1;
}
/**
 * Checks if a path is a FS root
 *
 * Optional uses a provided OS specific path implementation,
 * defaults to use the implementation for the current OS.
 *
 * @internal
 */
function isRoot(dir, osPathLib = path) {
    const parent = osPathLib.dirname(dir);
    return parent === dir;
}
/**
 * Run up project tree to find a file or directory
 *
 * @param lookFor the file or directory to look for
 * @param cwd current working directory, must be an absolute path
 * @returns path to the file or directory we are looking for, undefined if not found
 */
function findUp(lookFor, cwd = process.cwd()) {
    if ((0, fs_1.existsSync)(path.join(cwd, lookFor))) {
        return cwd;
    }
    if (isRoot(cwd)) {
        // This is a root
        return undefined;
    }
    return findUp(lookFor, path.dirname(cwd));
}
/**
 * Normalizes a path that is going to be persisted to have a cross platform representation.
 *
 * Normalized paths can be persisted and doesn't need to be modified when the platform changes.
 * `normalizePersistedPath` takes care of platform-specific properties like directory separator.
 * It uses `path.posix.sep` that is supported both in Windows and Unix platforms.
 *
 *
 * @param p the path to be normalized
 * @returns the normalized path
 */
function normalizePersistedPath(p) {
    return p.replace(/\\/g, path.posix.sep);
}
//# sourceMappingURL=data:application/json;base64,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