import os
import base64
import logging
from xero_python.api_client import ApiClient, serialize
from xero_python.api_client.oauth2 import OAuth2Token
from xero_python.api_client.configuration import Configuration
from xero_python.accounting import AccountingApi, Contact, LineItem, Account, BankTransaction, BankTransactions, LineItem, Invoice, Invoices, Contacts, AccountType, Currency, CurrencyCode
from functools import wraps
import jwt
from datetime import *
from xero_python.utils import getvalue
from xero_python.exceptions import AccountingBadRequestException
import dateutil
from dateutil.relativedelta import *
from .model import Booking, RefAndCommissionModel, InvoiceStatusEnum, TransactionStatusEnum
import decimal
import yaml

log = logging.getLogger(__name__)


DIR, _ = os.path.split(os.path.abspath(__file__))
xero_config = {}
with open(os.path.join(DIR, 'xero_config.yml')) as f:
    xero_config = yaml.safe_load(f)


client_id = os.environ.get('XERO_CLIENT_ID', '')
client_secret = os.environ.get('XERO_CLIENT_SECRET', '')
tenant_id = os.environ.get('XERO_TENANT_ID', '')

message_bytes = (client_id + ":" + client_secret).encode("utf-8")
encoded_bytes = base64.b64encode(message_bytes)
encoded_message = encoded_bytes.decode("utf-8")

# scope = "accounting.transactions accounting.transactions.read accounting.contacts accounting.contacts.read"

xero_token = {}

is_debug = False
if os.getenv('DEBUG'):
    is_debug = True

# configure xero-python sdk client
api_client = ApiClient(
    Configuration(
        debug=is_debug,
        oauth2_token=OAuth2Token(
            client_id=client_id, client_secret=client_secret
        ),
    ),
    pool_threads=1,
)



def xero_token_required(function):
    @wraps(function)
    def decorator(*args, **kwargs):
        old_token = get_token()

        return function(*args, **kwargs)

    return decorator

@api_client.oauth2_token_saver
def set_token(token):
    claims = jwt.decode(token["access_token"], options={"verify_signature": False})
    token["expires_at"] = claims["exp"]
    global xero_token
    xero_token = token


@api_client.oauth2_token_getter
def get_token():
    global xero_token
    if xero_token == {} or xero_token['expires_at'] - datetime.now().timestamp() < 60:
        log.info('expired or no token')
        temp_token = api_client.get_client_credentials_token()
        return temp_token
    return xero_token

"""
referral_model
[
    {
        uid: 
        referral_config: {
            platform_commission: float
            reward_layers: list[float]
        }
    },
    ...
]
"""
@xero_token_required
async def create_txns_on_order_confirmed(booking: Booking, provider_model: RefAndCommissionModel, referral_models: list[RefAndCommissionModel]):
    summarize_errors = 'True'
    api_instance = AccountingApi(api_client)

    referrer_uids = list(map(lambda m: m['uid'], referral_models))

    # fetch contact id [provider_xero_id, ...referral_xero_ids]
    contact_ids = get_contactId_from_account_numbers(
        [provider_model['uid']] + referrer_uids)
    
    xero_provider_id = contact_ids[0]
    xero_referal_ids = contact_ids[slice(1, None)]

    platform_commission_gross = booking['total_fee_aud'] * provider_model['referral_config']['platform_commission']
    platform_commission = platform_commission_gross
    p_contribution = booking['total_fee_aud'] - platform_commission_gross

    #TODO: bank account - bdating_wallet
    log.info("wallet === ", str(booking['bdating_wallet']))
    bank_account = Account(
        account_id="a29359e9-beac-4263-8dbe-e75903c4ff35")

    bank_transactions_array = []

    # iterate referrer
    for idx, ref in enumerate(referral_models):
        # first id is provider xero id
        r_contact = Contact(xero_referal_ids[idx])
        reference_contribution = platform_commission_gross * ref['referral_config']['reward_layers'][idx]
        platform_commission -= reference_contribution
        r_line_item = LineItem(
            description=f"bookingId={booking['booking_id']}, total={booking['total_fee_aud']}, referral={referrer_uids[idx]}, contribution={reference_contribution}, tier {idx+1}",
            quantity=1.0,
            unit_amount=reference_contribution,
            # account code
            account_code="601")
        line_items = []
        line_items.append(r_line_item)


        log.info(f"generating RECEIVE txn R{idx+1} reference referral_id={referrer_uids[idx]} for bookingid={booking['booking_id']} ")
        bank_transaction = BankTransaction(
            type="RECEIVE",
            contact=r_contact,
            line_items=line_items,
            reference=f"R{idx+1}, booking_id={booking['booking_id']}",
            bank_account=bank_account)

        bank_transactions_array.append(bank_transaction)
        log.info(f"generated RECEIVE txn R{idx+1} reference referral_id={referrer_uids[idx]} for bookingid={booking['booking_id']} ")

    # provider
    p_contact = Contact(xero_provider_id)
    p_line_item = LineItem(
        description=f"bookingId={booking['booking_id']}, total={booking['total_fee_aud']}, pId={booking['provider_uid']}, pContribution={p_contribution} ",
        quantity=1.0,
        unit_amount=p_contribution,
        account_code="601")
    line_items = []
    line_items.append(p_line_item)
    
    log.info(f"generating RECEIVE txn provider_id={xero_provider_id} for bookingid={booking['booking_id']} ")
    bank_transaction = BankTransaction(
        type="RECEIVE",
        contact=p_contact,
        line_items=line_items,
        reference=f"P, booking_id={booking['booking_id']}",
        bank_account=bank_account)

    log.info(f"generated RECEIVE txn provider_id={xero_provider_id} for bookingid={booking['booking_id']} ")
    bank_transactions_array.append(bank_transaction)

    # revenue
    bdating_contact = Contact("4bb78bac-0de6-49f9-82eb-f84d15a91dec")
    revenue_line_item = LineItem(
        description=f"bookingId={booking['booking_id']}, total={booking['total_fee_aud']}, pId={booking['provider_uid']}, platform_commission={platform_commission} ",
        quantity=1.0,
        unit_amount=platform_commission,
        account_code="603")
    line_items = []
    line_items.append(revenue_line_item)

    log.info(f"generating RECEIVE txn revenue for bookingid={booking['booking_id']} ")
    bank_transaction = BankTransaction(
        type="RECEIVE",
        contact=bdating_contact,
        line_items=line_items,
        reference=f"revenue={booking['booking_id']}",
        bank_account=bank_account)
    log.info(f"generated RECEIVE txn revenue for bookingid={booking['booking_id']} ")


    bank_transactions_array.append(bank_transaction)

    bankTransactions = BankTransactions(
        bank_transactions=bank_transactions_array)

    try:
        api_response = api_instance.create_bank_transactions(
            xero_tenant_id=tenant_id, bank_transactions=bankTransactions, summarize_errors=summarize_errors, unitdp=4)
        return api_response
    except AccountingBadRequestException as e:
        log.error("Accounting: Exception when calling AccountingApi->createBankTransactions: %s\n" % e)
        raise e

@xero_token_required
def create_contact_on_registration(uid: str):
    api_instance = AccountingApi(api_client)
    summarize_errors = 'True'
    contact = Contact(
        name = uid,
        account_number = uid)
    contacts = Contacts( 
        contacts = [contact])
    try:
        api_response = api_instance.create_contacts(tenant_id, contacts, summarize_errors)
        return getvalue(api_response, 'contacts', '')
    except AccountingBadRequestException as e:
        log.error("Exception when calling AccountingApi->createContacts: %s\n" % e)
        raise e

@xero_token_required
def withdraw_request_received(
    account_id: str,
    amount: decimal.Decimal
):
    api_instance = AccountingApi(api_client)

    balance = check_balance(account_id=account_id)
    now = datetime.now()
    delta = relativedelta(weeks=1)
    due_date = now + delta
    if balance - amount >= 0:
        contactId = get_contactId_from_account_numbers(acc_nos=[account_id])[0]

        contact = Contact(
            contact_id=contactId)
        line_item = LineItem(
            description=f"With draw by user {account_id} at amount {amount}, balance remaining {balance-amount}",
            quantity=1.0,
            unit_amount=amount,
            account_code="602")

        line_items = []
        line_items.append(line_item)

        bill = Invoice(
            type="ACCPAY",
            contact=contact,
            date=now,
            due_date=due_date,
            line_items=line_items,
            reference=f"Balance remaining={balance-amount}",
            status=TransactionStatusEnum.authorised)

        invoices = Invoices(
            invoices=[bill])

        try:
            api_response = api_instance.create_invoices(
                xero_tenant_id=tenant_id, invoices=invoices, unitdp=4)
            return api_response
        except AccountingBadRequestException as e:
            log.error("Exception when calling AccountingApi->createInvoices: %s\n" % e)
            raise e
    else:
        raise PermissionError("No enough balance")

def check_balance(account_id: str) -> decimal.Decimal:
    balance = decimal.Decimal(0)
    txns = get_txns(account_id=account_id)
    for txn in txns:
        balance += txn.total
    invoices = get_invoices(account_id=account_id)
    for invoice in invoices:
        if invoice.status != InvoiceStatusEnum.voided:
            balance -= invoice.total
    return balance

"""
return object contain balance & booking txn details & withdraw invoice detail
{
    balance: Decimal,
    txns: [],
    invoices: []
}
"""
def get_balance_with_txns_invoices(account_id: str) -> dict:
    balance = decimal.Decimal(0)
    txns = get_txns(account_id=account_id)
    for txn in txns:
        # 
        # if txn.status != "voided":
        balance += txn.total
    invoices = get_invoices(account_id=account_id)
    for invoice in invoices:
        if invoice.status != InvoiceStatusEnum.voided:
            balance -= invoice.total
    return {
        'balance': balance,
        'txns': txns,
        'invoices': invoices
    }

def get_invoices(account_id: str):
    api_instance = AccountingApi(api_client)
    where = f'(Status=="{InvoiceStatusEnum.authorised}" OR Status=="{InvoiceStatusEnum.paid}" OR Status=="{InvoiceStatusEnum.voided}") AND Contact.Name="{account_id}"'
    order = 'Date Desc'
    # created_by_my_app = 'False'
    # summary_only = 'True'

    try:
        api_response = api_instance.get_invoices(xero_tenant_id=tenant_id,
                                                 where=where,
                                                 page=0,
                                                 order=order,
                                                 unitdp=4, )
        return getvalue(api_response, "invoices", "")
    except AccountingBadRequestException as e:
        log.error("Exception when calling AccountingApi->getInvoices: %s\n" % e)
        return []

def get_txns(account_id: str):
    api_instance = AccountingApi(api_client)
    where = f'Type=="RECEIVE" AND Contact.Name=="{account_id}" AND Status=="AUTHORISED"'
    order = 'Date Desc'
    if_before = dateutil.parser.parse("2020-02-06T12:17:43.202-08:00")
    try:
        api_response = api_instance.get_bank_transactions(
            xero_tenant_id=tenant_id, if_modified_since=if_before, where=where, order=order, unitdp=4)
        txns = getvalue(api_response, 'bank_transactions', '')
        return txns
    except AccountingBadRequestException as e:
        log.error("Exception when calling AccountingApi->getBankTransactions: %s\n" % e)
        return []

def get_contactId_from_account_numbers(acc_nos: list):
    api_instance = AccountingApi(api_client)
    where = 'ContactStatus=="ACTIVE" AND '
    for id, acc in enumerate(acc_nos):
        if id != len(acc_nos)-1:
            where += 'AccountNumber=="' + acc + '" OR '
        else:
            where += 'AccountNumber=="' + acc + '"'

    order = 'Name ASC'
    include_archived = 'False'
    summary_only = 'True'

    try:
        api_response = api_instance.get_contacts(xero_tenant_id=tenant_id, where=where,
                                                 order=order, include_archived=include_archived, summary_only=summary_only)
        resp = getvalue(api_response, 'contacts', '')
        # sort response contact detail in original order
        result = []
        for acc in acc_nos:
            for r in resp:
                if r.account_number == acc:
                    result.append(r.contact_id)
        return result
    except AccountingBadRequestException as e:
        log.error("Exception when calling AccountingApi->getContacts: %s\n" % e)

def get_accountId_by_wallet(wallet: str):
    return;


# create currency code


@xero_token_required
async def createCurrencycode():
    api_instance = AccountingApi(api_client)

    for c in xero_config["currencies"]:
        currency = Currency(
            code=CurrencyCode[c['code']],
            description=c['description'])
        try:
            api_response = await api_instance.create_currency(tenant_id, currency)
        except AccountingBadRequestException as e:
            if e.status == 400:
                print("Currency exist, ignore")
                continue
            print("Exception when calling AccountingApi->createCurrency: %s\n" % e)
            raise e


# create accounting account
@xero_token_required
async def createAccountingAccounts():

    api_instance = AccountingApi(api_client)
    for currency in list(xero_config['accounts']):
        for acc in xero_config['accounts'][currency]:
            account = Account(
                code=acc['code'],
                name=acc['name'],
                type=AccountType[acc['type']],
                description=acc['description'],
                tax_type=acc['tax_type'])
            try:
                api_response = await api_instance.create_account(tenant_id, account)
            except AccountingBadRequestException as e:
                if e.status == 400:
                    print(f"Account {acc['code']} exist, ignore")
                    continue
                print("Exception when calling AccountingApi->createAccount: %s\n" % e)
                raise e

async def xero_health_check():
    try:
        log.error(f"Checking xero connection with client_id: {client_id} & client_secret: {client_secret}...")
        await createCurrencycode()
        await createAccountingAccounts()
    except Exception as ex:
        log.error(f"Fatal exception occurred during checking xero connection. Failed to create app. \n {ex}")
        raise ex