"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
;
/**
 * A CDK construct that creates an API Gateway and Lambda function that can be used to expose
 * a Micro API project for RenovoLive.
 */
class MicroApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const environment = props.environment ?? undefined;
        const stageName = props.stageName ?? 'dev';
        const deadLetterQueue = new aws_cdk_lib_1.aws_sqs.Queue(this, 'DeadLetterQueue');
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'ExecutionRole', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'),
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'),
            ],
        });
        const handlerSg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        const handler = new aws_cdk_lib_1.aws_lambda.Function(this, `handler${props.apiName}`, {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.DOTNET_6,
            code: props.code,
            handler: props.handler,
            description: `${props.apiName} Micro API`,
            environment,
            vpc: props.vpc,
            timeout: props.timeout ?? aws_cdk_lib_1.Duration.seconds(30),
            memorySize: props.memorySize ?? 128,
            deadLetterQueue,
            ephemeralStorageSize: props.ephemeralStorageSize ?? aws_cdk_lib_1.Size.mebibytes(512),
            insightsVersion: aws_cdk_lib_1.aws_lambda.LambdaInsightsVersion.VERSION_1_0_119_0,
            onFailure: new aws_cdk_lib_1.aws_lambda_destinations.SqsDestination(deadLetterQueue),
            role,
            securityGroups: [
                handlerSg,
            ],
        });
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, `LogGroup${props.apiName}`);
        const authorizationType = props.authorizationType ?? aws_cdk_lib_1.aws_apigateway.AuthorizationType.IAM;
        const api = new aws_cdk_lib_1.aws_apigateway.LambdaRestApi(this, `api${props.apiName}`, {
            handler,
            proxy: true,
            defaultCorsPreflightOptions: {
                allowMethods: aws_cdk_lib_1.aws_apigateway.Cors.ALL_METHODS,
                allowOrigins: props.allowedOrigins,
            },
            endpointConfiguration: {
                types: [aws_cdk_lib_1.aws_apigateway.EndpointType.PRIVATE],
                vpcEndpoints: [props.vpcEndpoint],
            },
            deployOptions: {
                accessLogDestination: new aws_cdk_lib_1.aws_apigateway.LogGroupLogDestination(logGroup),
                accessLogFormat: aws_cdk_lib_1.aws_apigateway.AccessLogFormat.jsonWithStandardFields({
                    caller: true,
                    httpMethod: true,
                    ip: true,
                    protocol: true,
                    requestTime: true,
                    resourcePath: true,
                    responseLength: true,
                    status: true,
                    user: true,
                }),
                stageName,
            },
            defaultMethodOptions: {
                authorizationType,
            },
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-resource-policies-examples.html#apigateway-resource-policies-source-vpc-example
            policy: new aws_cdk_lib_1.aws_iam.PolicyDocument({
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        effect: aws_cdk_lib_1.aws_iam.Effect.ALLOW,
                        principals: [new aws_cdk_lib_1.aws_iam.AnyPrincipal()],
                        actions: [
                            'execute-api:Invoke',
                        ],
                        resources: [
                            'execute-api:/*',
                        ],
                    }),
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        effect: aws_cdk_lib_1.aws_iam.Effect.DENY,
                        principals: [new aws_cdk_lib_1.aws_iam.AnyPrincipal()],
                        actions: [
                            'execute-api:Invoke',
                        ],
                        resources: [
                            'execute-api:/*',
                        ],
                        conditions: {
                            StringNotEquals: {
                                'aws:SourceVpce': props.vpcEndpoint.vpcEndpointId,
                            },
                        },
                    }),
                ],
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, 'PrivateExecuteUrl', {
            value: `https://${api.restApiId}-${props.vpcEndpoint.vpcEndpointId}.execute-api.${aws_cdk_lib_1.Stack.of(this).region}.${aws_cdk_lib_1.Stack.of(this).urlSuffix}/${api.deploymentStage.stageName}/`,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'LambdaName', {
            value: handler.functionName,
        });
    }
}
exports.MicroApi = MicroApi;
_a = JSII_RTTI_SYMBOL_1;
MicroApi[_a] = { fqn: "@renovosolutions/cdk-library-renovo-microapi.MicroApi", version: "0.0.64" };
//# sourceMappingURL=data:application/json;base64,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