# ruff: noqa
"""add_new_git_entities

Revision ID: 7f15f878c3a1
Revises: b9cd1c3fd762
Create Date: 2025-09-18 11:46:54.238720

"""

from typing import Sequence, Union

from alembic import op
import sqlalchemy as sa
from kodit.infrastructure.sqlalchemy.entities import PathType


# revision identifiers, used by Alembic.
revision: str = "7f15f878c3a1"
down_revision: Union[str, None] = "b9cd1c3fd762"
branch_labels: Union[str, Sequence[str], None] = None
depends_on: Union[str, Sequence[str], None] = None


def upgrade() -> None:
    """Upgrade schema."""
    # ### commands auto generated by Alembic - please adjust! ###

    # Drop dependent tables first to respect foreign key constraints in PostgreSQL
    # Drop embeddings first (it references snippets)
    op.drop_index(op.f("ix_embeddings_type"), table_name="embeddings")
    op.drop_index(op.f("ix_embeddings_snippet_id"), table_name="embeddings")
    op.drop_table("embeddings")

    # Drop author_file_mappings (references both authors and files)
    op.drop_index(
        "ix_author_file_mappings_author_id", table_name="author_file_mappings"
    )
    op.drop_index("ix_author_file_mappings_file_id", table_name="author_file_mappings")
    op.drop_table("author_file_mappings")

    # Drop snippets (references files and indexes)
    op.drop_index("ix_snippets_file_id", table_name="snippets")
    op.drop_index("ix_snippets_index_id", table_name="snippets")
    op.drop_table("snippets")

    # Drop indexes (references sources)
    op.drop_index("ix_indexes_source_id", table_name="indexes")
    op.drop_table("indexes")

    # Drop files (references sources)
    op.drop_index("ix_files_cloned_path", table_name="files")
    op.drop_index("ix_files_extension", table_name="files")
    op.drop_index("ix_files_mime_type", table_name="files")
    op.drop_index("ix_files_sha256", table_name="files")
    op.drop_index("ix_files_uri", table_name="files")
    op.drop_table("files")

    # Drop authors (no longer has dependencies)
    op.drop_index("ix_authors_email", table_name="authors")
    op.drop_index("ix_authors_name", table_name="authors")
    op.drop_table("authors")

    # Finally drop sources (base table)
    op.drop_index("ix_sources_cloned_path", table_name="sources")
    op.drop_index("ix_sources_type", table_name="sources")
    op.drop_index("ix_sources_uri", table_name="sources")
    op.drop_table("sources")

    # Drop the tasks table and recreate with a string type column
    op.drop_index(op.f("ix_tasks_type"), table_name="tasks")
    op.drop_index(op.f("ix_tasks_dedup_key"), table_name="tasks")
    op.drop_table("tasks")
    op.create_table(
        "tasks",
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("dedup_key", sa.String(length=255), nullable=False),
        sa.Column("type", sa.String(length=255), nullable=False),
        sa.Column("payload", sa.JSON(), nullable=False),
        sa.Column("priority", sa.Integer(), nullable=False),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index(op.f("ix_tasks_dedup_key"), "tasks", ["dedup_key"], unique=False)
    op.create_index(op.f("ix_tasks_type"), "tasks", ["type"], unique=False)

    # Create new tables in correct dependency order
    op.create_table(
        "commit_indexes",
        sa.Column("commit_sha", sa.String(length=64), nullable=False),
        sa.Column(
            "status",
            sa.Enum(
                "PENDING", "IN_PROGRESS", "COMPLETED", "FAILED", name="indexstatustype"
            ),
            nullable=False,
        ),
        sa.Column(
            "indexed_at",
            sa.DateTime(timezone=True),
            nullable=True,
        ),
        sa.Column("error_message", sa.UnicodeText(), nullable=True),
        sa.Column("files_processed", sa.Integer(), nullable=False),
        sa.Column("processing_time_seconds", sa.String(length=50), nullable=False),
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index(
        op.f("ix_commit_indexes_commit_sha"),
        "commit_indexes",
        ["commit_sha"],
        unique=True,
    )
    op.create_index(
        op.f("ix_commit_indexes_status"), "commit_indexes", ["status"], unique=False
    )

    op.create_table(
        "git_repos",
        sa.Column("sanitized_remote_uri", sa.String(length=1024), nullable=False),
        sa.Column("remote_uri", sa.String(length=1024), nullable=False),
        sa.Column(
            "cloned_path",
            PathType(length=1024),
            nullable=True,
        ),
        sa.Column(
            "last_scanned_at",
            sa.DateTime(timezone=True),
            nullable=True,
        ),
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index(
        op.f("ix_git_repos_sanitized_remote_uri"),
        "git_repos",
        ["sanitized_remote_uri"],
        unique=True,
    )

    op.create_table(
        "snippets_v2",
        sa.Column("sha", sa.String(length=64), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("content", sa.UnicodeText(), nullable=False),
        sa.Column("extension", sa.String(length=255), nullable=False),
        sa.PrimaryKeyConstraint("sha"),
    )
    op.create_index(
        op.f("ix_snippets_v2_extension"), "snippets_v2", ["extension"], unique=False
    )

    op.create_table(
        "enrichments",
        sa.Column("snippet_sha", sa.String(length=64), nullable=False),
        sa.Column(
            "type",
            sa.Enum("UNKNOWN", "SUMMARIZATION", name="enrichmenttype"),
            nullable=False,
        ),
        sa.Column("content", sa.UnicodeText(), nullable=False),
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.ForeignKeyConstraint(
            ["snippet_sha"],
            ["snippets_v2.sha"],
        ),
        sa.PrimaryKeyConstraint("id"),
        sa.UniqueConstraint("snippet_sha", "type", name="uix_snippet_enrichment"),
    )
    op.create_index(
        op.f("ix_enrichments_snippet_sha"), "enrichments", ["snippet_sha"], unique=False
    )
    op.create_index(op.f("ix_enrichments_type"), "enrichments", ["type"], unique=False)

    op.create_table(
        "git_commits",
        sa.Column("commit_sha", sa.String(length=64), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("repo_id", sa.Integer(), nullable=False),
        sa.Column(
            "date",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("message", sa.UnicodeText(), nullable=False),
        sa.Column("parent_commit_sha", sa.String(length=64), nullable=True),
        sa.Column("author", sa.String(length=255), nullable=False),
        sa.ForeignKeyConstraint(
            ["repo_id"],
            ["git_repos.id"],
        ),
        sa.PrimaryKeyConstraint("commit_sha"),
    )
    op.create_index(
        op.f("ix_git_commits_author"), "git_commits", ["author"], unique=False
    )
    op.create_index(
        op.f("ix_git_commits_repo_id"), "git_commits", ["repo_id"], unique=False
    )

    op.create_table(
        "commit_snippets_v2",
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column("commit_sha", sa.String(length=64), nullable=False),
        sa.Column("snippet_sha", sa.String(length=64), nullable=False),
        sa.ForeignKeyConstraint(
            ["commit_sha"],
            ["git_commits.commit_sha"],
        ),
        sa.ForeignKeyConstraint(
            ["snippet_sha"],
            ["snippets_v2.sha"],
        ),
        sa.PrimaryKeyConstraint("id"),
        sa.UniqueConstraint("commit_sha", "snippet_sha", name="uix_commit_snippet"),
    )
    op.create_index(
        op.f("ix_commit_snippets_v2_commit_sha"),
        "commit_snippets_v2",
        ["commit_sha"],
        unique=False,
    )
    op.create_index(
        op.f("ix_commit_snippets_v2_snippet_sha"),
        "commit_snippets_v2",
        ["snippet_sha"],
        unique=False,
    )

    op.create_table(
        "git_branches",
        sa.Column("repo_id", sa.Integer(), nullable=False),
        sa.Column("name", sa.String(length=255), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("head_commit_sha", sa.String(length=64), nullable=False),
        sa.ForeignKeyConstraint(
            ["head_commit_sha"],
            ["git_commits.commit_sha"],
        ),
        sa.ForeignKeyConstraint(
            ["repo_id"],
            ["git_repos.id"],
        ),
        sa.PrimaryKeyConstraint("repo_id", "name"),
        sa.UniqueConstraint("repo_id", "name", name="uix_repo_branch"),
    )
    op.create_index(
        op.f("ix_git_branches_name"), "git_branches", ["name"], unique=False
    )
    op.create_index(
        op.f("ix_git_branches_repo_id"), "git_branches", ["repo_id"], unique=False
    )

    op.create_table(
        "git_tracking_branches",
        sa.Column("repo_id", sa.Integer(), nullable=False),
        sa.Column("name", sa.String(length=255), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.ForeignKeyConstraint(
            ["repo_id", "name"],
            ["git_branches.repo_id", "git_branches.name"],
        ),
        sa.PrimaryKeyConstraint("repo_id"),
        sa.UniqueConstraint("repo_id", "name", name="uix_repo_tracking_branch"),
    )

    op.create_index(
        op.f("ix_git_tracking_branches_name"),
        "git_tracking_branches",
        ["name"],
        unique=False,
    )
    op.create_index(
        op.f("ix_git_tracking_branches_repo_id"),
        "git_tracking_branches",
        ["repo_id"],
        unique=False,
    )

    op.create_table(
        "git_commit_files",
        sa.Column("commit_sha", sa.String(length=64), nullable=False),
        sa.Column("path", sa.String(length=1024), nullable=False),
        sa.Column("blob_sha", sa.String(length=64), nullable=False),
        sa.Column("mime_type", sa.String(length=255), nullable=False),
        sa.Column("extension", sa.String(length=255), nullable=False),
        sa.Column("size", sa.Integer(), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.ForeignKeyConstraint(
            ["commit_sha"],
            ["git_commits.commit_sha"],
        ),
        sa.PrimaryKeyConstraint("commit_sha", "path"),
        sa.UniqueConstraint("commit_sha", "path", name="uix_commit_file"),
    )

    op.create_index(
        op.f("ix_git_commit_files_blob_sha"),
        "git_commit_files",
        ["blob_sha"],
        unique=False,
    )
    op.create_index(
        op.f("ix_git_commit_files_extension"),
        "git_commit_files",
        ["extension"],
        unique=False,
    )
    op.create_index(
        op.f("ix_git_commit_files_mime_type"),
        "git_commit_files",
        ["mime_type"],
        unique=False,
    )

    op.create_table(
        "git_tags",
        sa.Column("repo_id", sa.Integer(), nullable=False),
        sa.Column("name", sa.String(length=255), nullable=False),
        sa.Column(
            "created_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column(
            "updated_at",
            sa.DateTime(timezone=True),
            nullable=False,
        ),
        sa.Column("target_commit_sha", sa.String(length=64), nullable=False),
        sa.ForeignKeyConstraint(
            ["repo_id"],
            ["git_repos.id"],
        ),
        sa.ForeignKeyConstraint(
            ["target_commit_sha"],
            ["git_commits.commit_sha"],
        ),
        sa.PrimaryKeyConstraint("repo_id", "name"),
        sa.UniqueConstraint("repo_id", "name", name="uix_repo_tag"),
    )
    op.create_index(op.f("ix_git_tags_name"), "git_tags", ["name"], unique=False)
    op.create_index(op.f("ix_git_tags_repo_id"), "git_tags", ["repo_id"], unique=False)
    op.create_index(
        op.f("ix_git_tags_target_commit_sha"),
        "git_tags",
        ["target_commit_sha"],
        unique=False,
    )

    op.create_table(
        "snippet_v2_files",
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column("snippet_sha", sa.String(length=64), nullable=False),
        sa.Column("blob_sha", sa.String(length=64), nullable=False),
        sa.Column("commit_sha", sa.String(length=64), nullable=False),
        sa.Column("file_path", sa.String(length=1024), nullable=False),
        sa.ForeignKeyConstraint(
            ["commit_sha", "file_path"],
            ["git_commit_files.commit_sha", "git_commit_files.path"],
        ),
        sa.ForeignKeyConstraint(
            ["snippet_sha"],
            ["snippets_v2.sha"],
        ),
        sa.PrimaryKeyConstraint("id"),
        sa.UniqueConstraint(
            "snippet_sha",
            "blob_sha",
            "commit_sha",
            "file_path",
            name="uix_snippet_file",
        ),
    )
    op.create_index(
        op.f("ix_snippet_v2_files_blob_sha"),
        "snippet_v2_files",
        ["blob_sha"],
        unique=False,
    )
    op.create_index(
        op.f("ix_snippet_v2_files_commit_sha"),
        "snippet_v2_files",
        ["commit_sha"],
        unique=False,
    )
    op.create_index(
        op.f("ix_snippet_v2_files_file_path"),
        "snippet_v2_files",
        ["file_path"],
        unique=False,
    )
    op.create_index(
        op.f("ix_snippet_v2_files_snippet_sha"),
        "snippet_v2_files",
        ["snippet_sha"],
        unique=False,
    )

    # Create embeddings table with String snippet_id for snippets_v2 compatibility
    op.create_table(
        "embeddings",
        sa.Column("snippet_id", sa.String(length=64), nullable=False),
        sa.Column(
            "type", sa.Enum("CODE", "TEXT", name="embeddingtype"), nullable=False
        ),
        sa.Column("embedding", sa.JSON(), nullable=False),
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column("created_at", sa.DateTime(timezone=True), nullable=False),
        sa.Column("updated_at", sa.DateTime(timezone=True), nullable=False),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index(
        op.f("ix_embeddings_snippet_id"), "embeddings", ["snippet_id"], unique=False
    )
    op.create_index(op.f("ix_embeddings_type"), "embeddings", ["type"], unique=False)

    # Note: Skipping tasks table type column update due to SQLite limitations
    # VARCHAR length changes are often not enforced in SQLite anyway
    # ### end Alembic commands ###


def downgrade() -> None:
    """Downgrade schema."""
    # ### commands auto generated by Alembic - please adjust! ###
    # Note: Skipping tasks table type column downgrade due to SQLite limitations
    # Recreate embeddings table for downgrade (SQLite limitations)
    op.drop_index(op.f("ix_embeddings_type"), table_name="embeddings")
    op.drop_index(op.f("ix_embeddings_snippet_id"), table_name="embeddings")
    op.drop_table("embeddings")

    op.create_table(
        "embeddings",
        sa.Column("snippet_id", sa.INTEGER(), nullable=False),
        sa.Column(
            "type", sa.Enum("CODE", "TEXT", name="embeddingtype"), nullable=False
        ),
        sa.Column("embedding", sa.JSON(), nullable=False),
        sa.Column("id", sa.Integer(), autoincrement=True, nullable=False),
        sa.Column("created_at", sa.DateTime(), nullable=False),
        sa.Column("updated_at", sa.DateTime(), nullable=False),
        sa.ForeignKeyConstraint(
            ["snippet_id"],
            ["snippets.id"],
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index(
        op.f("ix_embeddings_snippet_id"), "embeddings", ["snippet_id"], unique=False
    )
    op.create_index(op.f("ix_embeddings_type"), "embeddings", ["type"], unique=False)
    op.create_table(
        "indexes",
        sa.Column("source_id", sa.INTEGER(), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.ForeignKeyConstraint(
            ["source_id"],
            ["sources.id"],
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index("ix_indexes_source_id", "indexes", ["source_id"], unique=True)
    op.create_table(
        "files",
        sa.Column("source_id", sa.INTEGER(), nullable=False),
        sa.Column("mime_type", sa.VARCHAR(length=255), nullable=False),
        sa.Column("uri", sa.VARCHAR(length=1024), nullable=False),
        sa.Column("cloned_path", sa.VARCHAR(length=1024), nullable=False),
        sa.Column("sha256", sa.VARCHAR(length=64), nullable=False),
        sa.Column("size_bytes", sa.INTEGER(), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.Column("extension", sa.VARCHAR(length=255), nullable=False),
        sa.Column("file_processing_status", sa.INTEGER(), nullable=False),
        sa.ForeignKeyConstraint(
            ["source_id"],
            ["sources.id"],
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index("ix_files_uri", "files", ["uri"], unique=False)
    op.create_index("ix_files_sha256", "files", ["sha256"], unique=False)
    op.create_index("ix_files_mime_type", "files", ["mime_type"], unique=False)
    op.create_index("ix_files_extension", "files", ["extension"], unique=False)
    op.create_index("ix_files_cloned_path", "files", ["cloned_path"], unique=False)
    op.create_table(
        "snippets",
        sa.Column("file_id", sa.INTEGER(), nullable=False),
        sa.Column("index_id", sa.INTEGER(), nullable=False),
        sa.Column("content", sa.TEXT(), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.Column("summary", sa.TEXT(), nullable=False),
        sa.ForeignKeyConstraint(
            ["file_id"],
            ["files.id"],
        ),
        sa.ForeignKeyConstraint(
            ["index_id"],
            ["indexes.id"],
        ),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index("ix_snippets_index_id", "snippets", ["index_id"], unique=False)
    op.create_index("ix_snippets_file_id", "snippets", ["file_id"], unique=False)
    op.create_table(
        "sources",
        sa.Column("uri", sa.VARCHAR(length=1024), nullable=False),
        sa.Column("cloned_path", sa.VARCHAR(length=1024), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.Column("type", sa.VARCHAR(length=7), nullable=False),
        sa.PrimaryKeyConstraint("id"),
    )
    op.create_index("ix_sources_uri", "sources", ["uri"], unique=True)
    op.create_index("ix_sources_type", "sources", ["type"], unique=False)
    op.create_index("ix_sources_cloned_path", "sources", ["cloned_path"], unique=False)
    op.create_table(
        "authors",
        sa.Column("name", sa.VARCHAR(length=255), nullable=False),
        sa.Column("email", sa.VARCHAR(length=255), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.PrimaryKeyConstraint("id"),
        sa.UniqueConstraint("name", "email", name="uix_author"),
    )
    op.create_index("ix_authors_name", "authors", ["name"], unique=False)
    op.create_index("ix_authors_email", "authors", ["email"], unique=False)
    op.create_table(
        "author_file_mappings",
        sa.Column("author_id", sa.INTEGER(), nullable=False),
        sa.Column("file_id", sa.INTEGER(), nullable=False),
        sa.Column("id", sa.INTEGER(), nullable=False),
        sa.Column("created_at", sa.DATETIME(), nullable=False),
        sa.Column("updated_at", sa.DATETIME(), nullable=False),
        sa.ForeignKeyConstraint(
            ["author_id"],
            ["authors.id"],
        ),
        sa.ForeignKeyConstraint(
            ["file_id"],
            ["files.id"],
        ),
        sa.PrimaryKeyConstraint("id"),
        sa.UniqueConstraint("author_id", "file_id", name="uix_author_file_mapping"),
    )
    op.create_index(
        "ix_author_file_mappings_file_id",
        "author_file_mappings",
        ["file_id"],
        unique=False,
    )
    op.create_index(
        "ix_author_file_mappings_author_id",
        "author_file_mappings",
        ["author_id"],
        unique=False,
    )
    op.drop_index(
        op.f("ix_snippet_v2_files_snippet_sha"), table_name="snippet_v2_files"
    )
    op.drop_index(op.f("ix_snippet_v2_files_file_path"), table_name="snippet_v2_files")
    op.drop_index(op.f("ix_snippet_v2_files_commit_sha"), table_name="snippet_v2_files")
    op.drop_index(op.f("ix_snippet_v2_files_blob_sha"), table_name="snippet_v2_files")
    op.drop_table("snippet_v2_files")
    op.drop_index(op.f("ix_git_tags_target_commit_sha"), table_name="git_tags")
    op.drop_index(op.f("ix_git_tags_repo_id"), table_name="git_tags")
    op.drop_index(op.f("ix_git_tags_name"), table_name="git_tags")
    op.drop_table("git_tags")
    op.drop_index(op.f("ix_git_commit_files_mime_type"), table_name="git_commit_files")
    op.drop_index(op.f("ix_git_commit_files_extension"), table_name="git_commit_files")
    op.drop_index(op.f("ix_git_commit_files_blob_sha"), table_name="git_commit_files")
    op.drop_table("git_commit_files")
    op.drop_index(op.f("ix_git_branches_repo_id"), table_name="git_branches")
    op.drop_index(op.f("ix_git_branches_name"), table_name="git_branches")
    op.drop_table("git_branches")
    op.drop_index(
        op.f("ix_commit_snippets_v2_snippet_sha"), table_name="commit_snippets_v2"
    )
    op.drop_index(
        op.f("ix_commit_snippets_v2_commit_sha"), table_name="commit_snippets_v2"
    )
    op.drop_table("commit_snippets_v2")
    op.drop_index(
        op.f("ix_git_tracking_branches_repo_id"), table_name="git_tracking_branches"
    )
    op.drop_index(
        op.f("ix_git_tracking_branches_name"), table_name="git_tracking_branches"
    )
    op.drop_table("git_tracking_branches")
    op.drop_index(op.f("ix_git_commits_repo_id"), table_name="git_commits")
    op.drop_index(op.f("ix_git_commits_author"), table_name="git_commits")
    op.drop_table("git_commits")
    op.drop_index(op.f("ix_enrichments_type"), table_name="enrichments")
    op.drop_index(op.f("ix_enrichments_snippet_sha"), table_name="enrichments")
    op.drop_table("enrichments")
    op.drop_index(op.f("ix_snippets_v2_extension"), table_name="snippets_v2")
    op.drop_table("snippets_v2")
    op.drop_index(op.f("ix_git_repos_sanitized_remote_uri"), table_name="git_repos")
    op.drop_table("git_repos")
    op.drop_index(op.f("ix_commit_indexes_status"), table_name="commit_indexes")
    op.drop_index(op.f("ix_commit_indexes_commit_sha"), table_name="commit_indexes")
    op.drop_table("commit_indexes")
    # ### end Alembic commands ###
