"""Class definition for threaded dequeuing of expanding matrices.

Usage:
        controller = prediction_saver()
        # Main loop adding data
        for _ in np.range(10):
                try:
                        controller.results_receiver_queue.put((1, new_data), timeout=5)
                except queue.Full:
                        if not controller.is_healthy():
                                print('Writer thread died unexpectedly.', file=sys.stderr)
                                sys.exit(1)
                        continue
        # Done with main loop, get data
        controller.results_receiver_queue.put((None, None))
        results_matrix = controller.get_results()
"""

import multiprocessing as mp

import numpy as np


class prediction_saver:
    """Threaded receiver of prediction data."""

    def __init__(
        self,
        resize_increment: int = 10000,
        dtype: np.dtype = np.float32,
        pad_value: float = 0,
    ):
        """Initializes a table storage mechanism for prediction data generated by batches.

        Args:
                resize_increment: increment to resize matrices along the first dimension. For data that grows in multiple dimensions, all higher dimensions only increase by the observed increases
                dtype: data type stored
                pad_value: value used when data is not present
        """
        self.results_receiver_queue = mp.Queue(5)
        self.__results_storage_thread = None
        self.results_queue = mp.JoinableQueue(1)
        self.__prediction_matrix = None
        self.__resize_increment = resize_increment
        self.__dtype = dtype
        self.__pad_value = dtype(pad_value)
        self.start_dequeue_results()

    def is_healthy(self):
        """Checks the health of queues and exits if needed.

        Returns:
                True if threads have not crashed. Closes all threads and returns False when something went wrong.
        """
        is_healthy = True
        if self.__results_storage_thread is not None:
            if (
                self.__results_storage_thread.exitcode is None
                or self.__results_storage_thread.exitcode == 0
            ):
                pass
            else:
                is_healthy = False
        # If something bad was detected, close down all threads so main code can exit.
        # Note: This will dangerously terminate all multiprocessing threads.
        if not is_healthy:
            for thread in mp.active_children():
                thread.terminate()
                thread.join()
        return is_healthy

    def __resize_prediction_mat(self, cur_preds, new_shape):
        """Resizes the internal prediction matrix.

        Args:
                cur_preds: current prediction matrix to be resizes
                new_shape: new shape of the prediction matrix
        """
        new_preds = cur_preds
        cur_mat_size = np.asarray(cur_preds.shape)
        for dim in np.arange(len(cur_mat_size)):
            change = new_shape[dim] - cur_mat_size[dim]
            # Unchanged dimensions
            if change <= 0:
                continue
            new_size = cur_mat_size
            new_size[dim] = change
            expansion = np.full(new_size, self.__pad_value, dtype=self.__dtype)
            new_preds = np.concatenate((new_preds, expansion), axis=dim)
            cur_mat_size = np.asarray(new_preds.shape)
        return new_preds

    def dequeue_thread(self, results_queue, output_queue):
        """Dequeues predictions into the prediction matrix.

        Args:
                results_queue: queue that this thread watches to receive data
                output_queue: queue that this thread places the final results

        Notes:
                Data sent should be a tuple of (num_predictions, prediction_data)
                        num_predictions: integer indicating the number of predictions contained within the first dimension of the data
                        prediction_data: np.ndarray of shape [batch, ...]. Number of dimensions must remain the same, but can change in length (e.g. axis can be [batch, n_animals_predicted, keypoint, 2] and n_animals_predicted can vary between batches).

                Sending a None value into the results queue indicates the last prediction was made and the output queue should be finalized.
        """
        prediction_matrix = None
        cur_mat_size = None
        cur_frames_used_count = None
        available_new_frames = None
        while True:
            prediction_count, predictions = results_queue.get()
            # Exit if None was passed
            if prediction_count is None:
                break
            # This is the first prediction, we need to initialize the matrix
            if prediction_matrix is None:
                prediction_matrix = predictions
                cur_mat_size = np.array(predictions.shape)
                cur_frames_used_count = prediction_count
                available_new_frames = cur_mat_size[0] - cur_frames_used_count
            else:
                # Resize storage if necessary
                next_mat_size = cur_mat_size.copy()
                # Add more frames if not enough to assign results
                if available_new_frames < prediction_count:
                    available_new_frames += self.__resize_increment
                    next_mat_size[0] += self.__resize_increment
                # If more space is needed in higher dims, add them
                next_mat_size[1:] = np.max(
                    [cur_mat_size[1:], predictions.shape[1:]], axis=0
                )
                if np.any(next_mat_size != cur_mat_size):
                    prediction_matrix = self.__resize_prediction_mat(
                        prediction_matrix, next_mat_size
                    )
                # Pad predictions for lazy slicing
                adjusted_prediction_shape = next_mat_size.copy()
                adjusted_prediction_shape[0] = prediction_count
                resized_predictions = self.__resize_prediction_mat(
                    predictions[:prediction_count], adjusted_prediction_shape
                )
                # Copy in new data
                prediction_matrix[
                    cur_frames_used_count : cur_frames_used_count + prediction_count, :
                ] = resized_predictions
                cur_frames_used_count += prediction_count
                available_new_frames -= prediction_count
                cur_mat_size = next_mat_size
        # Clip out unused info from the matrices
        if prediction_matrix is not None:
            prediction_matrix = prediction_matrix[:cur_frames_used_count]
        # Close down the dequeue thread
        output_queue.put(prediction_matrix)

    def start_dequeue_results(self):
        """Starts a thread that dequeues results."""
        if self.__results_storage_thread is None:
            self.__results_storage_thread = mp.Process(
                target=self.dequeue_thread,
                args=(
                    self.results_receiver_queue,
                    self.results_queue,
                ),
                daemon=True,
            )
            self.__results_storage_thread.start()

    def get_results(self):
        """Block pulling out results until results queue is complete."""
        if self.__results_storage_thread is not None:
            self.__prediction_matrix = self.results_queue.get()
            self.__results_storage_thread.join()
            self.__results_storage_thread = None
        return self.__prediction_matrix
