"""Default values used by the configuration classes."""

from typing import Final

from ropt.enums import BoundaryType, PerturbationType

DEFAULT_SEED: Final = 1
"""Default seed for random number generators.

The seed is used as the base value for random number generators within various
components of the optimization process, such as samplers. Using a consistent
seed ensures reproducibility across multiple runs with the same configuration.
To obtain unique results for each optimization run, modify this seed.
"""

DEFAULT_NUMBER_OF_PERTURBATIONS: Final = 5
"""Default number of perturbations for gradient estimation.

This value defines the default number of perturbed variables used to estimate
gradients. A higher number of perturbations can lead to more accurate gradient
estimates but also increases the number of function evaluations required.
"""

DEFAULT_PERTURBATION_MAGNITUDE: Final = 0.005
"""Default magnitude for variable perturbations.

This value specifies the default value of the scaling factor applied to the
perturbation values generated by samplers. The magnitude can be interpreted as
an absolute value or as a relative value, depending on the selected perturbation
type.

See also: [`PerturbationType`][ropt.enums.PerturbationType].
"""

DEFAULT_PERTURBATION_BOUNDARY_TYPE: Final = BoundaryType.MIRROR_BOTH
"""Default perturbation boundary handling type.

This value determines how perturbations that violate the defined variable bounds
are handled. The default, `BoundaryType.MIRROR_BOTH`, mirrors perturbations back
into the valid range if they exceed either the lower or upper bound.

See also: [`BoundaryType`][ropt.enums.BoundaryType].
"""

DEFAULT_PERTURBATION_TYPE: Final = PerturbationType.ABSOLUTE
"""Default perturbation type.

This value determines how the perturbation magnitude is interpreted. The
default, `PerturbationType.ABSOLUTE`, means that the perturbation magnitude is
added directly to the variable value. Other options, such as
`PerturbationType.RELATIVE`, scale the perturbation magnitude based on the
variable's bounds.

See also: [`PerturbationType`][ropt.enums.PerturbationType].
"""
