"""
The Model-Context Protocol (MCP) module for Pentest-Tools.com. It uses the ptt API to provide tools, resources and prompts
to LLM agents that implement MCP.
"""

try:
    from mcp.server.fastmcp import FastMCP
except ImportError:
    # We use this as a guard, in case `pentesttools` was not installed with mcp support
    FastMCP = None


import json
import os
import sys
import time
import logging

import pentesttools as ptt

if not getattr(ptt.api, "api_key", False):
    api_key = os.getenv("PTT_KEY", "")
    if not api_key:
        sys.exit(
            "API key not found. Please provide it using the --key flag or set the PTT_KEY environment variable."
        )

    ptt.api.api_key = api_key


os.environ["PTT_USER_AGENT"] = "ptt-cli-mcp"

API_URL = "https://app.pentest-tools.com/api/v2/"
ptt.api.init()

HEADERS = {
    "User-Agent": "ptt-cli-mcp",
    "Accept": "*/*",
}
HEADERS["Authorization"] = f"Bearer {ptt.api.api_key}"


def run_tool(
    tool_id: int, target: str, tool_params: dict, scan_params: dict, raw: bool = False
):
    res = ptt.api.start_scan(target, tool_id, tool_params, scan_params, headers=HEADERS)
    res_json = res.json()

    scan_id = res_json["data"]["created_id"]
    logging.info("Started scan %i, for tool_id %s", scan_id, tool_id)

    while True:
        time.sleep(2)

        # Get the status of our scan
        status = ptt.api.get_scan_status(scan_id, headers=HEADERS)
        status_name = status.json()["data"]["status_name"]

        if status_name == "finished":
            logging.debug("Scan status: %s" % res_json["data"])
            logging.debug("Getting JSON report")

            if raw:
                res = ptt.api.get_raw_output(scan_id, headers=HEADERS)
                return res.text

            res = ptt.api.get_output(scan_id, headers=HEADERS)
            output_json = json.dumps(res.json())
            return output_json


def website_scanner(target: str) -> str:
    tool_id = ptt.api.Tool.WEBSITE_SCANNER
    tool_params = {"scan_type": "light"}
    scan_params = {"max_scan_time": "5"}

    return run_tool(tool_id, target, tool_params, scan_params)


def icmp_ping(target: str) -> str:

    tool_id = 240  # ICMP Ping
    tool_params = {"scan_type": "light"}
    return run_tool(tool_id, target, tool_params, {}, raw=True)


def subdomain_finder(target: str) -> str:
    tool_id = ptt.api.Tool.SUBDOMAIN_FINDER
    tool_params = {"scan_type": "light"}
    scan_params = {"max_scan_time": "5"}

    return run_tool(tool_id, target, tool_params, scan_params)


def network_scanner(target: str) -> str:
    tool_id = ptt.api.Tool.NETWORK_SCANNER
    tool_params = {"scan_type": "light"}
    scan_params = {"max_scan_time": "5"}

    return run_tool(tool_id, target, tool_params, scan_params)


def url_fuzzer(target: str) -> str:
    tool_id = ptt.api.Tool.URL_FUZZER
    tool_params = {"scan_type": "light"}
    scan_params = {"max_scan_time": "5"}

    return run_tool(tool_id, target, tool_params, scan_params)


if FastMCP is not None:
    mcp = FastMCP("ptt")

    ### Resources
    # Resources are generally the GET requests from API
    @mcp.resource("scans://")
    def get_scans() -> str:
        """Get the current running scans"""
        res = ptt.api.get_scans(headers=HEADERS)

        # TODO: json data may not be suitable for LLMs, at least for the less powerful ones.
        # Make the data more LLM friendly
        return json.dumps(res.json())

    @mcp.resource("targets://")
    def get_targets() -> str:
        """Get the current targets available"""
        res = ptt.api.get_targets(headers=HEADERS)

        return json.dumps(res.json())

    ### Prompts
    # Prompts should match the intended use-cases for the pentesttools MCP
    @mcp.prompt()
    def pentest(target: str) -> str:
        """Create a pentesting report prompt"""
        return f"Please provide a comprehensive pentesting report for this target: {target}, using the tools at your disposal."

    @mcp.prompt()
    def is_up(target: str) -> str:
        """Create a simple target checking prompt"""
        return f"Please see if this target: {target} is up, using the tools at your disposal."

    @mcp.prompt()
    def subdomains(target: str) -> str:
        """Discover all subdomains for the target domain"""
        return f"Please find all the subdomains that you can for this domain: {target}, using the tools at your disposal."

    ### Tools
    # These are the actual pentesting tools, with some added management tools, available through `ptt.api`

    # Pentesting tools
    @mcp.tool()
    async def run_website_scanner(target: str) -> str:
        """
        Run a website scanner on the target URL.
        If you need to run a general website scan, this is the tool to use.
        If you need more information from a website under test, use this tool and tell the user you'll use this tool to accomplish your goal.

        Args:
            target: The target URL to scan.
        """
        return website_scanner(target)

    @mcp.tool()
    async def run_icmp_ping(target: str) -> str:
        """
        Run an icmp ping on the target URL.
        If you need to check that a target is reachable by ping or live, in general, this is the tool to use.

        Args:
            target: The target URL to ping.
        """
        return icmp_ping(target)

    @mcp.tool()
    async def run_subdomain_finder(target: str) -> str:
        """
        Run a subdomain finder scan on the target domain name.
        If you want to see all the findable subdomains on the target domain, this is the domain to use.

        It can be paired with website scans or api scans on the most important subdomains found.

        Args:
            target: The target domain to scan.
        """
        return subdomain_finder(target)

    @mcp.tool()
    async def run_network_scanner(target: str) -> str:
        """
        Run a network scanner on the target hostname.
        If you need to run a general network scan, something like nmap, this is the tool to use.
        If you need more information about the target network, use this tool and tell the user you'll use this tool to accomplish your goal.

        Args:
            target: The target hostname to scan.
        """
        return network_scanner(target)

    @mcp.tool()
    async def run_url_fuzzer(target: str) -> str:
        """
        This tool, The URL Fuzzer, can be used to find hidden files and directories on a web server by fuzzing.

        This is a discovery activity that allows you to discover resources that were not meant to be publicly accessible (e.g. /backups, /index.php.old, /archive.tgz, /source_code.zip, etc.).
        Since 'security by obscurity' is not a good practice, you can often find sensitive information in the hidden locations identified by the URL Fuzzer.

        Args:
            target: The target path to scan
            It has the following format: https://example.com/test?id=FUZZ
        """
        return url_fuzzer(target)

    # Administrative tools
    @mcp.tool()
    async def get_scan_output_by_id(scan_id: str) -> str:
        """Get the scan report as json output, by scan_id"""
        res = ptt.api.get_output(scan_id, headers=HEADERS)

        return json.dumps(res.json())

    @mcp.tool()
    async def get_findings() -> str:
        """Get all the findings that you have"""
        res = ptt.api.get_findings(headers=HEADERS)

        return json.dumps(res.json())


def entrypoint():
    """Entrypoint for ptt-mcp"""
    logging.basicConfig(
        level=logging.DEBUG,
        format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    )

    if FastMCP is None:
        logging.error(
            "PentestTools has been installed without MCP support. Install with pip install PentestTools[mcp] if you want MCP support."
        )
        return

    logging.info("Starting the PentestTools MCP server")
    return mcp.run(transport="stdio")
